"""
$RCSfile: TestXSLTMethod.py,v $
Unit tests for ZopeXMLMethods product

Author: Craeg Strong <cstrong@arielpartners.com>
Modified by Philipp von Weitershausen <philikon@philikon.de>

Release: 1.0
"""

__cvstag__  = '$Name:  $'[6:-2]
__date__    = '$Date: 2003/03/30 20:51:59 $'[6:-2]
__version__ = '$Revision: 1.27 $'[10:-2]

import os, sys
if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

# Load fixture
from Testing import ZopeTestCase

# ZopeXMLMethods
from Products.ZopeXMLMethods import XSLTMethod, ProcessorChooser

ZopeTestCase.installProduct('ZopeXMLMethods')
ZopeTestCase.installProduct('PageTemplates')
try:
    ZopeTestCase.installProduct('ParsedXML')
except:
    pass
try:
    ZopeTestCase.installProduct('ExternalFile')
except:
    pass
try:
    ZopeTestCase.installProduct('ExtFile')
except:
    pass

from Testing.ZopeTestCase import _folder_name, _standard_permissions
host, port = ZopeTestCase.utils.startZServer(3)
base = 'http://%s:%d/%s' %(host, port, _folder_name)

class XSLTMethodTestCase(ZopeTestCase.ZopeTestCase):
    """
    Unit tests for ZopeXMLMethods Zope Product.  These tests use the
    ZopeTestCase enhanced testing product.
    """

    ################################################################
    # Fixture
    ################################################################

    def beforeSetUp(self):
        get_transaction().begin()

    def afterSetUp(self):
        '''Add object to default fixture'''

        #
        # Some of our negative testcases can cause warning
        # strings to be printed out.  Suppress those,
        # because it makes it seem like there is something wrong
        # with the test when there is not.
        # CKS 9/10/2002
        #
        import warnings
        warnings.filterwarnings('ignore','',UserWarning)
        
        pass

    def beforeClose(self):
        # Commit the cleared app
        get_transaction().commit()

    def afterClear(self):
        '''Clean up after myself'''
        pass
    
    ################################################################
    # Test Cases
    ################################################################

    def test_01(self):
        "Ensure that XSLTMethod cannot be created with a blank ID"
        self.assertRaises(Exception, XSLTMethod.addInstance,
                          self.folder,id='',
                          xslTransformerId='')

    def test_02(self):
        "Ensure that XSLTMethod cannot be created without a valid transformer"
        self.assertRaises(Exception, XSLTMethod.addInstance,
                          self.folder,id='aXSLTMethod',
                          xslTransformerId='aTransformer')

    def test_03(self):
        "Ensure that XSLTMethod can be created with minimal correct params"
        self.folder.manage_addDTMLMethod('aTransformer')
        self.folder.manage_addDTMLMethod('aSource')
        src = self.folder['aSource']

        xform = XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                         xslTransformerId='aTransformer')

        self.failUnless(xform, "XSLTMethod successfully created should not be None")
        self.assertEquals(xform.id, 'aXSLTMethod')

    def test_04(self):
        "Ensure that basic transform works correctly"
        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_05(self):
        "Ensure that we can obtain our XSLT via containment aquisition"
        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)

        self.folder.manage_addFolder('one')
        one = self.folder['one']
        one.manage_addFolder('two')
        two = one['two']
        two.manage_addFolder('three')
        three = two['three']
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        three.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(one,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            three.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_06(self):
        "Ensure that we can obtain our XSLT via context aquisition"

        self.folder.manage_addFolder('some')
        some = self.folder['some']
        some.manage_addFolder('other')
        other = some['other']
        other.manage_addFolder('place')
        place = other['place']
        
        xsltFile     = open('testfiles/simple.xsl','rb')
        place.manage_addDTMLMethod('aTransformer', '', xsltFile)

        self.folder.manage_addFolder('one')
        one = self.folder['one']
        one.manage_addFolder('two')
        two = one['two']
        two.manage_addFolder('three')
        three = two['three']
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        three.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(one,id='aXSLTMethod',
                                 xslTransformerId='some/other/place/aTransformer')
        self.assertEquals(
            three.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_07(self):
        "Ensure that transformation obtains scalar parameters correctly"
        xsltFile     = open('testfiles/param.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.XSLparameters = [ "who", "message" ]
        self.folder.who           = "sneezy"
        self.folder.message       = "take some claritin"
        
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        result = self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST)
        self.assertEquals(result, "Hello, sneezy and take some claritin")

    def test_08(self):
        "Ensure that transformation obtains zope object parameters correctly"
        xsltFile     = open('testfiles/propertiesfile.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.XSLparameters = [ "properties" ]
        
        propFile      = open('testfiles/properties.xml', 'rb')
        self.folder.manage_addDTMLMethod('properties','',propFile)
        
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')

        expected = "url is " + base + "/properties"
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            expected)

    def test_09(self):
        """Ensure that transformation resolves URLs in documents correctly"""
        xsltFile     = open('testfiles/paramfile.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]
        self.folder.XSLparameters = [ "properties" ]
        self.folder.properties    = "arielpartners/propfolder/foo/properties"

        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('propfolder')
        propfolder = arielpartners['propfolder']
        propfolder.manage_addFolder('foo')
        foo = propfolder['foo']
        
        propFile      = open('testfiles/properties.xml', 'rb')
        foo.manage_addDTMLMethod('properties','',propFile)
        #
        # We must commit the transaction so the ZServer (running in a
        # different thread) can see our new Zope objects when it
        # tries to resolve the relative URL defined in self.folder.properties
        #
        get_transaction().commit()
        
        xform = XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                         xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),        
            "Hello, sneezy")

    def test_10(self):
        "Ensure that transformation resolves URNs in documents correctly"
        xsltFile     = open('testfiles/calldocumentfuncURN.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]
        
        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('propfolder')
        propfolder = arielpartners['propfolder']
        propfolder.manage_addFolder('foo')
        foo = propfolder['foo']
        
        propFile      = open('testfiles/properties.xml', 'rb')
        foo.manage_addDTMLMethod('properties','',propFile)
        
        XSLTMethod.addInstance(arielpartners,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            arielpartners.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, sneezy")

    def test_11(self):
        "Ensure that transformation resolves URNs in transformers correctly"
        xsltFile     = open('testfiles/includeURN.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)

        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]

        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('xforms')
        xforms = arielpartners['xforms']
        
        xsltFile      = open('testfiles/simple.xsl', 'rb')
        xforms.manage_addDTMLMethod('simple','',xsltFile)
        
        XSLTMethod.addInstance(arielpartners,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            arielpartners.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_12(self):
        "Verify error report if stylesheet URN resolver cannot find included stylesheet"
        xsltFile     = open('testfiles/includeURN.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)

        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]

        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('xforms')
        xforms = arielpartners['xforms']

        # Nope, not there.  
        #xsltFile      = open('testfiles/simple.xsl', 'rb')
        #xforms.manage_addDTMLMethod('simple','',xsltFile)
        
        XSLTMethod.addInstance(arielpartners,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        try:
            arielpartners.aSource.aXSLTMethod.testTransform(self.app.REQUEST)
        except Exception, e:
            self.failIf(str(e).find("urn") == -1)
            self.failIf(str(e).find("arielpartners:xforms/simple") == -1)            

    def test_13(self):
        "Verify error report if document URN resolver cannot find included document"
        xsltFile     = open('testfiles/calldocumentfuncURN.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]

        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('propfolder')
        propfolder = arielpartners['propfolder']

        # Nope.  not there.
        #propFile      = open('testfiles/properties.xml', 'rb')
        #propfolder.manage_addDTMLMethod('properties','',propFile)
        
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        
        try:
            self.folder.aSource.aXSLTMethod.testTransform(self.app.REQUEST)
        except Exception, e:
            self.failIf(str(e).find("document") == -1)

    def test_14(self):
        "Verify error report if document URN resolver gets badly formed URN"
        xsltFile     = open('testfiles/paramfile.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]
        self.folder.XSLparameters = [ "properties" ]
        self.folder.properties    = "urn:arielpartners:propfolder/i/dont/exist"

        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('propfolder')
        propfolder = arielpartners['propfolder']
        propfolder.manage_addFolder('i')
        i = propfolder['i']
        i.manage_addFolder('dont')
        dont = i['dont']
        dont.manage_addFolder('exist')

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')

        self.assertRaises(Exception,
                          arielpartners.aSource.aXSLTMethod.testTransform,
                          self.app.REQUEST)

    def test_15(self):
        "Ensure that transformation can produce valid DTML"
        xsltFile     = open('testfiles/dtmltest.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.who = "sneezy"

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        xform = self.folder.aSource.aXSLTMethod

        result = xform(xform, self.app.REQUEST)
        self.assertEquals(result.strip(), "Hello, sneezy")

    def test_16(self):
        "Ensure that transformation resolves DTML correctly"
        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')

        dtmlFile      = open('testfiles/dtmltest.dtml', 'rb')
        self.folder.manage_addDTMLDocument('dtmltest', '', dtmlFile)
        dtmltest = self.folder['dtmltest']
        result = dtmltest(dtmltest)
        self.assertEquals(result.strip(), "<div>Hello, world</div>")

    def test_17(self):
        "Ensure that transformation can produce valid ZPT"

        xsltFile     = open('testfiles/templatetest.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')

        # VERY IMPORTANT: PageTemplates are essentially HTML, and therefore
        # apply HTML syntax rules.  For example, all elements are *automatically*
        # lowercased.  Yikes!
        self.folder.manage_addProduct['PageTemplates'].manage_addPageTemplate(
            'aSource', '', xmlFile)

        self.folder.who = "sneezy"
        
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')

        xform = self.folder.aSource.aXSLTMethod
        result = xform(xform, self.app.REQUEST)
        self.failIf(result.find("Hello, sneezy") == -1)

    def test_18(self):
        "Ensure that transformation resolves page templates correctly"
            
        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.who = "sneezy"
        
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer',
                                 behave_like="Page Template")

        zptFile = open('testfiles/zpttest.zpt', 'rb')
        self.folder.manage_addProduct['PageTemplates'].manage_addPageTemplate('zpttest', '', zptFile)
        zpttest = self.folder['zpttest']

        result = zpttest(zpttest, self.app.REQUEST)
        self.assertEquals(result.strip(), "Hello, world")

    def test_19(self):
        "Ensure that transformation resolves relative URLs correctly"
        self.folder.manage_addFolder('sub')
        sub = self.folder['sub']
        xsltFile     = open('testfiles/include.xsl','rb')
        sub.manage_addDTMLMethod('aTransformer', '', xsltFile)

        xsltFile      = open('testfiles/simple.xsl', 'rb')
        self.folder.manage_addDTMLMethod('simple','',xsltFile)

        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)
        #
        # We must commit the transaction so the ZServer (running in a
        # different thread) can see our new Zope objects when it
        # tries to resolve the relative URL in include.xsl
        #
        get_transaction().commit()

        xform = XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                         xslTransformerId='sub/aTransformer')

        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_20(self):
        "Test URN resolution in XSLT"
        xsltFile     = open('testfiles/calldocURN2.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/include.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.URNnamespaces = [ "arielpartners" ]
        
        self.folder.manage_addFolder('arielpartners')
        arielpartners = self.folder['arielpartners']
        arielpartners.manage_addFolder('propfolder')
        propfolder = arielpartners['propfolder']
        propfolder.manage_addFolder('foo')
        foo = propfolder['foo']
        
        propFile      = open('testfiles/properties.xml', 'rb')
        foo.manage_addDTMLMethod('properties','',propFile)
        
        xform = XSLTMethod.addInstance(arielpartners,id='aXSLTMethod',
                                         xslTransformerId='aTransformer')

        result = arielpartners.aSource.aXSLTMethod.transform(self.app.REQUEST)
        self.assertEquals(result.strip(), "Hello, sneezy")

    def test_21(self):
        "Make sure page templates can be used for both XML and XSLT"

        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addProduct['PageTemplates'].manage_addPageTemplate('aTransformer',
                                                                              '', xsltFile)

        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addProduct['PageTemplates'].manage_addPageTemplate('aSource',
                                                                              '', xmlFile)
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")
        
    def test_22(self):
        "Try using File objects for XML and XSLT"

        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addFile('aTransformer', xsltFile)
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addFile('aSource', xmlFile)

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_23(self):
        "Try using ParsedXML objects for XML and XSLT"

        try:
            from Products.ParsedXML.ParsedXML import ParsedXML
        except:
            print "ParsedXML product not found, skipping test..."
            return 1

        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addProduct['ParsedXML'].manage_addParsedXML('aTransformer',
                                                                        '', xsltFile.read())
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addProduct['ParsedXML'].manage_addParsedXML('aSource',
                                                                       '', xmlFile.read())

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_24(self):
        "Try using DTML Document objects for XML and XSLT"

        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_25(self):
        "Try using ExternalFile objects for XML and XSLT"

        try:
            from Products.ExternalFile.ExternalFile import ExternalFile
        except:
            print "ExternalFile product not found, skipping test..."
            return 1

        xsltFile     = 'testfiles/simple.xsl'
        self.folder.manage_addProduct['ExternalFile'].manage_add('aTransformer','','',
                                                                 target_filepath=xsltFile)
        
        xmlFile      = 'testfiles/simple.xml'
        self.folder.manage_addProduct['ExternalFile'].manage_add('aSource','','',
                                                                 target_filepath=xmlFile)

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_26(self):
        "Try using ExtFile objects for XML and XSLT"

        try:
            from Products.ExtFile.ExtFile import ExtFile
        except:
            print "ExtFile product not found, skipping test..."
            return 1

        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addProduct['ExtFile'].manage_addExtFile('aTransformer', file=xsltFile)
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addProduct['ExtFile'].manage_addExtFile('aSource', file=xmlFile)

        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_27(self):
        "Test XSLT Pipelines"

        xsltFile    = open('testfiles/pipeline1.xsl','rb')
        self.folder.manage_addDTMLMethod('xslt1', '', xsltFile)

        xsltFile    = open('testfiles/pipeline2.xsl','rb')
        self.folder.manage_addDTMLMethod('xslt2', '', xsltFile)

        xsltFile    = open('testfiles/pipeline3.xsl','rb')
        self.folder.manage_addDTMLMethod('xslt3', '', xsltFile)

        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(self.folder,id='tran1',
                                 xslTransformerId='xslt1')

        XSLTMethod.addInstance(self.folder,id='tran2',
                                 xslTransformerId='xslt2')

        XSLTMethod.addInstance(self.folder,id='tran3',
                                 xslTransformerId='xslt3')

        text = "<div tal:replace=\"here/aSource/tran1/tran2/tran3\">Oh, crap</div>"

        self.folder.manage_addProduct['PageTemplates'].manage_addPageTemplate('aZPT',
                                                                              '',text)
        aZPT   = self.folder['aZPT']
        result = aZPT(aZPT, self.app.REQUEST)
        self.assertEquals(result.strip(), "Hello, world")

    def test_28(self):
        "Ensure that we can obtain our XSLT via more than one level of containment aquisition"
        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)

        self.folder.manage_addFolder('one')
        one = self.folder['one']
        one.manage_addFolder('two')
        two = one['two']
        two.manage_addFolder('three')
        three = two['three']
        
        xmlFile      = open('testfiles/simple.xml', 'rb')
        three.manage_addDTMLMethod('aSource', '', xmlFile)

        XSLTMethod.addInstance(two,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            three.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_29(self):
        "Ensure that we can call an XSLTMethod using context aquisition"
        xsltFile     = open('testfiles/simple.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)

        xmlFile      = open('testfiles/simple.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        self.folder.manage_addFolder('one')
        one = self.folder['one']
        one.manage_addFolder('two')
        two = one['two']
        two.manage_addFolder('three')
        three = two['three']

        XSLTMethod.addInstance(three,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.one.two.three.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, world")

    def test_30(self):
        "Ensure that transformation resolves URNs in documents correctly using XML catalog"
        xsltFile     = open('testfiles/calldocumentfuncURN.xsl','rb')
        self.folder.manage_addDTMLMethod('aTransformer', '', xsltFile)
        
        xmlFile      = open('testfiles/param.xml', 'rb')
        self.folder.manage_addDTMLMethod('aSource', '', xmlFile)

        catFile      = open('testfiles/catalog.xml', 'rb')
        self.folder.manage_addDTMLMethod('aCatalog', '', catFile)        

        self.folder.XMLcatalog = "aCatalog"
        
        self.folder.manage_addFolder('bar')
        bar = self.folder['bar']

        propFile      = open('testfiles/properties.xml', 'rb')
        bar.manage_addDTMLMethod('properties','',propFile)

        #
        # We must commit the transaction so the ZServer (running in a
        # different thread) can see our new Zope objects when it
        # tries to resolve the relative URL defined in self.folder.properties
        #
        get_transaction().commit()
        
        XSLTMethod.addInstance(self.folder,id='aXSLTMethod',
                                 xslTransformerId='aTransformer')
        self.assertEquals(
            self.folder.aSource.aXSLTMethod.transform(self.app.REQUEST),
            "Hello, sneezy")

    ################################################################
    # Test Runner Setup
    ################################################################

if __name__ == '__main__':
    # Specify which processor to use
    # ProcessorChooser.default = '4Suite-0.11.1'
        
    # report which processor we are using
    print "Using processor:", \
          ProcessorChooser.ProcessorChooser().defaultProcessor()

    framework(descriptions=1, verbosity=2) # 0=quiet 1=default 2=verbose
else:
    # While framework.py provides its own test_suite() 
    # method the testrunner utility does not.
    import unittest
    def test_suite():
        suite = unittest.TestSuite()
        suite.addTest(unittest.makeSuite(XSLTMethodTestCase))
        return suite
