/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.util.Vector;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;

import org.eclipse.wst.rdb.internal.outputview.actions.RemoveAllOutputTableRow;
import org.eclipse.wst.rdb.internal.outputview.actions.RemoveOutputTableRow;

/**
 * The table for displaying actions in the output view.
 */
public class OutputViewActionsTable extends TableViewer implements IMenuListener 
{
   //protected Composite control;
   /** The table. */
   protected Table aTable;
   /** The selected table item. */
   protected TableItem selectedTableItem;
   /** The model for the view. */
   protected OutputViewModel model;
   /** The view. */
   protected OutputView view;
   
   /**
    * Constructs an OutputViewActionsTable.
    * @param parent The parent composite
    * @param model  The output view model.
    * @param outputView The view we are a part of.
    */
   public OutputViewActionsTable(Composite parent, OutputViewModel model, OutputView outputView) 
   {
      super( new Table (parent, SWT.SINGLE | SWT.FULL_SELECTION |
               SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL) );
      
      this.model = model;
      view = outputView;
      aTable = getTable();
      setUpTable();
      
      setLabelProvider(new ActionsTableLabelProvider());
      setContentProvider(new ActionsTableContentProvider());
      
      
      MenuManager contextMenu = new MenuManager("#PopUp");
      contextMenu.add(new Separator("additions"));
      contextMenu.setRemoveAllWhenShown(true);
      contextMenu.addMenuListener(this);
      Menu menu= contextMenu.createContextMenu(getControl());
      getControl().setMenu(menu);
      
      setInput(model.getActionList());
   }
   
   /**
    * The context menu is about to appear.  Populate it.
    * @param menu The menu manager.
    */
   public void menuAboutToShow(IMenuManager menu)
   {
      RemoveOutputTableRow removeRow = new RemoveOutputTableRow(this);
      menu.add(removeRow);
      RemoveAllOutputTableRow removeAll = new RemoveAllOutputTableRow(this);
      menu.add(removeAll);
   }
   
   /**
    * Sets up the table.
    */
   protected void setUpTable() 
   {
      aTable.setLinesVisible (true);
      aTable.setHeaderVisible (true);
      
      TableColumn column = new TableColumn(aTable, SWT.NULL);
      column.setText(OutputViewPlugin.getString("OV_TABLE_TITLE_STATUS"));
      
      column = new TableColumn(aTable, SWT.NULL);
      column.setText(OutputViewPlugin.getString("OV_TABLE_TITLE_ACTION"));
      
      column = new TableColumn(aTable, SWT.NULL);
      column.setText(OutputViewPlugin.getString("OV_TABLE_TITLE_NAME"));
      
      TableLayout layout = new TableLayout();
      ColumnWeightData data = new ColumnWeightData(33,true);
      layout.addColumnData(data);
      layout.addColumnData(data);
      layout.addColumnData(new ColumnWeightData(34, false));
      
      aTable.setLayout(layout);
   }
   
   /**
    * Removes the selected item from the table.
    */
   public void removeSelectedItem()
   {
      int i = aTable.getSelectionIndex();
      if (i >= 0)
      {
         model.removeOutputItem(i);
         refresh();
      }
      view.refreshWithDefault();
   }
   
   /**
    * Removes all items from the table.
    */
   public void removeAll()
   {
      model.removeAll();
      view.refreshWithDefault();
   }
   
   /** 
    * Selects the first item.
    */
   protected void selectDefault()
   {
      if (!model.isActionListEmpty())   // select the first item  if there is one
      {
         aTable.select(0);
         refresh();
      }
   }
   
   /**
    * Selects a given item.
    * @param item The item to select.
    */
   protected void selectItem(OutputItem item)
   {
      int i = model.getActionList().indexOf(item);
      if ( i >= 0)
      {
         aTable.setFocus();
         aTable.select(i);
         refresh();
      }
   }
   
   /** 
    * Selects an item at a given index.
    * @param i The index of the item to select.
    */
   protected void selectItem(int i)
   {
      if ( i >= 0)
      {
         aTable.setFocus();
         aTable.select(i);
         refresh();
      }
   }
   
   protected int getIndex(OutputItem item){
      return indexForElement(item);
   }
   
   /**
    * Gets the selected item.
    * @return The selected OutputItem.
    */
   public OutputItem getSelectedItem()
   {
      int idx = aTable.getSelectionIndex();
      if (idx >= 0)
         return (OutputItem)model.getActionList().get(idx);
      else
         return null;
   }
   
   /**
    *  Provides the content of the Actions table in the form
    *  of an array of DataObjects.
    */
   protected class ActionsTableContentProvider implements IStructuredContentProvider 
   {
      /**
       * Method getElements.
       * @param collection A Vector that contains the DataObjects.
       * @return An array of dataobjects (one for each row).
       * @see IStructuredContentProvider#getElements(Object)
       */
      public Object[] getElements(Object collection) 
      {
         if (collection instanceof Vector)
         {
            return ((Vector)collection).toArray();
         }
         return null;
      }
      
      /**
       * Does nothing.
       * @see IContentProvider#dispose()
       */
      public void dispose() {
      }
      
      /**
       * Does nothing.
       * @see IContentProvider#inputChanged(Viewer, Object, Object)
       */
      public void inputChanged(Viewer viewer, Object old, Object newobj)
      {
      }
      
   } //END ContentProvider
   
   /**
    * LabelProvider for ActionsTable
    *
    */
   protected class ActionsTableLabelProvider	extends LabelProvider implements ITableLabelProvider 
   {
      /**
       * Gets the text of a item for a given column.
       * @param outputItem Represents a row.
       * @param columnIndex The index of the column.
       * @return The result to be displayed in the cell.
       *
       * @see ITableLabelProvider#getColumnText(Object, int)
       */
      public String getColumnText(Object outputItem, int columnIndex) 
      {
         String colText =  "";
         switch (columnIndex)
         {
         case 0 : { colText = ((OutputItem)outputItem).getStatus(); break; }
         case 1 : { colText = ((OutputItem)outputItem).getAction(); break; }
         case 2 : { colText = ((OutputItem)outputItem).getObjName(); break; }
         }
         if (colText == null)
            return "";
         else
            return colText;
      }
      
      /**
       * Gets the image for a item at a given column.
       * @param outputItem Represents a row.
       * @param columnIndex The index of the column.
       * @return The Image to be displayed in the cell, or null if this cell has no status image.
       *
       * @see ITableLabelProvider#getColumnImage(Object, int)
       */
      public Image getColumnImage(Object outputItem, int columnIndex) {
         
         if (columnIndex == 0)    //first column has an image
         {
            return ((OutputItem)outputItem).getStatusImage();
         }
         else
            return null;
      }
      
   } //END LabelProvider
   
}
