/*
 * Copyright 2005 - 2009  Zarafa B.V.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifndef ECLOGGER_H
#define ECLOGGER_H

#include <string>
#include <pthread.h>
#include <stdio.h>
#include <stdarg.h>
#include "ECConfig.h"

#ifndef __LIKE_PRINTF
#define __LIKE_PRINTF(_fmt, _va)
#endif

#define EC_LOGLEVEL_NONE	0
#define EC_LOGLEVEL_FATAL	1
#define EC_LOGLEVEL_ERROR	2
#define EC_LOGLEVEL_WARNING	3
#define EC_LOGLEVEL_INFO	4
#define EC_LOGLEVEL_ANY		5
#define EC_LOGLEVEL_DEBUG	6

#define _LOG_BUFSIZE		10240
#define _LOG_TSSIZE			64

#define LOG_DEBUG(_plog,_msg,...) if ((_plog)->Log(EC_LOGLEVEL_DEBUG)) (_plog)->Log(EC_LOGLEVEL_DEBUG, _msg, ##__VA_ARGS__)

enum logprefix { LP_NONE, LP_TID, LP_PID };

class ECLogger {
protected:
	char* MakeTimestamp();

	int max_loglevel;
	char *msgbuffer;
	pthread_mutex_t msgbuflock;
	char timestring[_LOG_TSSIZE];
	logprefix prefix;

public:
	ECLogger(int max_ll);
	virtual ~ECLogger();

	bool Log(int loglevel);
	void SetLoglevel(int max_ll);
	void SetLogprefix(logprefix lp);

	virtual void Reset() = 0;
	virtual int GetFileDescriptor();
	virtual void Log(int loglevel, const std::string &message) = 0;
	virtual void Log(int loglevel, const char *format, ...) __LIKE_PRINTF(3, 4) = 0;
};


/* Dummy null logger */
class ECLogger_Null : public ECLogger {
public:
	ECLogger_Null();
	~ECLogger_Null();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int Loglevel, const char *format, ...) __LIKE_PRINTF(3, 4);
};

/* File logger. Use "-" for stderr logging */
class ECLogger_File : public ECLogger {
private:
	typedef void* handle_type;
	typedef handle_type(*open_func)(const char*, const char*);
	typedef int(*close_func)(handle_type);
	typedef int(*printf_func)(handle_type, const char*, ...);
	typedef int(*fileno_func)(handle_type);
	typedef int(*flush_func)(handle_type);

	handle_type log;
	char *logname;
	pthread_mutex_t filelock;
	int timestamp;

	open_func fnOpen;
	close_func fnClose;
	printf_func fnPrintf;
	fileno_func fnFileno;
	flush_func fnFlush;
	const char *szMode;

	int prevcount;
	std::string prevmsg;
	bool DupFilter(const std::string &message);

public:
	ECLogger_File(int max_ll, int add_timestamp, const char *filename, bool compress = false);
	~ECLogger_File();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int loglevel, const char *format, ...) __LIKE_PRINTF(3, 4);

	int GetFileDescriptor();
};

/*	Linux syslog logger */
class ECLogger_Syslog : public ECLogger {
private:
	int levelmap[EC_LOGLEVEL_DEBUG+1];	/* converts to syslog levels */

public:
	ECLogger_Syslog(int max_ll, const char *ident);
	~ECLogger_Syslog();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int loglevel, const char *format, ...) __LIKE_PRINTF(3, 4);
};


/*  */
class ECLogger_Pipe : public ECLogger {
private:
	int m_fd;
	pid_t m_childpid;

public:
	ECLogger_Pipe(int fd, int childpid, int loglevel);
	~ECLogger_Pipe();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string &message);
	virtual void Log(int loglevel, const char *format, ...) __LIKE_PRINTF(3, 4);

	int GetFileDescriptor();
};

ECLogger* StartLoggerProcess(ECConfig *lpConfig, ECLogger *lpFileLogger);

ECLogger* CreateLogger(ECConfig *config, char *argv0, const char *lpszServiceName);
int DeleteLogger(ECLogger *lpLogger);
void LogConfigErrors(ECConfig *lpConfig, ECLogger *lpLogger);

#endif
