# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>


from elisa.core import common
from elisa.core.utils.text import name_to_shortcut

from elisa.plugins.base.models.audio import TrackModel

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.music_library import ArtistsViewMode, \
                                                 AlbumsViewMode, \
                                                 TracksViewMode

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from twisted.internet import task

class ArtistsDaapController(HierarchyController):

    def initialize(self, uri):
        self.uri = uri
        return super(ArtistsDaapController, self).initialize()

    def populate_model(self):

        def got_artists(model):
            artists = model.artists
            artists = sorted(artists, key=lambda a: a.name.lower())
            return artists

        model, dfr = common.application.resource_manager.get(self.uri)
        dfr.addCallback(got_artists)
        return dfr

    def node_clicked(self, widget, item):
        browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
        controller_path = '/poblesec/music/daap/albums'
        dfr = browser.history.append_controller(controller_path,
                                                item.name, artist=item)
        return dfr

class ArtistsDaapPreviewListController(ArtistsDaapController, MenuItemPreviewListController):
    view_mode = ArtistsViewMode
    fastscroller_enabled = True

    def get_shortcut_for_item(self, item):
        return name_to_shortcut(item.name)

class ArtistsDaapCoverflowController(ArtistsDaapController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsViewMode

class ArtistsDaapGridController(ArtistsDaapController, GridItemGridController):
    view_mode = ArtistsViewMode

class ArtistsDaapListSwitcherController(ListSwitcherController):
    modes = [ArtistsDaapPreviewListController,
             ArtistsDaapCoverflowController,
             ArtistsDaapGridController]
    default_mode = ArtistsDaapPreviewListController


class AlbumsDaapController(HierarchyController):

    def initialize(self, artist):
        self.artist = artist
        return super(AlbumsDaapController, self).initialize()

    def populate_model(self):

        def got_albums(albums_list_model):
            albums = albums_list_model.albums
            albums = sorted(albums, key=lambda a: a.name.lower())
            return albums

        uri = self.artist.elisa_uri
        model, dfr = common.application.resource_manager.get(uri)
        dfr.addCallback(got_albums)
        return dfr

    def node_clicked(self, widget, item):
        browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
        controller_path = '/poblesec/music/daap/tracks'
        dfr = browser.history.append_controller(controller_path,
                                                item.name, album=item)

class AlbumsDaapPreviewListController(AlbumsDaapController, MenuItemPreviewListController):
    view_mode = AlbumsViewMode
    fastscroller_enabled = True

    def get_shortcut_for_item(self, item):
        return name_to_shortcut(item.album)

class AlbumsDaapCoverflowController(AlbumsDaapController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsViewMode

class AlbumsDaapGridController(AlbumsDaapController, GridItemGridController):
    view_mode = AlbumsViewMode

class AlbumsDaapListSwitcherController(ListSwitcherController):
    modes = [AlbumsDaapPreviewListController,
             AlbumsDaapCoverflowController,
             AlbumsDaapGridController]
    default_mode = AlbumsDaapPreviewListController


class TracksDaapController(HierarchyController):

    def initialize(self, album):
        self.album = album
        return super(TracksDaapController, self).initialize()

    def populate_model(self):

        def got_tracks(tracks_list_model):
            tracks = sorted(tracks_list_model.tracks,
                            key=lambda i: i.track_number)
            return tracks

        uri = self.album.elisa_uri
        model, dfr = common.application.resource_manager.get(uri)
        dfr.addCallback(got_tracks)
        return dfr

    def node_clicked(self, widget, item):
        player = self.frontend.retrieve_controllers('/poblesec/music_player')[0]

        # Enqueue and play the clicked item
        player.player.play_model(item.playable_model)

        main = self.frontend.retrieve_controllers('/poblesec')[0]
        main.show_music_player()

        def iterate_tracks():
            # Enqueue all the following items of the container
            index = self.model.index(item)
            for item in self.model[index+1:]:
                if not isinstance(item, TrackModel):
                    continue

                player.player.enqueue_to_playlist(item.playable_model)
                yield item

        return task.coiterate(iterate_tracks())

class TracksDaapViewMode(TracksViewMode):

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        return None



class TracksDaapPreviewListController(TracksDaapController, MenuItemPreviewListController):
    view_mode = TracksDaapViewMode

class TracksDaapCoverflowController(TracksDaapController, ImageWithReflectionCoverflowController):
    view_mode = TracksDaapViewMode

class TracksDaapGridController(TracksDaapController, GridItemGridController):
    view_mode = TracksDaapViewMode

class TracksDaapListSwitcherController(ListSwitcherController):
    modes = [TracksDaapPreviewListController,
             TracksDaapCoverflowController,
             TracksDaapGridController]
    default_mode = TracksDaapPreviewListController
