# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

from twisted.trial.unittest import TestCase

from elisa.core.utils import sorting


class TestSorting(TestCase):

    def test_natural_sort(self):
        # trivial
        original = ["a2", "a1"]
        expected = ["a1", "a2"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        # mixed case
        original = ["a2", "A1"]
        expected = ["A1", "a2"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        # spaces are equal
        original = ["a 2", "a 1"]
        expected = ["a 1", "a 2"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        # more complicated
        original = ["A13", "A1", "A2"]
        expected = ["A1", "A2", "A13"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ["toto 12", "titi 42", "1 truc", "12 truc", "toto 1"]
        expected = ["1 truc", "12 truc", "titi 42", "toto 1", "toto 12"]

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

 
        # examples from ASPN recipe "Natural string sorting"
        # http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/285264
        original = ['ver-1.3.12', 'ver-1.3.3', 'ver-1.2.5', 'ver-1.2.15', \
                    'ver-1.2.3', 'ver-1.2.1']
        expected = ['ver-1.2.1', 'ver-1.2.3', 'ver-1.2.5', 'ver-1.2.15', \
                    'ver-1.3.3', 'ver-1.3.12']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ['C1H2', 'C1H4', 'C2H2', 'C2H6', 'C2N', 'C3H6']
        expected = ['C1H2', 'C1H4', 'C2H2', 'C2H6', 'C2N', 'C3H6']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ['Team 101', 'Team 58', 'Team 30', 'Team 1']
        expected = ['Team 1', 'Team 30', 'Team 58', 'Team 101']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = ['a5', 'A7', 'a15', 'a9', 'A8']
        expected = ['a5', 'A7', 'A8', 'a9', 'a15']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)

    """
    # DEACTIVATED, because our sorting is not suposted to support it
    def test_mixed_encodings_sorting(self):

        original = [u'ä01', 'ä20', 'a13', 'b14']
        expected = ['a13', 'b14', u'\xe401', '\xc3\xa420']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = [u'©20', '©01', '#k13', 'üä14', 'öñ']
        expected = ['#k13', '\xc2\xa901', u'\xa920', '\xc3\xb6\xc3\xb1', '\xc3\xbc\xc3\xa414']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = [u'Ö20', 'ö0100', 'Ö10', 'ö02']
        expected = ['\xc3\xb602', '\xc3\x9610', u'\xd620', '\xc3\xb60100']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)
    """

    def test_spaces(self):
        original = [u'  1Die1', '1Dae1']
        expected = ['1Dae1', u'  1Die1']
        sorting.natural_sort(original)
        self.assertEquals(original, expected)

        original = [u'  Zwo', 'zao']
        expected = [u'zao', u'  Zwo']

        sorting.natural_sort(original)
        self.assertEquals(original, expected)


class TestAsyncOrderedPlacement(TestCase):

    def test_simple_case(self):
        
        ref_list = [1, 3, 4, 5, 6]
        res_list = [1, 2, 3, 4, 5, 6]

        dfr = sorting.async_ordered_placement(ref_list, 2)
        dfr.addCallback(lambda x: self.assertEquals(ref_list, res_list))
        return dfr

    def test_merge(self):
        ref_list = [1, 3, 5, 7, 9]
        into = [2, 4, 6, 8]

        result = [1, 2, 3, 4, 5, 6, 7, 8, 9]

        dfr = sorting.async_sorted_merge(ref_list, into)
        dfr.addCallback(lambda x: self.assertEquals(ref_list, result))
        return dfr

    def test_merge_into_empty_list(self):
        ref_list = []
        into = [2, 4, 6, 8]

        result = [2, 4, 6, 8]

        dfr = sorting.async_sorted_merge(ref_list, into)
        dfr.addCallback(lambda x: self.assertEquals(ref_list, result))
        return dfr

    def test_merge_with_reverse_compare(self):
        ref_list = [9, 7, 5, 3, 1]
        into = [2, 4, 6, 8]

        def reverse_compare(self, other):
            return cmp(other, self)

        exp_result = [9, 8, 7, 6, 5, 4, 3, 2, 1]

        dfr = sorting.async_sorted_merge(ref_list, into, compare=reverse_compare)
        dfr.addCallback(lambda x: self.assertEquals(ref_list, exp_result))
        return dfr



