/***** BEGIN LICENSE BLOCK *****
 * Version: CPL 1.0/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Common Public
 * License Version 1.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.eclipse.org/legal/cpl-v10.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * Copyright (C) 2002 Benoit Cerrina <b.cerrina@wanadoo.fr>
 * Copyright (C) 2002-2004 Anders Bengtsson <ndrsbngtssn@yahoo.se>
 * Copyright (C) 2002-2004 Jan Arne Petersen <jpetersen@uni-bonn.de>
 * Copyright (C) 2004-2005 Thomas E Enebo <enebo@acm.org>
 * Copyright (C) 2004 Stefan Matthias Aust <sma@3plus4.de>
 * 
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the CPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the CPL, the GPL or the LGPL.
 ***** END LICENSE BLOCK *****/
package org.jruby.internal.runtime.methods;

import java.util.Iterator;

import org.jruby.IRuby;
import org.jruby.RubyArray;
import org.jruby.RubyModule;
import org.jruby.ast.ArgsNode;
import org.jruby.ast.ListNode;
import org.jruby.ast.Node;
import org.jruby.ast.ScopeNode;
import org.jruby.evaluator.AssignmentVisitor;
import org.jruby.evaluator.EvaluateVisitor;
import org.jruby.exceptions.JumpException;
import org.jruby.lexer.yacc.ISourcePosition;
import org.jruby.runtime.Arity;
import org.jruby.runtime.ICallable;
import org.jruby.runtime.Scope;
import org.jruby.runtime.ThreadContext;
import org.jruby.runtime.Visibility;
import org.jruby.runtime.builtin.IRubyObject;

/**
 *
 */
public final class DefaultMethod extends AbstractMethod {
    private ScopeNode body;
    private ArgsNode argsNode;
    private RubyModule parent;

    public DefaultMethod(RubyModule implementationClass, ScopeNode body, ArgsNode argsNode, 
        Visibility visibility, RubyModule parent) {
        super(implementationClass, visibility);
        this.body = body;
        this.argsNode = argsNode;
		this.parent = parent;
		
		assert body != null;
		assert argsNode != null;
    }

    /**
     * @see AbstractCallable#call(IRuby, IRubyObject, String, IRubyObject[], boolean)
     */
    public IRubyObject internalCall(IRuby runtime, IRubyObject receiver, String name, IRubyObject[] args, boolean noSuper) {
    	assert args != null;

        ThreadContext context = runtime.getCurrentContext();
        
        context.preDefMethodInternalCall(parent);
        
        Scope scope = context.getCurrentScope();
        if (body.getLocalNames() != null) {
            scope.resetLocalVariables(body.getLocalNames());
        }
        
        if (argsNode.getBlockArgNode() != null && context.isBlockGiven()) {
            scope.setValue(argsNode.getBlockArgNode().getCount(), runtime.newProc());
        }

        try {
            prepareArguments(runtime, scope, receiver, args);
            
            getArity().checkArity(runtime, args);

            traceCall(runtime, receiver, name);

            return receiver.eval(body.getBodyNode());
        } catch (JumpException je) {
        	if (je.getJumpType() == JumpException.JumpType.ReturnJump) {
	            if (je.getPrimaryData() == this) {
	                return (IRubyObject)je.getSecondaryData();
	            }
        	}
       		throw je;
        } finally {
            context.postDefMethodInternalCall();
            traceReturn(runtime, receiver, name);
        }
    }

    private void prepareArguments(IRuby runtime, Scope scope, IRubyObject receiver, IRubyObject[] args) {
        int expectedArgsCount = argsNode.getArgsCount();
        if (expectedArgsCount > args.length) {
            throw runtime.newArgumentError("Wrong # of arguments(" + args.length + " for " + expectedArgsCount + ")");
        }
        if (argsNode.getRestArg() == -1 && argsNode.getOptArgs() != null) {
            int opt = expectedArgsCount + argsNode.getOptArgs().size();

            if (opt < args.length) {
                throw runtime.newArgumentError("wrong # of arguments(" + args.length + " for " + opt + ")");
            }

            runtime.getCurrentContext().getCurrentFrame().setArgs(args);
        }

        if (scope.hasLocalVariables()) {
            if (expectedArgsCount > 0) {
                for (int i = 0; i < expectedArgsCount; i++) {
                    scope.setValue(i + 2, args[i]);
                }
            }

            if (argsNode.getOptArgs() != null) {
                ListNode optArgs = argsNode.getOptArgs();

                Iterator iter = optArgs.iterator();
                for (int i = expectedArgsCount; i < args.length && iter.hasNext(); i++) {
                    new AssignmentVisitor(runtime, receiver).assign((Node)iter.next(), args[i], true);
                    expectedArgsCount++;
                }

                // assign the default values.
                while (iter.hasNext()) {
                    EvaluateVisitor.getInstance().eval(receiver.getRuntime(), receiver, (Node)iter.next());
                }
            }

            if (argsNode.getRestArg() >= 0) {
                RubyArray array = runtime.newArray(args.length - expectedArgsCount);
                for (int i = expectedArgsCount; i < args.length; i++) {
                    array.append(args[i]);
                }
                scope.setValue(argsNode.getRestArg(), array);
            }
        }
    }

    private void traceReturn(IRuby runtime, IRubyObject receiver, String name) {
        if (runtime.getTraceFunction() == null) {
            return;
        }

        ISourcePosition position = runtime.getCurrentContext().getPreviousFrame().getPosition();
        runtime.callTraceFunction("return", position, receiver, name, getImplementationClass()); // XXX
    }

    private void traceCall(IRuby runtime, IRubyObject receiver, String name) {
        if (runtime.getTraceFunction() == null) {
            return;
        }

		ISourcePosition position = body.getBodyNode() != null ? 
            body.getBodyNode().getPosition() : body.getPosition();  

		runtime.callTraceFunction("call", position, receiver, name, getImplementationClass()); // XXX
    }

    public Arity getArity() {
        // TODO: Make special arity-related values use mnemonic
        // -2 means (*) signature to method def
        if (argsNode.getRestArg() == -2) {
        	return Arity.optional();
        } 

        if (argsNode.getOptArgs() != null || argsNode.getRestArg() >= 0) {
            return Arity.required(argsNode.getArgsCount());
        }

        return Arity.createArity(argsNode.getArgsCount());
    }
    
    public ICallable dup() {
        return new DefaultMethod(getImplementationClass(), body, argsNode, getVisibility(), parent);
    }	
}
