/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.history.core.storage;

import com.intellij.history.core.LocalVcs;
import com.intellij.history.core.storage.BrokenStorageException;
import com.intellij.history.core.storage.Content;
import com.intellij.history.core.storage.ContentStorage;
import com.intellij.history.core.storage.IContentStorage;
import com.intellij.history.core.storage.StoredContent;
import com.intellij.history.core.storage.Stream;
import com.intellij.history.core.storage.UnavailableContent;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.io.FileUtil;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

public class Storage {
    private static final Logger LOG = Logger.getInstance((String)("#" + Storage.class.getName()));
    private static final int VERSION = 19;
    private static final String BROKEN_MARK_FILE = ".broken";
    private static final String VERSION_FILE = "version";
    private static final String STORAGE_FILE = "storage";
    private static final String CONTENTS_FILE = "contents";
    private final File myDir;
    private IContentStorage myContentStorage;
    private boolean isBroken = false;

    public Storage(File dir) {
        this.myDir = dir;
        this.initStorage();
    }

    protected void initStorage() {
        this.initContentStorage();
        this.validate();
    }

    private void validate() {
        if (this.wasMarkedAsBroken() || !this.isValidVersion()) {
            this.deleteStorage();
            this.recreateStorage();
        }
    }

    private void deleteStorage() {
        this.close();
        FileUtil.delete((File)this.myDir);
    }

    private void recreateStorage() {
        this.myDir.mkdirs();
        this.initContentStorage();
        this.storeVersion();
    }

    private boolean wasMarkedAsBroken() {
        return new File(this.myDir, BROKEN_MARK_FILE).exists();
    }

    private boolean isValidVersion() {
        int storageVersion = this.load(VERSION_FILE, -1, new Loader<Integer>(){

            @Override
            public Integer load(Stream s) throws IOException {
                return s.readInteger();
            }
        });
        int contentVersion = this.myContentStorage.getVersion();
        return storageVersion == this.getVersion() && contentVersion == this.getVersion();
    }

    private void storeVersion() {
        this.store(VERSION_FILE, new Storer(){

            @Override
            public void store(Stream s) throws IOException {
                s.writeInteger(Storage.this.getVersion());
            }
        });
        this.myContentStorage.setVersion(this.getVersion());
    }

    private void initContentStorage() {
        try {
            this.myContentStorage = this.createContentStorage();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    protected IContentStorage createContentStorage() throws IOException {
        return ContentStorage.createContentStorage(new File(this.myDir, CONTENTS_FILE));
    }

    protected IContentStorage getContentStorage() {
        return this.myContentStorage;
    }

    public LocalVcs.Memento load() {
        return this.load(STORAGE_FILE, new LocalVcs.Memento(), new Loader<LocalVcs.Memento>(){

            @Override
            public LocalVcs.Memento load(Stream s) throws IOException {
                LocalVcs.Memento m = new LocalVcs.Memento();
                m.myRoot = s.readEntry();
                m.myEntryCounter = s.readInteger();
                m.myChangeList = s.readChangeList();
                return m;
            }
        });
    }

    public void saveState(final LocalVcs.Memento m) {
        this.store(STORAGE_FILE, new Storer(){

            @Override
            public void store(Stream s) throws IOException {
                s.writeEntry(m.myRoot);
                s.writeInteger(m.myEntryCounter);
                s.writeChangeList(m.myChangeList);
            }
        });
    }

    protected int getVersion() {
        return 19;
    }

    public void saveContents() {
        this.myContentStorage.save();
    }

    public void close() {
        this.myContentStorage.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T> T load(String fileName, T def, Loader<T> loader) {
        File f = new File(this.myDir, fileName);
        if (!f.exists()) {
            return def;
        }
        try {
            BufferedInputStream fs = new BufferedInputStream(new FileInputStream(f));
            try {
                T t = loader.load(new Stream(fs, this));
                return t;
            }
            finally {
                ((InputStream)fs).close();
            }
        }
        catch (IOException e) {
            this.deleteStorage();
            this.initStorage();
            return def;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void store(String fileName, Storer storer) {
        File f = new File(this.myDir, fileName);
        try {
            FileUtil.createParentDirs((File)f);
            BufferedOutputStream fs = new BufferedOutputStream(new FileOutputStream(f));
            try {
                storer.store(new Stream(fs));
            }
            finally {
                ((OutputStream)fs).close();
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Content storeContent(byte[] bytes) {
        if (this.isBroken) {
            return new UnavailableContent();
        }
        try {
            int id = this.myContentStorage.store(bytes);
            return new StoredContent(this, id);
        }
        catch (BrokenStorageException e) {
            this.markAsBroken(e);
            return new UnavailableContent();
        }
    }

    protected byte[] loadContentData(int id) throws BrokenStorageException {
        if (this.isBroken) {
            throw new BrokenStorageException();
        }
        try {
            return this.myContentStorage.load(id);
        }
        catch (BrokenStorageException e) {
            this.markAsBroken(e);
            throw e;
        }
    }

    private void markAsBroken(BrokenStorageException cause) {
        LOG.warn("Local History storage is broken. It will be rebuilt on project reopen.", (Throwable)cause);
        this.isBroken = true;
        FileUtil.createIfDoesntExist((File)new File(this.myDir, BROKEN_MARK_FILE));
    }

    public void purgeContents(List<Content> contents) {
        for (Content c : contents) {
            c.purge();
        }
    }

    protected void purgeContent(StoredContent c) {
        this.myContentStorage.remove(c.getId());
    }

    private static interface Storer {
        public void store(Stream var1) throws IOException;
    }

    private static interface Loader<T> {
        public T load(Stream var1) throws IOException;
    }
}

