/* $Id: gg.h,v 1.14.2.1 2004/10/18 15:15:54 cegger Exp $
******************************************************************************
  
   LibGG - API header file
  
   Copyright (C) 1998-1999 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
   
******************************************************************************
*/

#ifndef _GGI_GG_H
#define _GGI_GG_H

#include <ggi/errors.h>
#include <ggi/system.h>
#include <ggi/gg-defs.h>

#include <stdio.h>
#include <stdlib.h>



__BEGIN_DECLS
/*
******************************************************************************
 Initialize/Deinitialize library
******************************************************************************
*/

GGAPIFUNC int ggInit(void);

GGAPIFUNC int ggExit(void);

/*
******************************************************************************
 Time functions
******************************************************************************
*/

GGAPIFUNC int ggCurTime(struct timeval *tv);

GGAPIFUNC int ggUSleep(sint32 usecs);

GGAPIFUNC void ggUSlumber(sint32 usecs);

/*
******************************************************************************
 Functions for adding and removing cleanup callbacks
******************************************************************************
*/

typedef void (ggcleanup_func)(void *);

GGAPIFUNC int ggRegisterCleanup(ggcleanup_func *func, void *arg);

GGAPIFUNC int ggUnregisterCleanup(ggcleanup_func *func, void *arg);

/* Try to force _exit() when a handler is triggered */
GGAPIFUNC void ggCleanupForceExit(void);


/*
******************************************************************************
 Functions and structures for using the task scheduler
******************************************************************************
*/

/* Scheduler tick counter wraps at this value */
#define GG_SCHED_TICK_WRAP      32768

#define GG_SCHED_TICKS2USECS(ticks) ((ticks) * ggTimeBase())
#define GG_SCHED_USECS2TICKS(usecs) ((usecs) / ggTimeBase())

struct gg_task;
typedef int (gg_task_callback_fn)(struct gg_task *task);

struct gg_task {
	gg_task_callback_fn	*cb;    /* Function to call to run task	     */
	void			*hook;  /* Task data can be hung here        */
	int                     pticks; /* Run once every pticks ticks.      */
	int                     ncalls; /* Run ncalls times (0 = infinite)   */

/* The rest of this structure is private/internal, do not touch.
 * It should be initialized to all NULL before adding the task.
 */
	void                    *exelock;   /* Prevents reentry              */
	int                     lasttick;   /* # of tick when task last run  */

/* Double-linked, reorderable ring-list of all registered tasks.             */
	struct gg_task          *next;
	struct gg_task          *last;

/* Double-linked ordered linear-list of tasks with expired deadlines         */
	struct gg_task          *nextdl;
	struct gg_task          *lastdl;
};

GGAPIFUNC int ggAddTask(struct gg_task *task);
GGAPIFUNC int ggDelTask(struct gg_task *task);
GGAPIFUNC uint32 ggTimeBase(void);


/*
******************************************************************************
 Misc functions
******************************************************************************
*/

/* Get the users "home" directory, or whatever the equivalent is under the
   system we are running on. */
GGAPIFUNC const char *ggGetUserDir(void);

/*
******************************************************************************
 Safe string handling functions
******************************************************************************
*/

/*
 * Copy src to string dst of size siz.  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 * Returns strlen(src); if retval >= siz, truncation occurred.
 */
GGAPIFUNC size_t ggstrlcpy(char *dst, const char *src, size_t siz);

/*
 * Appends src to string dst of size siz (unlike strncat, siz is the
 * full size of dst, not space left).  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz <= strlen(dst)).
 * Returns strlen(src) + MIN(siz, strlen(initial dst)).
 * If retval >= siz, truncation occurred.
 */
GGAPIFUNC size_t ggstrlcat(char *dst, const char *src, size_t siz);


/*
******************************************************************************
 Configuration file handling
******************************************************************************
*/

GGAPIFUNC int ggLoadConfig(const char *file, void **confret);

GGAPIFUNC void ggFreeConfig(void *conf);

GGAPIFUNC const char *ggMatchConfig(const void *conf, const char *name,
				    const char *version);

GGAPIFUNC int ggConfigExpandAlias(void *confhandle, const char *list_in,
				  char *list_out, size_t outmax);

/*
******************************************************************************
 Dynamic module loading
******************************************************************************
*/

typedef void *gg_module;

#define GG_MODULE_NULL	NULL

#define GG_MODULE_GLOBAL 1

/* Dynamicly load file */
GGAPIFUNC gg_module ggLoadModule(const char *filename, int flags);

/* Try to match name in conf and load the resulting filename */
GGAPIFUNC gg_module ggMLoadModule(const void *conf, const char *name, 
				  const char *version, int flags);

/* Get Address of symbol in module */
GGAPIFUNC void *ggGetSymbolAddress(gg_module module, const char *symbol);

/* Unload a module and free resources */
GGAPIFUNC void ggFreeModule(gg_module module);


/*
******************************************************************************
 Parsing functions
******************************************************************************
*/

/* Read the first matching option from an option file into the corresponding
   result entry */
GGAPIFUNC int ggGetFileOpt(FILE *fp, const char **optnames,
			   char **results, int ressize);

GGAPIFUNC char *ggParseTarget(const char *str, char *target, int max);

#define GG_MAX_OPTION_NAME    32
#define GG_MAX_OPTION_RESULT  256

typedef struct gg_option
{
	char name[GG_MAX_OPTION_NAME];
	char result[GG_MAX_OPTION_RESULT];
} gg_option;

GGAPIFUNC char *ggParseOptions(const char *str, gg_option *optlist,
			       int count);


/*
******************************************************************************
 Debug output
******************************************************************************
*/

GGAPIFUNC void ggDPrintf(int _sync, const char *subsys,
			 const char *form, ...);

/*
******************************************************************************
 Mutex locking
******************************************************************************
*/

GGAPIFUNC void *ggLockCreate(void);

GGAPIFUNC int ggLockDestroy(void *lock);

GGAPIFUNC void ggLock(void *lock);

GGAPIFUNC void ggUnlock(void *lock);

GGAPIFUNC int ggTryLock(void *lock);

/*
******************************************************************************
 CPU Identification and SIMD disable/enable
******************************************************************************
*/

/* These are exclusive even between architectures to limit cpp complexity.
 * And who knows, perhaps cross-arch emulations may happen.
 */
enum gg_swartype {
	GG_SWAR_NONE	=	0x00000001, /* Vanilla C implementation */
	GG_SWAR_32BITC	=	0x00000002, /* Fast 32b math vs 16b */
	GG_SWAR_ALTIVEC	=	0x00000004,
	GG_SWAR_SSE	=	0x00000008,
	GG_SWAR_SSE2	=	0x00000010,
	GG_SWAR_MMX	=	0x00000020,
	GG_SWAR_MMXPLUS	=	0x00000040, /* Cyrix. */
	GG_SWAR_3DNOW	=	0x00000080, /* Implies MMX enhancements */
	GG_SWAR_ADV3DNOW=	0x00000100, /* Implies MMX enhancements */
	GG_SWAR_MAX	=	0x00000200,
	GG_SWAR_SIGD	=	0x00000400, /* 32b or 64b?? */
	GG_SWAR_SSE3    =	0x00000800,

#ifdef GG_HAVE_INT64

	GG_SWAR_64BITC	=	0x0000000100000000LL, /* Fast 64b vs 32b    */
	GG_SWAR_MVI	=	0x0000000200000000LL, /* avail in 32b mode? */
	GG_SWAR_MAX2	=	0x0000000400000000LL,
	GG_SWAR_MDMX	=	0x0000000800000000LL, /* avail in 32b mode? */
	GG_SWAR_MAJC	=	0x0000001000000000LL,
	GG_SWAR_VIS	=	0x0000002000000000LL,


	GG_SWAR_ALL = 0xffffffffffffffffLL
#else
	GG_SWAR_ALL = 0xffffffff

#endif

};

GGAPIFUNC enum gg_swartype ggGetSwarType(void);

__END_DECLS

#endif /* _GGI_GG_H */

