# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

from drawable import Drawable
import pgm

class Text(Drawable, pgm.Text):
    """
    Text node of the scenegraph inheriting from pgm.Text. It is a Pigment
    text drawable with a few extra characteristics:
     - sets a default font family
     - defines a 'font_size' property allowing to set the height of the text
       using literals ('small', 'medium', etc.)

    @ivar font_size: size of the font among 'xx-small', 'x-small', 'small',
                     'medium', 'large', 'x-large' and 'xx-large'
    @type font_size: str
    """

    def __init__(self):
        pgm.Text.__init__(self)
        Drawable.__init__(self)

        # default font size is medium
        self.font_size = "medium"
        self.font_family = "Liberation Sans"

        # Multiline is activated by default in Pigment as to not break existing
        # applications in that respect, however most texts in Elisa are monoline
        # thus setting all texts to monoline per default here.
        # A direct consequence of monolining is the automatic vertical centering
        # of the text.
        self.multiline = False
        # Activate ellipsize by default so that alignment is always respected:
        # In Pigment alignment has no effect when multiline is deactivated and
        # no ellipsization is set
        self.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self.cast_shadow = True
        self.shadow_color = (0, 0, 0, 255)
        self.line_spacing = 0.3

    def _font_size_literal_to_relative(self, literal):
        # conversion scale is loosely based on synoptic table found at:
        # http://style.cleverchimp.com/font_size_intervals/altintervals.html
        conversion = {"xx-small": 0.6,
                      "x-small": 0.75,
                      "small": 0.89,
                      "medium": 1.0,
                      "large": 1.12,
                      "x-large": 1.5,
                      "xx-large": 2.3
                     }

        # size in canvas space of 'medium'
        medium_in_canvas = 6.0

        return medium_in_canvas*conversion[literal]

    def font_size__set(self, size):
        self._font_size = size
        height = self._font_size_literal_to_relative(self._font_size)
        self.font_height = height

    def font_size__get(self):
        return self._font_size

    font_size = property(fget=font_size__get, fset=font_size__set)
