
# cdiff.nss - Configuration Diff Utility
# Copyright (c) 2009 Chris Mason <chris@noodles.org.uk>
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

use File::Path;
use Net::SMTP;

##
# Prerequisites:
#
#   This script requires a Custom Container Option to be defined for each node which 
#   specifies the command required to collect the configuration. This option needs to be
#   called "cfgcmd" and will be used by the script.
#
#   e.g.
#    container:
#     options:
#      custom:
#       cfgcmd: show running-config
##

##
# Configuration Repository
#  The $dir variable defines a directory where cdiff.nss will store the old configuation files
#  so we have an old version to run a diff against. This directory will be created if it doesn't
#  exist.
my $dir = "/tmp/cdiff";

##
# Update Flag
#  The $update variable defines where the repository is updated with the latest version 
#  of the configuration every time this script is executed. The default action is to update
#  the repository, however there is also the script variable "update" which can overwrite this.
my $update = "true";

##
# Mail Details
#  The following variables define who will be sent the report when changes have been detected
#  within the configuration.
my $mail_relay = "smtp.company.com"
my $mail_from = "ns4\@company.com";
my $mail_subject = "Configuration Diff Report - " . strftime ("%a, %b %d %H:%M %Z %Y", localtime);
my @mail_recipients = (
  "user\@company.com"
);

##
# ns4_version
#  This variable defines the minimum version of ns4 that is required to run this script. As this 
#  variable was introduced in v4.3.1 it doesn't really have any use at the moment as any script
#  within this format before v4.3.1 won't work anyway.
$ns4_version = "4.3.1";

##
# script_version
#  This variables defines the version banner that will be printed before ns4 connects to any nodes.
$script_version = <<SV;
Configuration Diff Utility v1.1
Copyright (c) 2009 Chris Mason <chris\@noodles.org.uk>
SV


######################
## MAIN SCRIPT BODY ##
######################

my $sO = new ScriptObject;
my $tfile = "/tmp/.cdiff." . $sO->dvar ("pid") . "." . strftime ("%Y%m%d.%H%M%S", localtime) . ".dat";
unlink ($tfile);

sub pre {
  $update = (defined $sO->svar ("update")) ? $sO->svar ("update") : $update;

  if ($update !~ m/^(true|false)$/i) {
    return ("Script Variable 'update' Not Recognised");
  }

  if (not -d $dir) {
    if (not mkpath ($dir)) {
      return ("Config Repository '" . $dir . "' - Permission Denied");
    }
  }
  elsif (not -w $dir) {
    return ("Config Repository '" . $dir . "' - Permission Denied");
  }
  return (100);
}

sub main {
  my ($plv, $rc) = (0, 0);
  my $loc_update = $update;

  if (not defined $sO->dvar ("c:cfgcmd")) {
    return ("Custom Node Option 'cfgcmd' Not Defined");
  }

  my @routerConfig = $sO->cmd ($sO->dvar ("c:cfgcmd"));
  my $cfg_b = $sO->create_config_tree (\@routerConfig);

  if (open (my $fh, "<", $dir . "/" . $sO->dvar ("node") . ".config.txt")) {
    my @s_cfg = <$fh>;
    close ($fh);

    my $cfg_a = $sO->create_config_tree (\@s_cfg);
    my $diff = $sO->diff_configs ($cfg_a, $cfg_b);

    if (keys %{$diff}) {
      if ($sO->lock) {
        my $fA = (-f $tfile) ? 1 : 0;
        if (open (my $fh, ">>", $tfile)) {
          print $fh "\n\n" if ($fA);
          print $fh "--- " . $sO->dvar ("node") . " - " . strftime ("%a, %b %d %H:%M %Z %Y", localtime) . " ---\n\n";
          $sO->output_config_diff ($diff, $fh);
          close ($fh);
        }
        else {
          $sO->unlock;
          return ("Permission Denied '" . $tfile . "'");
        }
        $sO->unlock;
      }
      else {
        return ("Lock Failed");
      }
    }
    else {
      $loc_update = "false";
    }
  }
  else {
    $loc_update = "true";
  }

  if ($loc_update eq "true") {
    if (open (my $fh, ">", $dir . "/" . $sO->dvar ("node") . ".config.txt")) {
      foreach my $l (@routerConfig) {
        print $fh $l . "\n";
      }
      close ($fh);
    }
    else {
      return ("Update Failed - Permission Denied");
    }
  }
  return (100);
}

sub post {
  if (-f $tfile) {
    my $sSMTP = new Net::SMTP ($mail_relay, Timeout => 10);
    if ($sSMTP) {
      $sSMTP->mail ($mail_from);
      if (not $sSMTP->ok) {
        (my $mR = $sSMTP->message) =~ s/^\s*(.*?)\s*$/$1/s;
        $sSMTP->quit; unlink ($tfile); return ($mR);
      }
      foreach my $r (@mail_recipients) {
        $sSMTP->to ($r);
        if (not $sSMTP->ok) {
          (my $mR = $sSMTP->message) =~ s/^\s*(.*?)\s*$/$1/s;
          $sSMTP->quit; unlink ($tfile); return ($mR);
        }
      }
      $sSMTP->data;
      if (not $sSMTP->ok) {
        (my $mR = $sSMTP->message) =~ s/^\s*(.*?)\s*$/$1/s;
        $sSMTP->quit; unlink ($tfile); return ($mR);
      }

      $sSMTP->datasend ("From: " . $mail_from . "\n");
      $sSMTP->datasend ("To: " . $_ . "\n") foreach (@mail_recipients);
      $sSMTP->datasend ("Subject: " . $mail_subject . "\n\n");

      if (open (my $fh, "<", $tfile)) {
        while (my $l = <$fh>) {
          $sSMTP->datasend ($l);
        }
        close ($fh);
      }
      else {
        $sSMTP->datasend ($!);
      }

      $sSMTP->dataend;
      if (not $sSMTP->ok) {
        (my $mR = $sSMTP->message) =~ s/^\s*(.*?)\s*$/$1/s;
        $sSMTP->quit; unlink ($tfile); return ($mR);
      }
      $sSMTP->quit;
    }
    else {
      unlink ($tfile);
      return ("Send Diff Failed - Connection Timed Out");
    }
    unlink ($tfile);
    return ("100:Diff Sent");
  }
  return ("100:No Changes");
}

1;
