;;;;;;;;;;;;;;;;;;;;;;;;;;;;;<--OGI-->;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                        ;;
;;             Center for Spoken Language Understanding                   ;;
;;        Oregon Graduate Institute of Science & Technology               ;;
;;                         Portland, OR USA                               ;;
;;                        Copyright (c) 2000                              ;;
;;                                                                        ;;
;;      This module is not part of the CSTR/University of Edinburgh       ;;
;;               release of the Festival TTS system.                      ;;
;;                                                                        ;;
;;  In addition to any conditions disclaimers below, please see the file  ;;
;;  "license_ogi_tts.txt" distributed with this software for information  ;;
;;  on usage and redistribution, and for a DISCLAIMER OF ALL WARRANTIES.  ;;
;;                                                                        ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;<--OGI-->;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(require 'pos)

(set! OGI_USE_TRAINED_LTS nil)

(define (setup_ogi_lex)
"(setup_ogi_lex)
Define and setup the OGI compilation of the MOBY and CMU lexicons.  This is derived 
from the public domain version of the Moby (TM) Pronunciator II lexicon and CMU dict."
 (if (not (member_string "ogi" (lex.list)))
     (begin  (lex.create "ogi")
	     (lex.set.compile.file (string-append lexdir "ogi_lex.out"))
	     (lex.set.phoneset "worldbet")

	     (if OGI_USE_TRAINED_LTS
		 (lex.set.lts.method 'ogi_lts_function)
		 (begin
		   (lex.set.lts.method 'lts_rules)
		   (lex.set.lts.ruleset 'lts_worldbet)
		   ))

	     (lex.set.pos.map english_pos_map_wp39_to_wp20)
	     (worldbet_addenda)
	     )))

(define (ogi_lts_function word feats)
  "(ogi_lts_function word feats)
Function called for OGILEX when word is not found in lexicon.  Uses
LTS rules trained from the original lexicon, and lexical stress
prediction rules."
  (require 'lts)
  (if OGI_USE_TRAINED_LTS
      (if (not (boundp 'ogilex_lts_rules))
	  (load (path-append lexdir "ogilex_lts_rules.scm")))
      )
  (let ((dcword (downcase word)) (syls) (phones))
    (if (string-matches dcword "[a-z]*")
	(begin
	  (if OGI_USE_TRAINED_LTS
	      (set! phones (lts_predict dcword ogilex_lts_rules))
	      (set! phones (lts_predict dcword lts_worldbet))
	      )
	  (set! syls (lex.syllabify.phstress phones))
	  )
	(set! syls nil))
    (list word nil syls)))

(define (worldbet_addenda)
"(worldbet_addenda)
Add a whole host of various entries to the current lexicon with
worldbet phones."
  (lex.add.entry 
   '("Tedline" nil (((t E d) 2) ((l i: n) 1))))
  (lex.add.entry 
   '("read" riid (((9r i: d) 1))))
  (lex.add.entry 
   '("read" red (((9r E d) 1))))
  (lex.add.entry
   '("a" n (((ei) 1))))
  (lex.add.entry
   '("ae" n (((ei) 1))))
  (lex.add.entry
   '("a" dt (((&) 0))))
  (lex.add.entry
   '("'s" pos (((& z) 0))))
  (lex.add.entry
   '("'s" n (((& z) 0))))
  (lex.add.entry
   '("the" det (((D &) 0))))
  (lex.add.entry 
   '("voices" n (((v >i) 1) ((s E z) 0))))
  (lex.add.entry 
   '("caught" v (((k > t) 1))))
  (lex.add.entry
   '("UN" n ((( j u ) 0) ((E n) 1))))
  (lex.add.entry
   '("parametrically" j (((p E 9r) 2) ((&) 0) ((m E) 1) ((tS 9r I k) 0) ((l i:) 0))))
  (lex.add.entry
   '("graduate" j (((g 9r @) 1) ((dZ u) 0) ((E t) 0))))
  (lex.add.entry
   '("graduate" n (((g 9r @) 1) ((dZ u) 0) ((E t) 0))))
  (lex.add.entry
   '("graduate" v (((g 9r @) 1) ((dZ u) 0) ((ei t) 0))))
  (lex.add.entry
   '("articulators" n (((A 9r) 0) ((t I k) 1) ((j &) 0) ((l ei t) 1) ((3r z) 0))))
  (lex.add.entry
   '("Institute" n (((I n) 1) ((s t I) 0) ((t u t) 0))))
  (lex.add.entry
   '("Hosom" n (((h oU) 1) ((s ^ m) 0))))
  (lex.add.entry
   '("Macon" n (((m ei) 1) ((k ^ n) 0))))
  (lex.add.entry
   '("edges" n (((E dZ) 1) ((I z) 0))))
  (lex.add.entry
   '("Festival" n (((f E s) 1) ((t I) 0) ((v & l) 0))))
  (lex.add.entry
   '("University" n ((( iU ) 1) ((n I) 0) ((v 3r) 1) ((s I) 0) ((t i:) 0))))
  (lex.add.entry 
   '( "awb" n ((( ei ) 1) ((d ^) 1) ((b & l) 0) ((y u) 0) ((b i:) 1))))
  (lex.add.entry 
   '( "pault" n ((( p > l) 1) ((t i:) 1))))
  (lex.add.entry 
   '( "Taylor" n ((( t ei) 1) ((l 3r) 0))))
  (lex.add.entry 
   '( "cstr" n ((( s i: ) 1) (( E s ) 1) (( t i: ) 1) (( A 9r) 1)) ))
  (lex.add.entry 
   '( "edinburgh" n ((( E d ) 1) (( I n ) 0) ((b 9r &) 0))))
  (lex.add.entry 
   '( "tts" n ((( t i: ) 1) (( t i: ) 1) (( E s ) 1))))
  (lex.add.entry 
   '( "emacs" n ((( i: ) 1) (( m @ k s) 0))))
  (lex.add.entry 
   '( "mbrola" n (((E m) 1) ((b 9r) 1 ) (( oU l ) 0) (( & ) 0))))
  (lex.add.entry 
   '( "psola" n ((( p I ) 0) (( s oU  ) 1) (( l & ) 0 ))))
  (lex.add.entry 
   '( "tobi" n ((( t oU ) 1) (( b i: ) 0))))
  (lex.add.entry 
   '( "ulaw" n ((( m iU ) 1) (( l > ) 0))))
  (lex.add.entry 
   '( "Los" n ((( l > s) 1))))
  (lex.add.entry 
   '( "Angeles" n (((@ n) 1) ((dZ &) 0) ((l I s) 0))))
  (lex.add.entry 
   '( "Jr" n (((dZ u n) 1) ((j 3r) 0)) ((pos "K6%" "OA%"))))
  (lex.add.entry
   '("diphone" n (((d aI) 1) ((f oU n) 0))))
  (lex.add.entry
   '("diphones" n (((d aI) 1) ((f oU n s) 0))))
  (lex.add.entry 
   '("reuter" n (((9r >i) 1) ((t 3r) 0))))
  (lex.add.entry 
   '("reuters" n (((9r >i) 1) ((t 3r z) 0))))
  (lex.add.entry 
   '("alices" n (((@ l) 1) ((I s) 0) ((E z) 0))))
  (lex.add.entry
   '("cdrom" n (((s i:) 1) ((d i:) 1) ((9r A m) 1))))
  (lex.add.entry
   '("cdroms" n (((s i:) 1) ((d i:) 1) ((9r A m z) 1))))
  (lex.add.entry
   '("sony" n (((s oU) 1) ((n i:) 1))))
  (lex.add.entry
   '("ogi" n (((oU) 1) ((dZ i:) 1) ((aI) 1))))
  (lex.add.entry
   '("doin" v (((d u) 1) ((I n) 0))))
  (lex.add.entry
   '("prototyping" n (((p 9r oU) 1) ((t &) 0) ((t aI) 0) ((p I N) 0))))
  (lex.add.entry 
   '("ae" n (((ei) 1))))  ;; used in spelling 'a'
  (lex.add.entry 
   '("*" n (((@ s) 1) ((t &) 0) ((9r I s k) 0)) ((pos "K6%"))))
  (lex.add.entry 
   '("%" n (((p 3r) 1) ((s E n t) 1)) ((pos "K9%"))))
  (lex.add.entry 
   '("&" n (((@ n d) 1))))
  (lex.add.entry 
   '("$" n (((d A l) 1) ((3r) 0))))
  (lex.add.entry 
   '("#" n (((h @ S) 1))))
  (lex.add.entry 
   '("@" n (((@ t) 1))))
  (lex.add.entry 
   '("+" n (((p l ^ s) 0)) ((pos "K7%" "OA%" "T-%"))))
  (lex.add.entry 
   '("^" n (((k E 9r) 1) ((E t) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("~" n (((t I l d) 1) ((&) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("=" n (((i: k) 1) ((w & l) 0))))
  (lex.add.entry 
   '("/" n (((s l @ S) 1))))
  (lex.add.entry 
   '("\\" n (((b @ k) 1) ((s l @ S) 1))))
  (lex.add.entry 
   '("_" n (((^ n d) 1) ((3r) 0) ((s k > 9r) 1))))
  (lex.add.entry 
   '("|" n (((v 3r t) 1) ((I k) 0) ((& l) 0) ((b A 9r) 1))))
  (lex.add.entry 
   '(">" n ((( g 9r ei ) 1) ((t 3r) 0) ((T @ n) 1))))
  (lex.add.entry 
   '("<" n ((( l E s) 1) ((T @ n) 1))))
  (lex.add.entry 
   '("[" n ((( l E f t) 1) ((b 9r @ k) 1) ((I t) 0))))
  (lex.add.entry 
   '("]" n ((( 9r aI t) 1) ((b 9r @ k) 1) ((I t) 0))))
  (lex.add.entry 
   '(" " n (((s p ei s) 1))))
  (lex.add.entry 
   '("\t" n (((t @ b) 1))))
  (lex.add.entry 
   '("\n" n (((n iU) 1) ((l aI n) 1))))
)

(provide 'ogi_lexicons)

