#!/usr/bin/python3
# Copyright (C) 2022-2024 Free Software Foundation, Inc.
# This file is part of the GNU C Library.
#
# The GNU C Library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# The GNU C Library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with the GNU C Library; if not, see
# <https://www.gnu.org/licenses/>.
"""Generate macros for getting GPR name of a certain size

Inputs: None
Output: Prints header fill to stdout

API:
    V{upcase_GPR_name}
        - Get register name REG_WIDTH component of `upcase_GPR_name`
    {upcase_mask_insn_without_postfix}
        - Get proper REG_WIDTH mask insn for `upcase_mask_insn_without_postfix`
    VGPR(reg_name)
        - Get register name REG_WIDTH component of `reg_name`
    VKINSN(mask_insn)
        - Get proper REG_WIDTH mask insn for `mask_insn`
    VGPR_SZ(reg_name, reg_size)
        - Get register name `reg_size` component of `reg_name`
    VKINSN_SZ(mask_insn, insn_size)
        - Get proper `insn_size` mask insn for `mask_insn`
"""

import sys
import os
from datetime import datetime

registers = [["rax", "eax", "ax", "al"], ["rbx", "ebx", "bx", "bl"],
             ["rcx", "ecx", "cx", "cl"], ["rdx", "edx", "dx", "dl"],
             ["rbp", "ebp", "bp", "bpl"], ["rsp", "esp", "sp", "spl"],
             ["rsi", "esi", "si", "sil"], ["rdi", "edi", "di", "dil"],
             ["r8", "r8d", "r8w", "r8b"], ["r9", "r9d", "r9w", "r9b"],
             ["r10", "r10d", "r10w", "r10b"], ["r11", "r11d", "r11w", "r11b"],
             ["r12", "r12d", "r12w", "r12b"], ["r13", "r13d", "r13w", "r13b"],
             ["r14", "r14d", "r14w", "r14b"], ["r15", "r15d", "r15w", "r15b"]]

mask_insns = [
    "kmov",
    "kortest",
    "kor",
    "ktest",
    "kand",
    "kxor",
    "knot",
    "kxnor",
]
mask_insns_ext = ["b", "w", "d", "q"]

cr = """
   Copyright (C) {} Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <https://www.gnu.org/licenses/>.  */
"""

print("/* This file was generated by: {}.".format(os.path.basename(
    sys.argv[0])))
print(cr.format(datetime.today().year))

print("#ifndef _REG_MACROS_H")
print("#define _REG_MACROS_H\t1")
print("")
for reg in registers:
    for i in range(0, 4):
        print("#define {}_{}\t{}".format(reg[0], 8 << i, reg[3 - i]))

print("")
for mask_insn in mask_insns:
    for i in range(0, 4):
        print("#define {}_{}\t{}{}".format(mask_insn, 8 << i, mask_insn,
                                           mask_insns_ext[i]))
for i in range(0, 3):
    print("#define kunpack_{}\tkunpack{}{}".format(8 << i, mask_insns_ext[i],
                                                   mask_insns_ext[i + 1]))
mask_insns.append("kunpack")

print("")
print(
    "/* Common API for accessing proper width GPR is V{upcase_GPR_name}.  */")
for reg in registers:
    print("#define V{}\tVGPR({})".format(reg[0].upper(), reg[0]))

print("")

print(
    "/* Common API for accessing proper width mask insn is {upcase_mask_insn}.  */"
)
for mask_insn in mask_insns:
    print("#define {} \tVKINSN({})".format(mask_insn.upper(), mask_insn))
print("")

print("#ifdef USE_WIDE_CHAR")
print("# define REG_WIDTH 32")
print("#else")
print("# define REG_WIDTH VEC_SIZE")
print("#endif")
print("")
print("#define VPASTER(x, y)\tx##_##y")
print("#define VEVALUATOR(x, y)\tVPASTER(x, y)")
print("")
print("#define VGPR_SZ(reg_name, reg_size)\tVEVALUATOR(reg_name, reg_size)")
print("#define VKINSN_SZ(insn, reg_size)\tVEVALUATOR(insn, reg_size)")
print("")
print("#define VGPR(reg_name)\tVGPR_SZ(reg_name, REG_WIDTH)")
print("#define VKINSN(mask_insn)\tVKINSN_SZ(mask_insn, REG_WIDTH)")

print("\n#endif")
