/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifndef _VRGUPPISOURCE_H_
#define _VRGUPPISOURCE_H_

extern "C" {
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
}

#include <VrSource.h>
#include <VrGuppiBuffer.h>
#include <guppi.h>

template<class oType> 
class VrGuppiSource: public VrSource<oType> {
protected:
  int gupfd; 
  //the guppi driver page index corresponding to WP (or ts after sync in work)
  //(the NEXT block the guppi will give us)
  int currentIndex;
  int minIndex; //the smallest guppi driver block we are using
  //if minIndex == currentIndex we aren't using any blocks
  int extrablocks; //(how many blocks beyond WP are actually valid)

public: 
  virtual const char *name() { return "VrGuppiSource"; }
  virtual float memoryTouched() {
    return 0; //no outputs are cached
  }
  virtual int work2(VrSampleRange output, void *o[]);
  virtual void initOutputBuffer(int n);
  VrGuppiSource(double sampling_freq);
};

template<class oType> int
VrGuppiSource<oType>::work2(VrSampleRange output, void *ao[])
{
  struct guppi_status status;
  int num;
  int blocksleft;

  sync(output.index);

  blocksleft = output.size/getOutputSize();
  num=currentIndex - minIndex;
  status.index=minIndex;
  blocksleft-=extrablocks;

  while (blocksleft>0) { 
    /*free some blocks if appropriate*/
    //how many blocks the driver thinks we're using
    // minus how many blocks we are really using
    status.num=num-(output.index+output.size-proc_minRP()+PAGE_SIZE-1)/PAGE_SIZE*sizeof(oType)+blocksleft;

    if (PARANOID && status.num < 0) {
      fprintf(stderr,"Guppi index inconsistency.\n");
      abort ();
    }

    //guppi driver won't let us throw out more blocks than we've read
    if((int)status.num>num) status.num=num-1;

    num-=status.num;

    //free the blocks and find out how many blocks are ready
    if((ioctl(gupfd, GIOCSETGETSTATUS, &status)) < 0) {
      perror("Failed to get guppi status");
      exit(-1);
    }

    //subtract how many new blocks we got
    blocksleft-=status.num-num;
    num=status.num;

    //do something about lost blocks
    if(status.lost) {
      fprintf(stderr,"Lost %i pages\n",status.lost);
    }

    if(blocksleft>0)
      YIELD(); //FIX return how many we actually got.
  }
  minIndex=status.index;
  currentIndex=status.index+num;
  extrablocks=-blocksleft;
  return output.size;
}

template<class oType> void
VrGuppiSource<oType>::initOutputBuffer(int n)
{
  if(n!=0) {
    fprintf(stderr,"GuppiSource can only have one output buffer.\n");
    exit(-1);
  }
  outBuffer[0]=new VrGuppiBuffer<oType>(this,gupfd);
}

template<class oType>
VrGuppiSource<oType>::VrGuppiSource(double sampling_freq)
  :gupfd(-1), currentIndex(0), minIndex(0), extrablocks(0)
{
  if ((gupfd = open("/dev/guppi0", (O_RDONLY | O_NONBLOCK))) < 0) {
    printf("Can't open GuPPI\n");
    exit(1);
  }
  setSamplingFrequency (sampling_freq);
  
  // add a query here to find out what data type(s) the GuPPI supports
  // Defaults to char for now
  setOutputSize (PAGE_SIZE/sizeof(oType));
}

#endif 
