/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRFSKDEMOD_H_
#define _VRFSKDEMOD_H_

// need to manually adjust threshold
#define FSK_THRESHOLD 5000

#include <VrSigProc.h>


template<class iType> 
class VrFSKDemod : public VrSigProc<iType,bit> {
protected:
  int bitLength;
  int dataRate;
  int syncStatus;
  virtual void initialize();
public:   
  virtual void work(int n);  
  VrFSKDemod(int bit_rate);
  VrFSKDemod(int bit_rate,int output_size);
};

template<class iType> void
VrFSKDemod<iType>::work(int n)
{
  static int below_threshold = 0;
  iType prevVal, inputVal;
  float a,b,x,y;
  float imaginary;
  int thresh_count=0;
  int zero_count = 0;
  int step_count = 0;

  while (n > 0) {
    if (!syncStatus) {
      
      float base = 0.0;
      do {
	incReadPtr(bitLength);
	inputVal = inputRead(0);
	a = real(inputVal);
	b = imag(inputVal);
	
	if ((a*a+b*b) > FSK_THRESHOLD) {
	  thresh_count++;
	  prevVal = inputRead(-1);
	  x = real(prevVal);
	  y = imag(prevVal);
	  base = x*b - a*y;
	  if (base < 0) zero_count++;
	  else zero_count = 0;
	}
	else {
	  thresh_count = 0;
	  zero_count = 0;
	}
	//	printf("%f\n",abs(base));
      }	while (thresh_count < 5);

      imaginary = base;
      if (base > 0) {
	while (imaginary > 0) {
	  incReadPtr(1);
	  prevVal = inputVal;
	  inputVal = inputRead();
	  x = a;
	  y = b;
	  a = real(inputVal);
	  b = imag(inputVal);
	  imaginary = x*b - a*y;
	}
      }
      else {
	while (imaginary < 0) {
	  step_count++;
	  incReadPtr(1);
	  prevVal = inputVal;
	  inputVal = inputRead();
	  x = a;
	  y = b;
	  a = real(inputVal);
	  b = imag(inputVal);
	  imaginary = x*b - a*y;
	}     
      }
      // found bit boundary, output the zeros that have been previously saved 
      int zeros = zero_count + (step_count/bitLength);
      if (zeros > 0) {
	if (zeros==14) zeros=15; /** stupid hack, sometimes not catching all
				   15 of the zeros in training seq.
				   Same hack in VrFHFSKDemod.h **/
	if (zeros > 15) zeros = 15; /** don't want to overload output buffer*/
	for (int i=0;i<zeros;i++)
	  outputWrite(0);
	outputWrite(1);
	n -= zeros+1;
      }

      thresh_count = 0;
      zero_count = 0;
      step_count = 0;

      incReadPtr(bitLength/2);  // move to center of a bit

      syncStatus = 1;      
      printf("got sync\n");
    }
    
    if (n <= 0) break;
    
    /** need to only look at img part to see if the freq is + or - **/
    prevVal = inputRead(-1);
    inputVal = inputRead(0);
    
    // float multiply
    a = real(inputVal);
    b = imag(inputVal);
    x = real(prevVal);
    y = imag(prevVal);
    
    imaginary = x*b - a*y;
    //    printf("%f %f \n",y/(x+.0000001),b/(a+.0000001));
    //    printf("%f %f %f %f %f\n",abs(imaginary),a,b,x,y);
    if (abs(imaginary) < FSK_THRESHOLD) {
      below_threshold++;
      //      printf("losing sync: %d\n",below_threshold);
    }
    else below_threshold = 0;

    if (below_threshold >= 3) {
      below_threshold = 0;
      syncStatus=0;
      printf("lost sync, sending %d nulls\n",n);
      /*  need to send NULL bits to signal end/loss of transmission */
      bit* nullbits = (bit*)new bit[n];
      memset(nullbits,-1,n);
      outputWrite(nullbits,n);
      n=0;
    }
    else
      {
	if (imaginary > 0) 
	  outputWrite(1); // positive freq
	else 
	  outputWrite(0); // negative freq
	n--;
      }
    incReadPtr(bitLength);
  }
  return;
}

template<class iType> void
VrFSKDemod<iType>::initialize() {
  int sf = getInputSamplingFrequencyN(0);
  bitLength = sf / dataRate;
  setHistory (2);

  if ((bitLength * dataRate) != sf) {
    dataRate = sf/bitLength;
    cout << "Warning: Adjusting data rate to: " << dataRate << endl;
  }
}

template<class iType> 
VrFSKDemod<iType>::VrFSKDemod(int bit_rate)
  :dataRate(bit_rate),syncStatus(0)
{
  setOutputSize(16);
}

template<class iType> 
VrFSKDemod<iType>::VrFSKDemod(int bit_rate, int os)
  :dataRate(bit_rate),syncStatus(0)
{
  setOutputSize (os);
}

#endif




