/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gr_unpacked_to_packed_bb.h>
#include <gr_io_signature.h>
#include <assert.h>

static const int BITS_PER_BYTE = 8;

gr_unpacked_to_packed_bb_sptr 
gr_make_unpacked_to_packed_bb (unsigned int bits_per_chunk, gr_endianness_t endianness)
{
  return gr_unpacked_to_packed_bb_sptr 
    (new gr_unpacked_to_packed_bb (bits_per_chunk,endianness));
}

gr_unpacked_to_packed_bb::gr_unpacked_to_packed_bb (unsigned int bits_per_chunk, 
						    gr_endianness_t endianness)
  : gr_block ("unpacked_to_packed_bb",
	      gr_make_io_signature (1, 1, sizeof (unsigned char)),
	      gr_make_io_signature (1, 1, sizeof (unsigned char))),
    d_bits_per_chunk(bits_per_chunk),d_endianness(endianness),d_index(0)
{
  assert (bits_per_chunk <= 8);
  assert (bits_per_chunk > 0);

  set_relative_rate (bits_per_chunk/8.0);
}

void
gr_unpacked_to_packed_bb::forecast(int noutput_items, gr_vector_int &ninput_items_required)
{
  ninput_items_required[0] =
    (int) ceil( (d_index+noutput_items *  8.0)/d_bits_per_chunk);
}


int
get_bit_be1 (const unsigned char *in_vector,unsigned int bit_addr, unsigned int bits_per_chunk) {
  unsigned int byte_addr = (int)bit_addr/bits_per_chunk;
  unsigned char x = in_vector[byte_addr];
  unsigned int residue = bit_addr - byte_addr * bits_per_chunk;
  //printf("Bit addr %d  byte addr %d  residue %d  val  %d\n",bit_addr,byte_addr,residue,(x>>(bits_per_chunk-1-residue))&1);
  return (x >> (bits_per_chunk-1-residue))&1;
}

int
gr_unpacked_to_packed_bb::general_work (int noutput_items,
					gr_vector_int &ninput_items,
					gr_vector_const_void_star &input_items,
					gr_vector_void_star &output_items)
{
  const unsigned char *in = (unsigned char *) input_items[0];
  unsigned char *out = (unsigned char *) output_items[0];

  //assert (noutput_items % BITS_PER_BYTE == 0);
  //assert((ninput_items[0]-d_index)*d_bits_per_chunk >= noutput_items*8);
  
  if(d_endianness == GR_MSB_FIRST){
      for(int i=0;i<noutput_items;i++) {
	  unsigned char tmp=0;
	  for(unsigned int j=0; j<8; j++) {
	      tmp = (tmp<<1) | get_bit_be1(in,d_index,d_bits_per_chunk);
	      d_index++;
	  }
	  out[i] = tmp;
      }
  }
  else {
      for(int i=0;i<noutput_items;i++) {
	  unsigned char tmp=0;
	  for(unsigned int j=0; j<8; j++) {
	      tmp = (tmp>>1)| (get_bit_be1(in,d_index,d_bits_per_chunk)<<7);
	      d_index++;
	  }
	  out[i] = tmp;
      }
  }

  consume_each ((int)(d_index/d_bits_per_chunk));
  d_index = d_index%d_bits_per_chunk;

  return noutput_items;
}
