/*
 GNU Maverik - a system for managing display and interaction in 
               Virtual Environment applications.
 Copyright (C) 1999-2002 Advanced Interfaces Group

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA


 The authors can be contacted via:
 www   - http://aig.cs.man.ac.uk
 email - maverik@aig.cs.man.ac.uk
 mail  - Advanced Interfaces Group, Room 2.94, Kilburn Building, 
         University of Manchester, Manchester, M13 9PL, UK
*/


#include "mavlib_objects.h"
#include <stdio.h>
#include <math.h>

MAV_class *mav_class_teapot;

MAV_vector mavlib_teapotPatches[32][16] = {
  {
    { 0.174329, 0.126437, 0 }, { 0.174329, 0.126437, -0.120153 }, { 0.0799234, 0.126437, -0.214559 }, { -0.0402299, 0.126437, -0.214559 }, 
    { 0.164751, 0.146552, 0 }, { 0.164751, 0.146552, -0.114789 }, { 0.0745594, 0.146552, -0.204981 }, { -0.0402299, 0.146552, -0.204981 }, 
    { 0.180077, 0.146552, 0 }, { 0.180077, 0.146552, -0.123372 }, { 0.0831418, 0.146552, -0.220307 }, { -0.0402299, 0.146552, -0.220307 }, 
    { 0.189655, 0.126437, 0 }, { 0.189655, 0.126437, -0.128736 }, { 0.0885057, 0.126437, -0.229885 }, { -0.0402299, 0.126437, -0.229885 }
  },
  {
    { -0.0402299, 0.126437, -0.214559 }, { -0.160383, 0.126437, -0.214559 }, { -0.254789, 0.126437, -0.120153 }, { -0.254789, 0.126437, 0 }, 
    { -0.0402299, 0.146552, -0.204981 }, { -0.155019, 0.146552, -0.204981 }, { -0.245211, 0.146552, -0.114789 }, { -0.245211, 0.146552, 0 }, 
    { -0.0402299, 0.146552, -0.220307 }, { -0.163602, 0.146552, -0.220307 }, { -0.260536, 0.146552, -0.123372 }, { -0.260536, 0.146552, 0 }, 
    { -0.0402299, 0.126437, -0.229885 }, { -0.168966, 0.126437, -0.229885 }, { -0.270115, 0.126437, -0.128736 }, { -0.270115, 0.126437, 0 }
  },
  {
    { -0.254789, 0.126437, 0 }, { -0.254789, 0.126437, 0.120153 }, { -0.160383, 0.126437, 0.214559 }, { -0.0402299, 0.126437, 0.214559 }, 
    { -0.245211, 0.146552, 0 }, { -0.245211, 0.146552, 0.114789 }, { -0.155019, 0.146552, 0.204981 }, { -0.0402299, 0.146552, 0.204981 }, 
    { -0.260536, 0.146552, 0 }, { -0.260536, 0.146552, 0.123372 }, { -0.163602, 0.146552, 0.220307 }, { -0.0402299, 0.146552, 0.220307 }, 
    { -0.270115, 0.126437, 0 }, { -0.270115, 0.126437, 0.128736 }, { -0.168966, 0.126437, 0.229885 }, { -0.0402299, 0.126437, 0.229885 }
  },
  {
    { -0.0402299, 0.126437, 0.214559 }, { 0.0799234, 0.126437, 0.214559 }, { 0.174329, 0.126437, 0.120153 }, { 0.174329, 0.126437, 0 }, 
    { -0.0402299, 0.146552, 0.204981 }, { 0.0745594, 0.146552, 0.204981 }, { 0.164751, 0.146552, 0.114789 }, { 0.164751, 0.146552, 0 }, 
    { -0.0402299, 0.146552, 0.220307 }, { 0.0831418, 0.146552, 0.220307 }, { 0.180077, 0.146552, 0.123372 }, { 0.180077, 0.146552, 0 }, 
    { -0.0402299, 0.126437, 0.229885 }, { 0.0885057, 0.126437, 0.229885 }, { 0.189655, 0.126437, 0.128736 }, { 0.189655, 0.126437, 0 }
  },
  {
    { 0.189655, 0.126437, 0 }, { 0.189655, 0.126437, -0.128736 }, { 0.0885057, 0.126437, -0.229885 }, { -0.0402299, 0.126437, -0.229885 }, 
    { 0.227969, 0.045977, 0 }, { 0.227969, 0.045977, -0.150192 }, { 0.109962, 0.045977, -0.268199 }, { -0.0402299, 0.045977, -0.268199 }, 
    { 0.266284, -0.0344828, 0 }, { 0.266284, -0.0344828, -0.171648 }, { 0.131418, -0.0344828, -0.306513 }, { -0.0402299, -0.0344828, -0.306513 }, 
    { 0.266284, -0.103448, 0 }, { 0.266284, -0.103448, -0.171648 }, { 0.131418, -0.103448, -0.306513 }, { -0.0402299, -0.103448, -0.306513 }
  },
  {
    { -0.0402299, 0.126437, -0.229885 }, { -0.168966, 0.126437, -0.229885 }, { -0.270115, 0.126437, -0.128736 }, { -0.270115, 0.126437, 0 }, 
    { -0.0402299, 0.045977, -0.268199 }, { -0.190421, 0.045977, -0.268199 }, { -0.308429, 0.045977, -0.150192 }, { -0.308429, 0.045977, 0 }, 
    { -0.0402299, -0.0344828, -0.306513 }, { -0.211877, -0.0344828, -0.306513 }, { -0.346743, -0.0344828, -0.171648 }, { -0.346743, -0.0344828, 0 }, 
    { -0.0402299, -0.103448, -0.306513 }, { -0.211877, -0.103448, -0.306513 }, { -0.346743, -0.103448, -0.171648 }, { -0.346743, -0.103448, 0 }
  },
  {
    { -0.270115, 0.126437, 0 }, { -0.270115, 0.126437, 0.128736 }, { -0.168966, 0.126437, 0.229885 }, { -0.0402299, 0.126437, 0.229885 }, 
    { -0.308429, 0.045977, 0 }, { -0.308429, 0.045977, 0.150192 }, { -0.190421, 0.045977, 0.268199 }, { -0.0402299, 0.045977, 0.268199 }, 
    { -0.346743, -0.0344828, 0 }, { -0.346743, -0.0344828, 0.171648 }, { -0.211877, -0.0344828, 0.306513 }, { -0.0402299, -0.0344828, 0.306513 }, 
    { -0.346743, -0.103448, 0 }, { -0.346743, -0.103448, 0.171648 }, { -0.211877, -0.103448, 0.306513 }, { -0.0402299, -0.103448, 0.306513 }
  },
  {
    { -0.0402299, 0.126437, 0.229885 }, { 0.0885057, 0.126437, 0.229885 }, { 0.189655, 0.126437, 0.128736 }, { 0.189655, 0.126437, 0 }, 
    { -0.0402299, 0.045977, 0.268199 }, { 0.109962, 0.045977, 0.268199 }, { 0.227969, 0.045977, 0.150192 }, { 0.227969, 0.045977, 0 }, 
    { -0.0402299, -0.0344828, 0.306513 }, { 0.131418, -0.0344828, 0.306513 }, { 0.266284, -0.0344828, 0.171648 }, { 0.266284, -0.0344828, 0 }, 
    { -0.0402299, -0.103448, 0.306513 }, { 0.131418, -0.103448, 0.306513 }, { 0.266284, -0.103448, 0.171648 }, { 0.266284, -0.103448, 0 }
  },
  {
    { 0.266284, -0.103448, 0 }, { 0.266284, -0.103448, -0.171648 }, { 0.131418, -0.103448, -0.306513 }, { -0.0402299, -0.103448, -0.306513 }, 
    { 0.266284, -0.172414, 0 }, { 0.266284, -0.172414, -0.171648 }, { 0.131418, -0.172414, -0.306513 }, { -0.0402299, -0.172414, -0.306513 }, 
    { 0.189655, -0.206897, 0 }, { 0.189655, -0.206897, -0.128736 }, { 0.0885057, -0.206897, -0.229885 }, { -0.0402299, -0.206897, -0.229885 }, 
    { 0.189655, -0.218391, 0 }, { 0.189655, -0.218391, -0.128736 }, { 0.0885057, -0.218391, -0.229885 }, { -0.0402299, -0.218391, -0.229885 }
  },
  {
    { -0.0402299, -0.103448, -0.306513 }, { -0.211877, -0.103448, -0.306513 }, { -0.346743, -0.103448, -0.171648 }, { -0.346743, -0.103448, 0 }, 
    { -0.0402299, -0.172414, -0.306513 }, { -0.211877, -0.172414, -0.306513 }, { -0.346743, -0.172414, -0.171648 }, { -0.346743, -0.172414, 0 }, 
    { -0.0402299, -0.206897, -0.229885 }, { -0.168966, -0.206897, -0.229885 }, { -0.270115, -0.206897, -0.128736 }, { -0.270115, -0.206897, 0 }, 
    { -0.0402299, -0.218391, -0.229885 }, { -0.168966, -0.218391, -0.229885 }, { -0.270115, -0.218391, -0.128736 }, { -0.270115, -0.218391, 0 }
  },
  {
    { -0.346743, -0.103448, 0 }, { -0.346743, -0.103448, 0.171648 }, { -0.211877, -0.103448, 0.306513 }, { -0.0402299, -0.103448, 0.306513 }, 
    { -0.346743, -0.172414, 0 }, { -0.346743, -0.172414, 0.171648 }, { -0.211877, -0.172414, 0.306513 }, { -0.0402299, -0.172414, 0.306513 }, 
    { -0.270115, -0.206897, 0 }, { -0.270115, -0.206897, 0.128736 }, { -0.168966, -0.206897, 0.229885 }, { -0.0402299, -0.206897, 0.229885 }, 
    { -0.270115, -0.218391, 0 }, { -0.270115, -0.218391, 0.128736 }, { -0.168966, -0.218391, 0.229885 }, { -0.0402299, -0.218391, 0.229885 }
  },
  {
    { -0.0402299, -0.103448, 0.306513 }, { 0.131418, -0.103448, 0.306513 }, { 0.266284, -0.103448, 0.171648 }, { 0.266284, -0.103448, 0 }, 
    { -0.0402299, -0.172414, 0.306513 }, { 0.131418, -0.172414, 0.306513 }, { 0.266284, -0.172414, 0.183908 }, { 0.266284, -0.172414, 0 }, 
    { -0.0402299, -0.206897, 0.229885 }, { 0.0885057, -0.206897, 0.229885 }, { 0.189655, -0.206897, 0.128736 }, { 0.189655, -0.206897, 0 }, 
    { -0.0402299, -0.218391, 0.229885 }, { 0.0885057, -0.218391, 0.229885 }, { 0.189655, -0.218391, 0.128736 }, { 0.189655, -0.218391, 0 }
  },
  {
    { -0.285441, 0.0689655, 0 }, { -0.285441, 0.0689655, -0.045977 }, { -0.270115, 0.103448, -0.045977 }, { -0.270115, 0.103448, 0 }, 
    { -0.39272, 0.0689655, 0 }, { -0.39272, 0.0689655, -0.045977 }, { -0.423372, 0.103448, -0.045977 }, { -0.423372, 0.103448, 0 }, 
    { -0.454023, 0.0689655, 0 }, { -0.454023, 0.0689655, -0.045977 }, { -0.5, 0.103448, -0.045977 }, { -0.5, 0.103448, 0 }, 
    { -0.454023, 0.0344828, 0 }, { -0.454023, 0.0344828, -0.045977 }, { -0.5, 0.0344828, -0.045977 }, { -0.5, 0.0344828, 0 }
  },
  {
    { -0.270115, 0.103448, 0 }, { -0.270115, 0.103448, 0.045977 }, { -0.285441, 0.0689655, 0.045977 }, { -0.285441, 0.0689655, 0 }, 
    { -0.423372, 0.103448, 0 }, { -0.423372, 0.103448, 0.045977 }, { -0.39272, 0.0689655, 0.045977 }, { -0.39272, 0.0689655, 0 }, 
    { -0.5, 0.103448, 0 }, { -0.5, 0.103448, 0.045977 }, { -0.454023, 0.0689655, 0.045977 }, { -0.454023, 0.0689655, 0 }, 
    { -0.5, 0.0344828, 0 }, { -0.5, 0.0344828, 0.045977 }, { -0.454023, 0.0344828, 0.045977 }, { -0.454023, 0.0344828, 0 }
  },
  {
    { -0.454023, 0.0344828, 0 }, { -0.454023, 0.0344828, -0.045977 }, { -0.5, 0.0344828, -0.045977 }, { -0.5, 0.0344828, 0 }, 
    { -0.454023, 7.30785e-09, 0 }, { -0.454023, 7.30785e-09, -0.045977 }, { -0.5, -0.0344828, -0.045977 }, { -0.5, -0.0344828, 0 }, 
    { -0.423372, -0.0689655, 0 }, { -0.423372, -0.0689655, -0.045977 }, { -0.44636, -0.0977011, -0.045977 }, { -0.44636, -0.0977011, 0 }, 
    { -0.346743, -0.103448, 0 }, { -0.346743, -0.103448, -0.045977 }, { -0.331418, -0.149425, -0.045977 }, { -0.331418, -0.149425, 0 }
  },
  {
    { -0.5, 0.0344828, 0 }, { -0.5, 0.0344828, 0.045977 }, { -0.454023, 0.0344828, 0.045977 }, { -0.454023, 0.0344828, 0 }, 
    { -0.5, -0.0344828, 0 }, { -0.5, -0.0344828, 0.045977 }, { -0.454023, 7.30785e-09, 0.045977 }, { -0.454023, 7.30785e-09, 0 }, 
    { -0.44636, -0.0977011, 0 }, { -0.44636, -0.0977011, 0.045977 }, { -0.423372, -0.0689655, 0.045977 }, { -0.423372, -0.0689655, 0 }, 
    { -0.331418, -0.149425, 0 }, { -0.331418, -0.149425, 0.045977 }, { -0.346743, -0.103448, 0.045977 }, { -0.346743, -0.103448, 0 }
  },
  {
    { 0.220307, -0.0229885, 0 }, { 0.220307, -0.0229885, -0.101149 }, { 0.220307, -0.149425, -0.101149 }, { 0.220307, -0.149425, 0 }, 
    { 0.358238, -0.0229885, 0 }, { 0.358238, -0.0229885, -0.101149 }, { 0.434866, -0.114943, -0.101149 }, { 0.434866, -0.114943, 0 }, 
    { 0.312261, 0.0804598, 0 }, { 0.312261, 0.0804598, -0.0383142 }, { 0.327586, 0.0689655, -0.0383142 }, { 0.327586, 0.0689655, 0 }, 
    { 0.373563, 0.126437, 0 }, { 0.373563, 0.126437, -0.0383142 }, { 0.465517, 0.126437, -0.0383142 }, { 0.465517, 0.126437, 0 }
  },
  {
    { 0.220307, -0.149425, 0 }, { 0.220307, -0.149425, 0.101149 }, { 0.220307, -0.0229885, 0.101149 }, { 0.220307, -0.0229885, 0 }, 
    { 0.434866, -0.114943, 0 }, { 0.434866, -0.114943, 0.101149 }, { 0.358238, -0.0229885, 0.101149 }, { 0.358238, -0.0229885, 0 }, 
    { 0.327586, 0.0689655, 0 }, { 0.327586, 0.0689655, 0.0383142 }, { 0.312261, 0.0804598, 0.0383142 }, { 0.312261, 0.0804598, 0 }, 
    { 0.465517, 0.126437, 0 }, { 0.465517, 0.126437, 0.0383142 }, { 0.373563, 0.126437, 0.0383142 }, { 0.373563, 0.126437, 0 }
  },
  {
    { 0.373563, 0.126437, 0 }, { 0.373563, 0.126437, -0.0383142 }, { 0.465517, 0.126437, -0.0383142 }, { 0.465517, 0.126437, 0 }, 
    { 0.388889, 0.137931, 0 }, { 0.388889, 0.137931, -0.0383142 }, { 0.5, 0.140805, -0.0383142 }, { 0.5, 0.140805, 0 }, 
    { 0.404215, 0.137931, 0 }, { 0.404215, 0.137931, -0.0229885 }, { 0.488506, 0.143678, -0.0229885 }, { 0.488506, 0.143678, 0 }, 
    { 0.388889, 0.126437, 0 }, { 0.388889, 0.126437, -0.0229885 }, { 0.450192, 0.126437, 0 }, { 0.450192, 0.126437, 0 }
  },
  {
    { 0.465517, 0.126437, 0 }, { 0.465517, 0.126437, 0.0383142 }, { 0.373563, 0.126437, 0.0383142 }, { 0.373563, 0.126437, 0 }, 
    { 0.5, 0.140805, 0 }, { 0.5, 0.140805, 0.0383142 }, { 0.388889, 0.137931, 0.0383142 }, { 0.388889, 0.137931, 0 }, 
    { 0.488506, 0.143678, 0 }, { 0.488506, 0.143678, 0.0229885 }, { 0.404215, 0.137931, 0.0229885 }, { 0.404215, 0.137931, 0 }, 
    { 0.450192, 0.126437, 0 }, { 0.450192, 0.126437, 0.0229885 }, { 0.388889, 0.126437, 0.0229885 }, { 0.388889, 0.126437, 0 }
  },
  {
    { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, 
    { 0.0823755, 0.241379, 0 }, { 0.0823755, 0.241379, -0.0689655 }, { 0.0287356, 0.241379, -0.122605 }, { -0.0402299, 0.241379, -0.122605 }, 
    { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, 
    { -0.00957854, 0.172414, 0 }, { -0.00957854, 0.172414, -0.0171648 }, { -0.0230651, 0.172414, -0.0306513 }, { -0.0402299, 0.172414, -0.0306513 } 
  },
  {
    { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, 
    { -0.0402299, 0.241379, -0.122605 }, { -0.109195, 0.241379, -0.122605 }, { -0.162835, 0.241379, -0.0689655 }, { -0.162835, 0.241379, 0 }, 
    { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, 
    { -0.0402299, 0.172414, -0.0306513 }, { -0.0573946, 0.172414, -0.0306513 }, { -0.0708812, 0.172414, -0.0171648 }, { -0.0708812, 0.172414, 0 }
  },
  {
    { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, 
    { -0.162835, 0.241379, 0 }, { -0.162835, 0.241379, 0.0689655 }, { -0.109195, 0.241379, 0.122605 }, { -0.0402299, 0.241379, 0.122605 }, 
    { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, 
    { -0.0708812, 0.172414, 0 }, { -0.0708812, 0.172414, 0.0171648 }, { -0.0573946, 0.172414, 0.0306513 }, { -0.0402299, 0.172414, 0.0306513 }
  },
  {
    { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, { -0.0402299, 0.241379, 0 }, 
    { -0.0402299, 0.241379, 0.122605 }, { 0.0287356, 0.241379, 0.122605 }, { 0.0823755, 0.241379, 0.0689655 }, { 0.0823755, 0.241379, 0 }, 
    { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, { -0.0402299, 0.195402, 0 }, 
    { -0.0402299, 0.172414, 0.0306513 }, { -0.0230651, 0.172414, 0.0306513 }, { -0.00957854, 0.172414, 0.0171648 }, { -0.00957854, 0.172414, 0 }
  },
  {
    { -0.00957854, 0.172414, 0 }, { -0.00957854, 0.172414, -0.0171648 }, { -0.0230651, 0.172414, -0.0306513 }, { -0.0402299, 0.172414, -0.0306513 }, 
    { 0.0210728, 0.149425, 0 }, { 0.0210728, 0.149425, -0.0343295 }, { -0.00590038, 0.149425, -0.0613027 }, { -0.0402299, 0.149425, -0.0613027 }, 
    { 0.159004, 0.149425, 0 }, { 0.159004, 0.149425, -0.111571 }, { 0.071341, 0.149425, -0.199234 }, { -0.0402299, 0.149425, -0.199234 }, 
    { 0.159004, 0.126437, 0 }, { 0.159004, 0.126437, -0.111571 }, { 0.071341, 0.126437, -0.199234 }, { -0.0402299, 0.126437, -0.199234 }
  },
  {
    { -0.0402299, 0.172414, -0.0306513 }, { -0.0573946, 0.172414, -0.0306513 }, { -0.0708812, 0.172414, -0.0171648 }, { -0.0708812, 0.172414, 0 }, 
    { -0.0402299, 0.149425, -0.0613027 }, { -0.0745594, 0.149425, -0.0613027 }, { -0.101533, 0.149425, -0.0343295 }, { -0.101533, 0.149425, 0 }, 
    { -0.0402299, 0.149425, -0.199234 }, { -0.151801, 0.149425, -0.199234 }, { -0.239464, 0.149425, -0.111571 }, { -0.239464, 0.149425, 0 }, 
    { -0.0402299, 0.126437, -0.199234 }, { -0.151801, 0.126437, -0.199234 }, { -0.239464, 0.126437, -0.111571 }, { -0.239464, 0.126437, 0 }
  },
  {
    { -0.0708812, 0.172414, 0 }, { -0.0708812, 0.172414, 0.0171648 }, { -0.0573946, 0.172414, 0.0306513 }, { -0.0402299, 0.172414, 0.0306513 }, 
    { -0.101533, 0.149425, 0 }, { -0.101533, 0.149425, 0.0343295 }, { -0.0745594, 0.149425, 0.0613027 }, { -0.0402299, 0.149425, 0.0613027 }, 
    { -0.239464, 0.149425, 0 }, { -0.239464, 0.149425, 0.111571 }, { -0.151801, 0.149425, 0.199234 }, { -0.0402299, 0.149425, 0.199234 }, 
    { -0.239464, 0.126437, 0 }, { -0.239464, 0.126437, 0.111571 }, { -0.151801, 0.126437, 0.199234 }, { -0.0402299, 0.126437, 0.199234 }
  },
  {
    { -0.0402299, 0.172414, 0.0306513 }, { -0.0230651, 0.172414, 0.0306513 }, { -0.00957854, 0.172414, 0.0171648 }, { -0.00957854, 0.172414, 0 }, 
    { -0.0402299, 0.149425, 0.0613027 }, { -0.00590038, 0.149425, 0.0613027 }, { 0.0210728, 0.149425, 0.0343295 }, { 0.0210728, 0.149425, 0 }, 
    { -0.0402299, 0.149425, 0.199234 }, { 0.071341, 0.149425, 0.199234 }, { 0.159004, 0.149425, 0.111571 }, { 0.159004, 0.149425, 0 }, 
    { -0.0402299, 0.126437, 0.199234 }, { 0.071341, 0.126437, 0.199234 }, { 0.159004, 0.126437, 0.111571 }, { 0.159004, 0.126437, 0 }
  },
  {
    { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, 
    { 0.178161, -0.241379, 0 }, { 0.178161, -0.241379, 0.122299 }, { 0.082069, -0.241379, 0.218391 }, { -0.0402299, -0.241379, 0.218391 }, 
    { 0.189655, -0.229885, 0 }, { 0.189655, -0.229885, 0.128736 }, { 0.0885057, -0.229885, 0.229885 }, { -0.0402299, -0.229885, 0.229885 }, 
    { 0.189655, -0.218391, 0 }, { 0.189655, -0.218391, 0.128736 }, { 0.0885057, -0.218391, 0.229885 }, { -0.0402299, -0.218391, 0.229885 }
  },
  {
    { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, 
    { -0.0402299, -0.241379, 0.218391 }, { -0.162529, -0.241379, 0.218391 }, { -0.258621, -0.241379, 0.122299 }, { -0.258621, -0.241379, 0 }, 
    { -0.0402299, -0.229885, 0.229885 }, { -0.168966, -0.229885, 0.229885 }, { -0.270115, -0.229885, 0.128736 }, { -0.270115, -0.229885, 0 }, 
    { -0.0402299, -0.218391, 0.229885 }, { -0.168966, -0.218391, 0.229885 }, { -0.270115, -0.218391, 0.128736 }, { -0.270115, -0.218391, 0 }
  },
  {
    { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, 
    { -0.258621, -0.241379, 0 }, { -0.258621, -0.241379, -0.122299 }, { -0.162529, -0.241379, -0.218391 }, { -0.0402299, -0.241379, -0.218391 }, 
    { -0.270115, -0.229885, 0 }, { -0.270115, -0.229885, -0.128736 }, { -0.168966, -0.229885, -0.229885 }, { -0.0402299, -0.229885, -0.229885 }, 
    { -0.270115, -0.218391, 0 }, { -0.270115, -0.218391, -0.128736 }, { -0.168966, -0.218391, -0.229885 }, { -0.0402299, -0.218391, -0.229885 }
  },
  {
    { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, { -0.0402299, -0.241379, 0 }, 
    { -0.0402299, -0.241379, -0.218391 }, { 0.082069, -0.241379, -0.218391 }, { 0.178161, -0.241379, -0.122299 }, { 0.178161, -0.241379, 0 }, 
    { -0.0402299, -0.229885, -0.229885 }, { 0.0885057, -0.229885, -0.229885 }, { 0.189655, -0.229885, -0.128736 }, { 0.189655, -0.229885, 0 }, 
    { -0.0402299, -0.218391, -0.229885 }, { 0.0885057, -0.218391, -0.229885 }, { 0.189655, -0.218391, -0.128736 }, { 0.189655, -0.218391, 0 }
  }
};

const MAV_matrix mavlib_bezierMatrix= {{{-1,3,-3,1},{3,-6,3,0},{-3,3,0,0},{1,0,0,0}}};

void mavlib_4x4vectorMult(float result[4], MAV_matrix matrix, float v[4])
{
  result[0] = v[0]*matrix.mat[0][0] + v[1]*matrix.mat[0][1] + v[2]*matrix.mat[0][2] + v[3]*matrix.mat[0][3];
  result[1] = v[0]*matrix.mat[1][0] + v[1]*matrix.mat[1][1] + v[2]*matrix.mat[1][2] + v[3]*matrix.mat[1][3];
  result[2] = v[0]*matrix.mat[2][0] + v[1]*matrix.mat[2][1] + v[2]*matrix.mat[2][2] + v[3]*matrix.mat[2][3];
  result[3] = v[0]*matrix.mat[3][0] + v[1]*matrix.mat[3][1] + v[2]*matrix.mat[3][2] + v[3]*matrix.mat[3][3];
}

void mav_teapotDrawPatch(MAV_vector *P, int subdivisions, MAV_vector *vertices, MAV_vector *normals, MAV_texCoord *texs, MAV_surfaceParams *sp, float sc)
{
  float s,t,inc;
  float eps;
  float v[4], tmp[4];
  MAV_matrix Gx,Gy,Gz;
  MAV_vector r,ds,dt;
  int i,j;

  inc= 1.0/subdivisions;

  Gx.mat[0][0]= P[0].x;  Gx.mat[0][1]= P[1].x;  Gx.mat[0][2]= P[2].x;  Gx.mat[0][3]= P[3].x;
  Gx.mat[1][0]= P[4].x;  Gx.mat[1][1]= P[5].x;  Gx.mat[1][2]= P[6].x;  Gx.mat[1][3]= P[7].x;
  Gx.mat[2][0]= P[8].x;  Gx.mat[2][1]= P[9].x;  Gx.mat[2][2]= P[10].x; Gx.mat[2][3]= P[11].x;
  Gx.mat[3][0]= P[12].x; Gx.mat[3][1]= P[13].x; Gx.mat[3][2]= P[14].x; Gx.mat[3][3]= P[15].x;

  Gy.mat[0][0]= P[0].y;  Gy.mat[0][1]= P[1].y;  Gy.mat[0][2]= P[2].y;  Gy.mat[0][3]= P[3].y;
  Gy.mat[1][0]= P[4].y;  Gy.mat[1][1]= P[5].y;  Gy.mat[1][2]= P[6].y;  Gy.mat[1][3]= P[7].y;
  Gy.mat[2][0]= P[8].y;  Gy.mat[2][1]= P[9].y;  Gy.mat[2][2]= P[10].y; Gy.mat[2][3]= P[11].y;
  Gy.mat[3][0]= P[12].y; Gy.mat[3][1]= P[13].y; Gy.mat[3][2]= P[14].y; Gy.mat[3][3]= P[15].y;

  Gz.mat[0][0]= P[0].z;  Gz.mat[0][1]= P[1].z;  Gz.mat[0][2]= P[2].z;  Gz.mat[0][3]= P[3].z;
  Gz.mat[1][0]= P[4].z;  Gz.mat[1][1]= P[5].z;  Gz.mat[1][2]= P[6].z;  Gz.mat[1][3]= P[7].z;
  Gz.mat[2][0]= P[8].z;  Gz.mat[2][1]= P[9].z;  Gz.mat[2][2]= P[10].z; Gz.mat[2][3]= P[11].z;
  Gz.mat[3][0]= P[12].z; Gz.mat[3][1]= P[13].z; Gz.mat[3][2]= P[14].z; Gz.mat[3][3]= P[15].z;

  Gx= mav_matrixMult(Gx,mavlib_bezierMatrix);
  Gx= mav_matrixMult(mavlib_bezierMatrix,Gx);

  Gy= mav_matrixMult(Gy,mavlib_bezierMatrix);
  Gy= mav_matrixMult(mavlib_bezierMatrix,Gy);

  Gz= mav_matrixMult(Gz,mavlib_bezierMatrix);
  Gz= mav_matrixMult(mavlib_bezierMatrix,Gz);

  i= 0;
  for (t= 0.0; t<= 1.0+inc/2.0; t+= inc) {
    for (s= 0.0; s<= 1.0+inc/2.0; s+= inc)
      {
	/* calculate vertex position */
	v[0]= t*t*t; v[1]= t*t; v[2]= t; v[3]= 1.0;
	mavlib_4x4vectorMult(tmp,Gx,v);
	r.x= s*s*s*tmp[0]+s*s*tmp[1]+s*tmp[2]+tmp[3];	
	mavlib_4x4vectorMult(tmp,Gy,v);
	r.y= s*s*s*tmp[0]+s*s*tmp[1]+s*tmp[2]+tmp[3];	
	mavlib_4x4vectorMult(tmp,Gz,v);
	r.z= s*s*s*tmp[0]+s*s*tmp[1]+s*tmp[2]+tmp[3];	

	/* calculate ds normal */
	if (t < MAV_EPSILON)
	  eps= MAV_EPSILON;
	else if (t > 1.0-MAV_EPSILON)
	  eps= 1.0-MAV_EPSILON;
	else
	  eps= t;
	v[0]= eps*eps*eps; v[1]= eps*eps; v[2]= eps; v[3]= 1.0;
	mavlib_4x4vectorMult(tmp,Gx,v);
	ds.x= 3.0*s*s*tmp[0]+2.0*s*tmp[1]+tmp[2];
	mavlib_4x4vectorMult(tmp,Gy,v);
	ds.y= 3.0*s*s*tmp[0]+2.0*s*tmp[1]+tmp[2];
	mavlib_4x4vectorMult(tmp,Gz,v);
	ds.z= 3.0*s*s*tmp[0]+2.0*s*tmp[1]+tmp[2];

	/* calculate dt normal */
	if (s < MAV_EPSILON)
	  eps= MAV_EPSILON;
	else if (s > 1.0-MAV_EPSILON)
	  eps= 1.0-MAV_EPSILON;
	else
	  eps= s;
	v[0]= 3.0*t*t; v[1]= 2.0*t; v[2]= 1.0; v[3]= 0.0;
	mavlib_4x4vectorMult(tmp,Gx,v);
	dt.x= eps*eps*eps*tmp[0]+eps*eps*tmp[1]+eps*tmp[2]+tmp[3];	
	mavlib_4x4vectorMult(tmp,Gy,v);
	dt.y= eps*eps*eps*tmp[0]+eps*eps*tmp[1]+eps*tmp[2]+tmp[3];	
	mavlib_4x4vectorMult(tmp,Gz,v);
	dt.z= eps*eps*eps*tmp[0]+eps*eps*tmp[1]+eps*tmp[2]+tmp[3];	

	normals[i]= mav_vectorNormalise(mav_vectorCrossProduct(ds,dt));
	texs[i].s=s;
	texs[i].t=t;
	vertices[i++]= mav_vectorScalar(r, sc);
      }  
  }

  /* draw patch */
  for (j= 0; j< subdivisions; j++) {
    for (i= 0; i< subdivisions; i++)
      {
	mav_gfxPolygonBegin();

	if (sp->mode==MAV_MATERIAL || sp->mode>=MAV_LIT_TEXTURE) mav_gfxNormal(normals[j*(subdivisions+1)+i]);
	if (sp->mode>=MAV_TEXTURE) mav_gfxTexCoord(texs[j*(subdivisions+1)+i]);
	mav_gfxVertex(vertices[j*(subdivisions+1)+i]);

	if (sp->mode==MAV_MATERIAL || sp->mode>=MAV_LIT_TEXTURE) mav_gfxNormal(normals[(j+1)*(subdivisions+1)+i]);
	if (sp->mode>=MAV_TEXTURE) mav_gfxTexCoord(texs[(j+1)*(subdivisions+1)+i]);
	mav_gfxVertex(vertices[(j+1)*(subdivisions+1)+i]);

	if (sp->mode==MAV_MATERIAL || sp->mode>=MAV_LIT_TEXTURE) mav_gfxNormal(normals[(j+1)*(subdivisions+1)+i+1]);
	if (sp->mode>=MAV_TEXTURE) mav_gfxTexCoord(texs[(j+1)*(subdivisions+1)+i+1]);
	mav_gfxVertex(vertices[(j+1)*(subdivisions+1)+i+1]);

	if (sp->mode==MAV_MATERIAL || sp->mode>=MAV_LIT_TEXTURE) mav_gfxNormal(normals[j*(subdivisions+1)+i+1]);
	if (sp->mode>=MAV_TEXTURE) mav_gfxTexCoord(texs[j*(subdivisions+1)+i+1]);
	mav_gfxVertex(vertices[j*(subdivisions+1)+i+1]);

	mav_gfxPolygonEnd();
      }
  }
}



/* Routine to render a teapot */

int mav_teapotDraw(MAV_object *obj, MAV_drawInfo *di)
{
  MAV_teapot *tpt= (MAV_teapot *) mav_objectDataGet(obj);
  MAV_vector *vertices;
  MAV_vector *normals;
  MAV_texCoord *texs;
  int i;
  int subdivisions;

  if ((mav_opt_curveLOD || tpt->subdivisions<1) && di)
  {
    float dist;
    MAV_vector centre;

    centre.x= centre.y= centre.z= 0.0;
    centre= mav_vectorMult (centre, tpt->matrix);

    centre= mav_vectorSub(centre,di->vp.eye);
    dist= mav_vectorDotProduct(centre,centre);
    dist/= (tpt->size*tpt->size);

    if (dist > 200.0) subdivisions= 1;
    else if (dist > 50.0) subdivisions= 2;
    else if (dist > 10.0) subdivisions= 3;
    else if (dist > 2.0) subdivisions= 4;
    else if (dist > 1.0) subdivisions= 5;
    else if (dist > 0.5) subdivisions= 6;
    else if (dist > 0.3) subdivisions= 7;
    else if (dist > 0.1) subdivisions= 8;
    else subdivisions= 9;
  }
  else
    subdivisions= tpt->subdivisions;

/* Malloc off memory to store the vertices */

  vertices= (MAV_vector *) mav_malloc((subdivisions+1)*(subdivisions+1)*sizeof(MAV_vector));
  normals= (MAV_vector *) mav_malloc((subdivisions+1)*(subdivisions+1)*sizeof(MAV_vector));
  texs= (MAV_texCoord *) mav_malloc((subdivisions+1)*(subdivisions+1)*sizeof(MAV_texCoord));

/* Set the correct colouring */

  mav_surfaceParamsUse(tpt->sp);

/* Store the current transformation matrix - then multiply it by the local transformation */

  mav_gfxMatrixPush();
  mav_gfxMatrixMult(tpt->matrix);

/* Render the patches */

  for (i= 0; i< 32; i++)
    mav_teapotDrawPatch(mavlib_teapotPatches[i], subdivisions, vertices, normals, texs, tpt->sp, tpt->size);

/* Restore original transformation matrix */

  mav_gfxMatrixPop();

  mav_free(vertices);
  mav_free(normals);  
  mav_free(texs);

  return 1;
}



/* Routine to calculate the bounding box of a teapot */

int mav_teapotBB(MAV_object *obj, MAV_BB *bb)
{
  MAV_teapot *tpt= (MAV_teapot *) mav_objectDataGet(obj);
  MAV_BB tmp;

  tmp.min.x= -0.5;
  tmp.min.y= -0.24137931;
  tmp.min.z= -0.30651341;

  tmp.max.x= 0.5;
  tmp.max.y= 0.24137931;
  tmp.max.z= 0.30651341;

  tmp.min= mav_vectorScalar(tmp.min, tpt->size);
  tmp.max= mav_vectorScalar(tmp.max, tpt->size);

  mav_BBAlign(tmp,tpt->matrix,bb);

  return 1;
}



/* Routine to identify a teapot */

int mav_teapotID(MAV_object *o, char **id)
{
   *id="teapot";
   return 1;
}



/* Routine to return the userdef field of a teapot */

int mav_teapotGetUserdef(MAV_object *o, void ***ud)
{
  MAV_teapot *tpt= (MAV_teapot *) mav_objectDataGet(o);

  *ud= &tpt->userdef;

  return 1;
}



/* Routine to return the matrix field of a teapot */

int mav_teapotGetMatrix(MAV_object *o, MAV_matrix **m)
{
  MAV_teapot *tpt= (MAV_teapot *) mav_objectDataGet(o);

  *m= &tpt->matrix;

  return 1;
}



/* Routine to return the surface params field of a teapot */

int mav_teapotGetSurfaceParams(MAV_object *o, MAV_surfaceParams ***sp)
{
  MAV_teapot *tpt= (MAV_teapot *) mav_objectDataGet(o);

  *sp= &tpt->sp;

  return 1;
}



/* Routine to dump a teapot */

int mav_teapotDump(MAV_object *o)
{
  MAV_teapot *tpt = (MAV_teapot *) mav_objectDataGet(o);

  printf("*** Dumping object %p - a MAV_teapot with data pointer %p\n", o, mav_objectDataGet(o));
  mav_surfaceParamsPrint("surface params ", *tpt->sp);
  mav_matrixPrint("matrix\n", tpt->matrix);
  printf("userdef %p\n", tpt->userdef);

  return 1;
}
