/*
 * feedsettingswidget.cpp
 *
 * Copyright (c) 2007 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include "feedsettingswidget.h"
#include "newsfeedmanager.h"
#include "settings.h"

#include <KInputDialog>
#include <KProgressDialog>

FeedSettingsWidget::FeedSettingsWidget( QWidget *parent )
    : QWidget( parent ),
    m_downloadMessageBox( 0 )
{
    ui.setupUi( this );
    ui.feedListWidget->addItems( Settings::feedUrls() );
    connect( ui.feedListWidget, SIGNAL( itemSelectionChanged() ),
             this, SLOT( feedItemChanged() ) );
    connect( ui.addButton, SIGNAL( clicked() ),
             this, SLOT( addButtonClicked() ) );
    connect( ui.removeButton, SIGNAL( clicked() ),
             this, SLOT( removeButtonClicked() ) );

    if ( ui.feedListWidget->count() > 0 ) {
        ui.feedListWidget->setCurrentRow( 0 );
        feedItemChanged();
    }

    ui.addButton->setIcon( KIcon( "list-add" ) );
    ui.removeButton->setIcon( KIcon( "list-remove" ) );
}

QStringList FeedSettingsWidget::feedUrls() const
{
    QStringList urls;
    for ( int i = 0; i < ui.feedListWidget->count(); ++i ) {
        urls.append( ui.feedListWidget->item( i )->text() );
    }
    return urls;
}

void FeedSettingsWidget::feedItemChanged()
{
    QListWidgetItem *item = ui.feedListWidget->currentItem();
    ui.removeButton->setEnabled( item != 0 );
    if ( item == 0 ) {
        return;
    }

    QMap<QUrl, Syndication::FeedPtr> availableFeeds = NewsFeedManager::self()->availableFeeds();
    QMap<QUrl, Syndication::FeedPtr>::ConstIterator it = availableFeeds.find( item->text() );
    if ( it == availableFeeds.end() ) {
        kDebug( 500 ) << "Don't have this item " << item->text();
        return;
    }

    Syndication::FeedPtr feed = *it;
    ui.feedTitleLabel->setText( feed->title() );
    ui.feedUrlLabel->setText( feed->link() );
    ui.feedDescriptionLabel->setText( feed->description() );

}

void FeedSettingsWidget::addButtonClicked()
{
     bool ok;
     QString url = KInputDialog::getText( i18n( "New Newsfeed" ),
                                          i18n( "Enter the Address (URL) of the Newsfeed to be added:" ),
                                          QString(),
                                          &ok,
                                          this );

     if ( ok && !url.isEmpty() ) {
         m_addedFeedUrl = url;
         connect( NewsFeedManager::self(), SIGNAL( feedLoaded( const QUrl & ) ),
                  this, SLOT( feedLoaded( const QUrl & ) ) );

         NewsFeedManager::self()->updateFeed( url );
         m_downloadMessageBox = new KProgressDialog( this,
                                                     i18n( "Please wait..." ),
                                                     i18n( "Please wait while the newsfeed is downloaded..." ) );
         m_downloadMessageBox->progressBar()->setRange( 0, 0 );
         m_downloadMessageBox->exec();
     }
}

void FeedSettingsWidget::removeButtonClicked()
{
    int row = ui.feedListWidget->currentRow();
    delete ui.feedListWidget->takeItem( row );

    const int remainingItems = ui.feedListWidget->count();
    if ( remainingItems > 0 ) {
        if ( row == remainingItems ) {
            row = remainingItems - 1;
        }
        ui.feedListWidget->setCurrentRow( row );
    }
}


void FeedSettingsWidget::feedLoaded( const QUrl &url )
{
    if ( url.toString() != m_addedFeedUrl ) {
        return;
    }

    disconnect( NewsFeedManager::self(), SIGNAL( feedLoaded( const QUrl & ) ),
                this, SLOT( feedLoaded( const QUrl & ) ) );

    delete m_downloadMessageBox;
    m_downloadMessageBox = 0;

    QListWidgetItem *item = new QListWidgetItem( url.toString() );
    ui.feedListWidget->addItem( item );
    ui.feedListWidget->setCurrentItem( item );
}

#include "feedsettingswidget.moc"

