//
// This file is part of the Marble Virtual Globe.
//
// This program is free software licensed under the GNU LGPL. You can
// find a copy of this license in LICENSE.txt in the top directory of
// the source code.
//
// Copyright 2006-2010 Torsten Rahn <tackat@kde.org>
// Copyright 2007      Inge Wallin  <ingwa@kde.org>
//


#ifndef MARBLE_MAINWINDOW_H
#define MARBLE_MAINWINDOW_H


#include <QtGui/QMainWindow>
#include <QtCore/QDateTime>
#include <QtCore/QVariantMap>
#include "ControlView.h"

class QAction;
class QLabel;
class QMenu;
class QPrinter;
class QProgressBar;

namespace Marble
{

class MarbleWidget;
class SunControlWidget;
class TimeControlWidget;
class QtMarbleConfigDialog;
class DownloadRegionDialog;
class RoutingWidget;
class StackableWindow;
class GoToDialog;

class MainWindow : public QMainWindow
{
    Q_OBJECT

#ifdef Q_WS_MAEMO_5
    enum Orientation {
        OrientationAutorotate,
        OrientationLandscape,
        OrientationPortrait
    };
#endif

public:
    explicit MainWindow(const QString& marbleDataPath = QString(),
                        const QVariantMap& cmdLineSettings = QVariantMap(),
                        QWidget *parent=0);

    ControlView* marbleControl() {
        return m_controlView;
    }
    MarbleWidget* marbleWidget() {
        return m_controlView->marbleWidget();
    }

    void addGeoDataFile( const QString &fileName );

#ifdef Q_WS_MAEMO_5
    Orientation orientation() const;
#endif

protected:
    void  closeEvent(QCloseEvent *event);

private:
    void  createToolBar();
    void  createActions();
    void  createMenus( const QList<QAction*> &panelActions );
    void  createStatusBar();

    QString  readMarbleDataPath();
    void  readSettings(const QVariantMap& overrideSettings = QVariantMap());
    void  writeSettings();

private Q_SLOTS:
    void  showPosition( const QString& position);
    void  showDistance( const QString& position);
    void  showZoom( int );
    void  showDateTime();

    void  initObject(const QVariantMap& cmdLineSettings);
    void  editSettings();
    void  updateSettings();
    
    void  fallBackToDefaultTheme();

    // File Menu
    void  openFile();
    void  openMapDialog();
    void  exportMapScreenShot();
    void  showDownloadRegionDialog();
    void  printMapScreenShot();
    void  workOffline( bool );
    void  showMapWizard();

    // Edit Menu
    void  copyMap();
    void  copyCoordinates();
    void  updateMapEditButtonVisibility( const QString &mapTheme );

    // View Menu
    void  lockPosition( bool );
    void  createPluginsMenus();
    void  createPluginMenus();
    void  showClouds( bool );
    void  controlSun();
    void  controlTime();
    void  showSun( bool );
    void  reload();

    // Settings Menu
    void  showFullScreen( bool );
#ifdef Q_WS_MAEMO_5
    void  setOrientation( Orientation orientation );
#endif
    void  showStatusBar( bool );
    void  setupStatusBar();
    void  setupDownloadProgressBar();

    // Help Menu
    void  enterWhatsThis();
    void  aboutMarble();
    void  handbook();

    //Bookmark Menu
    void  openEditBookmarkDialog();
    void  setHome();
    void  createBookmarksListMenu( QMenu *bookmarksListMenu, const GeoDataContainer *container );
    void  lookAtBookmark( QAction * action );
    void  manageBookmarks();
    void  createBookmarkMenu();
    void  createFolderList( QMenu *bookmarksListMenu, const GeoDataContainer *container );
    void  showBookmarks( bool show );

    // Download region dialog
    void  connectDownloadRegionDialog();
    void  disconnectDownloadRegionDialog();
    void  downloadRegion();

    // Small screen devices specific slots
    void showMapViewDialog();
    void showRoutingDialog();
    void showTrackingDialog();
    void showGoToDialog();

    void showZoomLevel( bool show );

    void downloadJobAdded();
    void downloadJobRemoved();

private:
    void setupZoomButtons();
    void initializeTrackingWidget();

    ControlView *m_controlView;
    SunControlWidget* m_sunControlDialog;
    TimeControlWidget* m_timeControlDialog;
    QtMarbleConfigDialog *m_configDialog;
    DownloadRegionDialog* m_downloadRegionDialog;

    /// Store plugin toolbar pointers so that they can be removed/updated later
    QList<QToolBar*> m_pluginToolbars;
    /// Store plugin menus so that they can be removed/updated later
    QList<QMenu*> m_pluginMenus;

    QMenu *m_fileMenu;
    QMenu *m_viewMenu;
    QMenu *m_helpMenu;
    QMenu *m_settingsMenu;
    QMenu *m_panelMenu;
    QMenu *m_infoBoxesMenu;
    QMenu *m_onlineServicesMenu;
    QMenu *m_bookmarkMenu;
    // File Menu
    QAction *m_openAct;
    QAction *m_exportMapAct;
    QAction *m_downloadAct;
    QAction *m_downloadRegionAction;
    QAction *m_printPreviewAct;
    QAction *m_printAct;
    QAction *m_workOfflineAct;
    QAction *m_quitAct;
    QAction *m_mapWizardAct;

    // Edit Menu
    QAction *m_copyMapAct;
    QAction *m_copyCoordinatesAct;
    QAction *m_osmEditAction;

    // View Menu
    QAction *m_showCloudsAct;
    QAction *m_controlSunAct;
    QAction *m_controlTimeAct;
    QAction *m_reloadAct;

    // Settings Menu
    QAction *m_fullScreenAct;
    QAction *m_statusBarAct;
    QAction *m_configDialogAct;

    // Help Menu
    QAction *m_whatsThisAct;
    QAction *m_aboutMarbleAct;
    QAction *m_aboutQtAct;
    QAction *m_lockFloatItemsAct;
    QAction *m_handbookAct;

    // Status Bar
    QString     m_position;
    QString     m_distance;
    QString     m_zoom;
    QString     m_clock;
    QLabel      *m_positionLabel;
    QLabel      *m_distanceLabel;
    QLabel      *m_zoomLabel;
    QLabel      *m_clockLabel;
    QProgressBar *m_downloadProgressBar;
    void updateStatusBar();

    //Bookmark Menu
    QAction *m_addBookmarkAct;
    QAction *m_setHomeAct;
    QAction *m_toggleBookmarkDisplayAct;
    QAction *m_manageBookmarksAct;

    // Small screen devices
    QAction *m_showMapViewDialogAction;
    QAction *m_toggleRoutingTabAction;
    QAction *m_showTrackingDialogAction;

    QDialog *m_mapViewWindow;
    StackableWindow *m_routingWindow;
    StackableWindow *m_trackingWindow;
    GoToDialog *m_gotoDialog;

    RoutingWidget *m_routingWidget;

    QString m_lastFileOpenPath;
    QStringList m_commandlineFilePaths;
};

} // namespace Marble

#endif
