/*
 * disks.h
 *
 * Copyright (c) 1998 Michael Kropfberger <michael.kropfberger@gmx.net>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef __DISKS_H__
#define __DISKS_H__

#include <QtCore/QObject>

#include <QtGui/QLabel>
#include <QtGui/QPushButton>
#include <QtGui/QProgressBar>
#include <QtCore/QFile>

#include <kio/global.h>
#include <klocale.h>

class K3Process;
class K3ShellProcess;

class DiskEntry : public QObject
{
  Q_OBJECT
public:
  explicit DiskEntry(QObject *parent=0, const char *name=0);
  explicit DiskEntry(const QString & deviceName, QObject *parent=0, const char *name=0);
  ~DiskEntry();
  QString lastSysError() {return sysStringErrOut; }
  QString deviceName() const { return device; }
  // The real device (in case deviceName() is a symlink)
  QString deviceRealName() const;
  QString mountPoint() const { return mountedOn; }
  QString mountOptions() const { return options; }
  // The real device (in case deviceName() is a symlink)
  QString realMountPoint() const;
  /**
   * sets the used mountCommand for the actual DiskEntry.
   * @param mntcmd   is a string containing the executable file and
   *                 special codes which will be filled in when used: <BR>
   *                 %m : mountpoint <BR>
   *                 %d : deviceName <BR>
   *                 %t : filesystem type <BR>
   *                 %o : mount options <BR>
   *                 all this information is gained from the objects' data
   *                 if no mountCommand is set it defaults to "mount %d"
   **/
  QString mountCommand() const { return mntcmd; }
  /**
   * sets the used umountCommand for the actual DiskEntry.
   * @param mntcmd   is a string containing the executable file and
   *                 special codes which will be filled in when used: <BR>
   *                 %m : mountpoint <BR>
   *                 %d : deviceName <BR>
   *                 all this information is gained from the objects' data
   *                 if no umountCommand is set it defaults to "umount %d"
   **/
  QString umountCommand() const { return umntcmd; }
  QString fsType() const { return type; }
  bool mounted() const { return isMounted; }
  qulonglong kBSize() const { return size; }
  QString iconName();
  QString realIconName() { return icoName; }
  QString prettyKBSize() const { return KIO::convertSizeFromKiB(size); }
  qulonglong kBUsed() const { return used; }
  QString prettyKBUsed() const { return KIO::convertSizeFromKiB(used); }
  qulonglong kBAvail() const  { return avail; }
  QString prettyKBAvail() const { return KIO::convertSizeFromKiB(avail); }
  float percentFull() const;

signals:
  void sysCallError(DiskEntry *disk, int err_no);
  void deviceNameChanged();
  void mountPointChanged();
  void mountOptionsChanged();
  void fsTypeChanged();
  void mountedChanged();
  void kBSizeChanged();
  void kBUsedChanged();
  void kBAvailChanged();
  void iconNameChanged();

public slots:

  int toggleMount();
  int mount();
  int umount();
  int remount();
  void setMountCommand(const QString & mnt);
  void setUmountCommand(const QString & umnt);
  void setDeviceName(const QString & deviceName);
  void setMountPoint(const QString & mountPoint);
  void setIconName(const QString & iconName);
  void setMountOptions(const QString & mountOptions);
  void setFsType(const QString & fsType);
  void setMounted(bool nowMounted);
  void setKBSize(qulonglong kb_size);
  void setKBUsed(qulonglong kb_used);
  void setKBAvail(qulonglong kb_avail);
  QString guessIconName();

private slots:
   void receivedSysStdErrOut(K3Process *, char *data, int len);

private:
  void init(const char *name);
  int sysCall(const QString & command);
  QString prettyPrint(int kBValue) const;

  K3ShellProcess     *sysProc;
  QString           sysStringErrOut;
  bool              readingSysStdErrOut;

  QString     device,
              type,
              mountedOn,
              options,
              icoName,
              mntcmd,
              umntcmd;

  qulonglong  size,
              used,
              avail;       // ATTENTION: used+avail != size (clustersize!)

  bool        isMounted,
              iconSetByUser;
};

#endif
