/*
  Copyright (c) 2012-2015 Montel Laurent <montel@kde.org>

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "addtagdialog.h"
#include "mailcommon/tag/tagwidget.h"
#include "mailcommon_debug.h"
#include <AkonadiCore/tagcreatejob.h>

#include <KLocalizedString>
#include <KLineEdit>
#include <KMessageBox>
#include <KJob>

#include <QVBoxLayout>
#include <KConfigGroup>
#include <QDialogButtonBox>
#include <QPushButton>

using namespace MailCommon;

AddTagDialog::AddTagDialog(const QList<KActionCollection *> &actions, QWidget *parent)
    : QDialog(parent)
{
    setModal(true);
    setWindowTitle(i18n("Add Tag"));
    QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);
    QVBoxLayout *mainLayout = new QVBoxLayout;
    setLayout(mainLayout);
    mOkButton = buttonBox->button(QDialogButtonBox::Ok);
    mOkButton->setDefault(true);
    mOkButton->setShortcut(Qt::CTRL | Qt::Key_Return);
    connect(buttonBox, &QDialogButtonBox::accepted, this, &AddTagDialog::slotSave);
    connect(buttonBox, &QDialogButtonBox::rejected, this, &AddTagDialog::reject);
    mOkButton->setDefault(true);

    mTagWidget = new MailCommon::TagWidget(actions, this);

    connect(mTagWidget->tagNameLineEdit(), &KLineEdit::textChanged, this, &AddTagDialog::slotTagNameChanged);
    mOkButton->setEnabled(false);
    mainLayout->addWidget(mTagWidget);
    mainLayout->addWidget(buttonBox);
}

AddTagDialog::~AddTagDialog()
{
}

void AddTagDialog::setTags(const QList<MailCommon::Tag::Ptr> &tags)
{
    mTags = tags;
}

void AddTagDialog::slotTagNameChanged(const QString &text)
{
    mOkButton->setEnabled(!text.trimmed().isEmpty());
}

void AddTagDialog::slotSave()
{
    const QString name(mTagWidget->tagNameLineEdit()->text());

    Q_FOREACH (const MailCommon::Tag::Ptr &tag, mTags) {
        if (tag->name() == name) {
            KMessageBox::error(this, i18n("Tag %1 already exists", name));
            mTagWidget->tagNameLineEdit()->setFocus();
            mTagWidget->tagNameLineEdit()->selectAll();
            return;
        }
    }

    MailCommon::Tag::Ptr tag(Tag::createDefaultTag(name));
    mTagWidget->recordTagSettings(tag);
    MailCommon::Tag::SaveFlags saveFlags = mTagWidget->saveFlags();
    const Akonadi::Tag akonadiTag = tag->saveToAkonadi(saveFlags);
    Akonadi::TagCreateJob *createJob = new Akonadi::TagCreateJob(akonadiTag, this);
    connect(createJob, &Akonadi::TagCreateJob::result, this, &AddTagDialog::onTagCreated);

    mLabel = name;
}

void AddTagDialog::onTagCreated(KJob *job)
{
    if (job->error()) {
        qCWarning(MAILCOMMON_LOG) << "Failed to create tag: " << job->errorString();
        reject();
        return;
    }
    Akonadi::TagCreateJob *createJob = static_cast<Akonadi::TagCreateJob *>(job);
    mTag = createJob->tag();
    accept();
}

QString AddTagDialog::label() const
{
    return mLabel;
}

Akonadi::Tag AddTagDialog::tag() const
{
    return mTag;
}

