/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <kwm.h>
#include <qobject.h>
#include "khdisplaywidget.h"
#include "khchip.h"
#include "khdockwidget.h"
#include "sensors.h"
#include "sensorid.h"
#include <stdio.h>
#include <stdlib.h>
#include "khglobals.h"
#include "khealthcare.h"
#include "khlmprofile.h"
#include "khadm9240.h"
#include "kh_gl5xx_profile.h"

KHealthCare::KHealthCare( int argc, char* argv[] ) : KApplication( argc, argv ) {

	FILE *file = 0;
	file = fopen( "/etc/sensors.conf", "r" );
	if ( sensors_init( file ) < 0 ) abort( i18n( 	"Couldn't init sensor !\n\n"
																							 	"Please make sure that the lm_sensor modules\n"
																								"are loaded and working properly and the\n"
																								"configuration file is located in \"/etc/sensors.conf\"\n\n"
																								"You do have a hardware monitoring chip, don't you ?" ) );
	else chipname = detectChip( !( this->isRestored() ) );

	if ( !( profile = createProfile( chipname ) ) )
			abort( i18n( "Sorry, your chip is not yet supported\n"
									 "in this version of KHealthCare.\n\n"
									 "Please contact the author !." ) );

	/* check if a configuration entry is existing
		 if not existing, assume missing config file and
		 create default entries for all
	 */	
	conf = getConfig();
	conf->setGroup( "Monitoring" );	
	if ( !( conf->hasKey( "Fan1Enabled" ) ) ) {

		conf->setGroup( "General" );
		conf->writeEntry( GeneralKeys[0], 15 );
		conf->writeEntry( GeneralKeys[1], false );
		conf->writeEntry( GeneralKeys[2], 60 );
		conf->writeEntry( GeneralKeys[3], 20 );
		conf->writeEntry( GeneralKeys[4], true );
		conf->writeEntry( GeneralKeys[5], 4 );

		conf->setGroup  ( "Monitoring" );
		conf->writeEntry( EnableKeys[Fan1], false );
		conf->writeEntry( EnableKeys[Fan2], false );
		conf->writeEntry( EnableKeys[Fan3], true );
		conf->writeEntry( EnableKeys[Temp1], true );
		conf->writeEntry( EnableKeys[Temp2], false );
		conf->writeEntry( EnableKeys[Temp3], false );
		conf->writeEntry( EnableKeys[VCore], true );
		conf->writeEntry( EnableKeys[VCore2], true );
		conf->writeEntry( EnableKeys[V33], true );
		conf->writeEntry( EnableKeys[V5], true );
		conf->writeEntry( EnableKeys[Vm5], true );
		conf->writeEntry( EnableKeys[V12], true );
		conf->writeEntry( EnableKeys[Vm12], true );
		conf->writeEntry( EnableKeys[VID], true );
		
		conf->setGroup  ( "NumFaults" );
		for ( SensorId i = Fan1; i < VID; i++ ) conf->writeEntry( FaultsKeys[i], 3 );
		
		conf->setGroup	( "Beeping" );
		conf->writeEntry( BeepKeys[Fan1], false );
		conf->writeEntry( BeepKeys[Fan2], false );
		conf->writeEntry( BeepKeys[Fan3], true );
		conf->writeEntry( BeepKeys[Temp1], true );
		conf->writeEntry( BeepKeys[Temp2], false );
		conf->writeEntry( BeepKeys[Temp3], false );
		conf->writeEntry( BeepKeys[VCore], true );
		conf->writeEntry( BeepKeys[VCore2], true );
		conf->writeEntry( BeepKeys[V33], true );
		conf->writeEntry( BeepKeys[V5], true );
		conf->writeEntry( BeepKeys[Vm5], true );
		conf->writeEntry( BeepKeys[V12], true );
		conf->writeEntry( BeepKeys[Vm12], true );
	

		conf->setGroup	( "Alarms" );
		conf->writeEntry( "Fan1Alarm", 2700 );
		conf->writeEntry( "Fan2Alarm", 2700 );
		conf->writeEntry( "Fan3Alarm", 2700 );
		conf->writeEntry( "Temp1Alarm", 40 );
		conf->writeEntry( "Temp2Alarm", 50 );
		conf->writeEntry( "Temp3Alarm", 40 );
    conf->writeEntry( "VCoreHiAlarm", 22 );
		conf->writeEntry( "VCore2HiAlarm", 17 );
		conf->writeEntry( "V33HiAlarm", 36 );
		conf->writeEntry( "V5HiAlarm", 55 );
		conf->writeEntry( "Vm5HiAlarm", -45 );
		conf->writeEntry( "V12HiAlarm", 130 );
		conf->writeEntry( "Vm12HiAlarm", -130 );
		conf->writeEntry( "VCoreLoAlarm", 18 );
		conf->writeEntry( "VCore2LoAlarm", 13 );
		conf->writeEntry( "V33LoAlarm", 30 );
		conf->writeEntry( "V5LoAlarm", 45 );
		conf->writeEntry( "Vm5LoAlarm", -55 );
		conf->writeEntry( "V12LoAlarm", 110 );
		conf->writeEntry( "Vm12LoAlarm", -110 );
	}

	ioObj  = new KHChip( chipname, profile );
	gui = new KHDisplayWidget( ioObj, profile );
	setTopWidget( gui ); 	// the gui is always existing, although not always visible
	dockwidget = 0;
	conf->setGroup( "General" );
	setDocking( conf->readBoolEntry( "DockingEnabled", true ) );
	
}

KHealthCare::~KHealthCare(){
	if ( gui ) delete gui;
	if ( ioObj ) delete ioObj;
	if ( dockwidget ) delete dockwidget;
}

/**  */
void KHealthCare::setDocking( int enabled ){

	//debug( "docking %i", enabled );

	if ( !enabled ) {
		setMainWidget( gui );
		if ( dockwidget )	delete dockwidget;
		dockwidget = 0;
		}
	if ( enabled )
		if ( !dockwidget ) {
			//debug( "dockwidget not existing " );
			dockwidget = new KHDockWidget ( gui );
			dockwidget->setAlignment( AlignCenter );
			KWM::setDockWindow( dockwidget->winId() );
			dockwidget->show();
			setMainWidget( dockwidget ); // if dockwidget is existing, make it the main widget
			connect( ioObj, SIGNAL( newDataAcquired	( int, SensorId ) ),
	  		  		 dockwidget, SLOT( display	( int, SensorId ) ) );
/*			connect( ioObj, SIGNAL( newDataAcquired	( float, SensorId ) ),
	  		  		 dockwidget, SLOT( display	( float, SensorId ) ) );
*/
		}
}

void KHealthCare::showGui(){
	if ( gui ) gui->show();
}

void KHealthCare::showDocking(){
	if ( dockwidget ) dockwidget->show();
}

void KHealthCare::processSessionConfig(){
	KConfig *sc = getSessionConfig();
	sc->setGroup( "Default" );
	if ( sc->readBoolEntry( "DisplayWidgetVisible", false ) ) gui->show();
	setDocking( sc->readBoolEntry( "DockingWidgetVisible", false ) );
}

sensors_chip_name KHealthCare::detectChip( bool display_message ){

	int chipNr = 0;
	const sensors_chip_name *chip;
	QString message;

	/* detect hardware monitoring chip ( one should be enough ) */
	if ( ( chip = sensors_get_detected_chips( &chipNr ) ) ) {
		if ( display_message ) {
			message = QString( chip->prefix ).append( i18n( " has been detected.\n" ) );
			message.append( i18n( "Accessing via " ) ).append( sensors_get_adapter_name( chip->bus ) ).append( "," );
			message.append( i18n( "\nusing the " ) ).append( sensors_get_algorithm_name( chip->bus ) ).append( "." );
			QMessageBox::information( NULL, NULL, message );
		}
	}
	else abort ( i18n(  "No hardware monitoring chip\n"
											"has been detected !\n" ) );
	return *chip;
}

/**  */
KHFeatureProfile *KHealthCare::createProfile( sensors_chip_name chipname ){
	if 	( ( QString( chipname.prefix ).contains( "w8378" ) ) ||
				( QString( chipname.prefix ).contains( "lm7" ) )   ||
				( QString( chipname.prefix ).contains( "lm80" ) )  ||
				( QString( chipname.prefix ).contains( "sis5595" ) ) )
		return new KHLMProfile( chipname );
	else if ( ( QString( chipname.prefix ).contains( "adm9240" ) ) ||
				( QString( chipname.prefix ).contains( "ds1780" ) )		   ||
				( QString( chipname.prefix ).contains( "lm81" ) ) )
		return new KH_ADM9240_Profile( chipname );
	else if ( QString( chipname.prefix ).contains( "gl5" ) )
		return new KH_GL5XX_Profile( chipname );
	else return 0;
}














