/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <kapp.h>
#include <kiconloader.h>

#include "conflict.h"
#include "devicelist.h"
#include "proprestab.h"
#include "resentry.h"


extern DeviceList  devicelist;


ResourcePropertiesTab::ResourcePropertiesTab(Device *device, QWidget *parent, 
					     const char *name) : QWidget(parent, name),
					     			 _device(device)
{
  QString   devname   = _device->deviceName();
  QString   classname = _device->deviceClassName();
  Resource  *res;
  
  if (!classname.isEmpty()) devname += " ("+classname+")";
  
  _devtitle  = new DeviceTitle(_device->bigPixmapFile(), devname, this, "devtitle");
    
  _baselabel = new QLabel(i18n("Settings based on:"), this, "baselabel");  
  _baselabel->adjustSize();
  
  _basecombo = new QComboBox(this, "basecombo");
  connect(_basecombo, SIGNAL(activated(int)), SLOT(slotConfigChanged(int)));  
    
  _listframe = new QFrame(this, "listframe");
  _listframe->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  _listframe->setLineWidth(2);

  _listview = new QListView(_listframe, "listview");
  _listview->setAllColumnsShowFocus(true);
  _listview->addColumn(i18n("Resource type"));
  _listview->addColumn(i18n("Setting"));

  _changebut = new QPushButton(i18n("Change &setting..."), this, "changebut");
  _changebut->adjustSize();
  _changebut->setEnabled(false);
  
  _conflictlabel = new QLabel(i18n("Device conflicts:"), this, "conflictlabel");
  _conflictlabel->adjustSize();
  
  _conflictdisplay = new ConflictDisplay(this, "conflictdisplay");
  
  _reslist.clear();
  
  for (res = _device->firstResource(); res; res = _device->nextResource())
    _reslist.append(res);

  fillInConfigurations();
  fillInResources();
}


/*
 * Protected methods
 ********************/
 
void  ResourcePropertiesTab::resizeEvent(QResizeEvent *)
{
  uint  margin = 16;
  uint  w      = width();
  uint  titleh = 50;
  uint  labelw = _baselabel->width();
  uint  labelh = _baselabel->height();
  uint  combow = w-2*margin-labelw-16;
  uint  comboh = labelh+8;
  uint  framew = w-2*margin;
  uint  frameh = 100;
  uint  framey = margin+titleh+2*margin;
  uint  basey  = framey-comboh-10;
  uint  butdy  = 8;
  uint  buty   = framey+frameh+butdy;
  uint  confy  = buty+_changebut->height()+butdy;
  uint  dispy;
  uint  disph;
  
  _devtitle->setGeometry(margin, margin, w-2*margin, titleh);
  
  _baselabel->move(margin, basey+4);
  _basecombo->setGeometry(margin+labelw+16, basey, combow, comboh);
  
  _listframe->setGeometry(margin, framey, framew, frameh);
  _listview->setGeometry(2, 2, framew-4, frameh-4);

  _changebut->move(margin, buty);
  
  _conflictlabel->move(margin, confy);
  
  dispy = confy+_conflictlabel->height()+4;
  disph = height()-dispy-margin;
  
  _conflictdisplay->setGeometry(margin, dispy, framew, disph);
}


/*
 * Private methods
 ******************/
 
void  ResourcePropertiesTab::copyResources(ResourceBaseConfig *baseconf)
{
  Resource         *res;
  QList<Resource>  conflist = baseconf->resourceList();
  
  _reslist.clear();
  
  for (res = conflist.first(); res; res = conflist.next())
    _reslist.append(res);  
}
 
 
void  ResourcePropertiesTab::fillInConfigurations()
{
  QString  basetext, combotext;
  uint     numconf = _device->numConfigs();
  uint     conf;
  int      based   = _device->settingsBasedOn();
  
  debug("Based on configuration %i", based);
  
  _basecombo->clear();
  
  if (!numconf) 
  {
    _baselabel->setEnabled(false);
    _basecombo->setEnabled(false);
  }
  else 
  {
    for (conf = 0; conf < numconf; conf++)
    {
      basetext = i18n("Base configuration");
      combotext.sprintf("%s %04u", basetext.data(), conf);
 
      _basecombo->insertItem(combotext.data());
    }
    
    if (based >= 0) _basecombo->setCurrentItem(based);
  }
} 

 
void  ResourcePropertiesTab::fillInResources()
{
  KIconLoader      *loader   = kapp->getIconLoader();
  QPixmap          respixmap = loader->loadIcon("mini-res.xpm");
  QPixmap          cflpixmap = loader->loadIcon("mini-conflict.xpm");
  QPixmap          xpm;
  Resource         *res;
  ResourceEntry    *entry;
  QList<Conflict>  conflictlist;
  Conflict         *conflict;
  
  _listview->clear();
  
  if (!(conflictlist = devicelist.conflicts(_device, _reslist)).isEmpty())
    _conflictdisplay->showConflicts(conflictlist);
  else _conflictdisplay->reset();
  
  for (res = _reslist.first(); res; res = _reslist.next())
  {  
    entry = new ResourceEntry(_listview, res);
    xpm   = respixmap;
    
    for (conflict = conflictlist.first(); conflict; conflict = conflictlist.next())
    {
      if (*(conflict->resource()) == *res) 
      {
        xpm = cflpixmap;
	break;
      }
    }
    
    entry->setPixmap(0, xpm);
  } 
}


/*
 * Private slots
 ****************/
 
void  ResourcePropertiesTab::slotConfigChanged(int index)
{
  copyResources(_device->baseConfig(index));
  fillInResources();
}

