/*
    kviddata.cpp - holds the main data of the KDE VideoTune

    Requires the Qt widget libraries, available at no cost at
    http://www.troll.no

    Copyright (C) 1998 Frithjof Brestrich
                       brestrich@kde.org


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h> 

#include <kapp.h>
#include <kconfig.h>

#include "xf86configfile.h"
#include "kviddata.h"

KVidData::KVidData()
{
    vidCfg   = NULL;
    xf86cfg  = NULL;
    xf86vid  = NULL;
    doc      = new tDoc;
}

KVidData::~KVidData()
{
    if (vidCfg)  
        delete vidCfg;
    if (xf86cfg) 
        delete xf86cfg;
    if (xf86vid) {
        xf86vid->Disconnect();
        delete xf86vid;
    }
    if ( doc ) {
	delete doc;
    }
}

void KVidData::SetErrorText(const QString &txt)
{
    ErrorText = txt;
}

const QString &KVidData::LastErrorText()
{
    return ErrorText;
}


bool KVidData::loadConfig()
{
    if ( vidCfg == NULL ) 
        vidCfg = new KVidTuneCfg;

    vidCfg->configLoad();

    return TRUE;
}

bool KVidData::storeConfig()
{
    if ( vidCfg != NULL ) 
        vidCfg->configStore();

    return TRUE;
}

bool KVidData::loadXf86Config(const QString &filename)
{
    if ( xf86cfg != NULL ) {
        sectionScreen  = NULL;
	sectionDevice  = NULL;
	sectionMonitor = NULL;
        xf86DriverList.clear();
        delete xf86cfg;
    }

    xf86cfg = new XF86ConfigFile;
    if ( xf86cfg  == NULL ) return FALSE;

    return xf86cfg->Load(filename);
}

bool KVidData::storeXf86Config(const QString &filename)
{
    if ( xf86cfg == NULL ) 
	return FALSE;

    if ( !updateXF86Config() ) 
	return FALSE;

    if ( !xf86cfg->Store(filename,TRUE) ) 
        return FALSE;

    return TRUE;
}

bool KVidData::getXF86DriverList(QStrList &driverlist, QStrList &devicelist, QStrList &monitorlist)
{
    tXF86CfgDirectory *d;
    char              *monVendor, *monModel;

    xf86DriverList.clear();

    driverlist.clear();
    devicelist.clear();
    monitorlist.clear();

    monVendor = xf86vid->MonitorVendor();
    monModel  = xf86vid->MonitorModel();

    if ( !xf86cfg->GetScreenSections(xf86DriverList,monVendor,monModel) ) {
        free(monVendor);
        free(monModel);
        if ( !xf86cfg->GetScreenSections(xf86DriverList) ) {
            SetErrorText( klocale->translate("The configuration file does not contain any valid screen sections.") );
        } else {
            SetErrorText( klocale->translate("The configuration file does not contain at least one screen sections for your monitor.") );
        }
        return FALSE;
    }

    free(monVendor);
    free(monModel);

    for ( d = xf86DriverList.first() ; d != NULL ; d = xf86DriverList.next() ) {
        driverlist.append(xf86cfg->FindEntry(&(d->Data),"Driver")->Value);
        devicelist.append(xf86cfg->FindEntry(&(d->Data),"Device")->Value);
        monitorlist.append(xf86cfg->FindEntry(&(d->Data),"Monitor")->Value);
    }

    return TRUE;
}

bool KVidData::setXF86Driver(unsigned num)
{
    sectionScreen = xf86DriverList.at(num);

    if ( sectionScreen == NULL ) {
        SetErrorText(klocale->translate("requested driver is unknown"));
	return FALSE;
    }

    doc->screen.driver            = xf86cfg->GetValueStr(sectionScreen,"Driver");
    doc->screen.device            = xf86cfg->GetValueStr(sectionScreen,"Device");
    doc->screen.monitor           = xf86cfg->GetValueStr(sectionScreen,"Monitor");
    // TODO: blankTime
    // TODO: suspendTime
    // TODO: offTime
    doc->screen.defaultColorDepth = xf86cfg->GetValueInt(sectionScreen,"DefaultColorDepth");

    if ( ! getXF86ResolutionList(doc->screen.resolutionList) ) 
        return FALSE;

    if ( ! setXF86Card(doc->screen.device) )
        return FALSE;

    if ( ! setXF86Monitor(doc->screen.monitor) )
        return FALSE;

    return TRUE;
}

bool KVidData::getXF86ResolutionList(XF86DisplaySectionList &reslist)
{
  const unsigned transTable[] = { 1, 4, 8, 15, 16, 24, 32, 0 };
  tXF86CfgDirectory  *d;
  tXF86CfgEntry      *e;
  XF86DisplaySection *ds;
  unsigned           i;

  for ( i = 0 ; i < (sizeof(transTable)/sizeof(unsigned)) ; i++ ) {

      ds = new XF86DisplaySection;
      if ( ds == NULL ) {
          SetErrorText(klocale->translate("not enough memory to create data structure"));
	  break;
      } else {
          ds->virtualX = 0;
          ds->virtualY = 0;
          ds->modes.clear();
      }

      ds->depth = transTable[i];
      d = xf86cfg->GetSection(sectionScreen,"Display","Depth",ds->depth);

      if ( d == NULL ) {
	  delete ds;
	  continue;
      }

      e  = xf86cfg->FindEntry(&(d->Data),"Modes");

      if ( e == NULL ) {
	  delete ds;
	  continue;
      }

      // e->Value is something like: '"1280x1024" "1024x768" "800x600" "640x480"'
      QString    s;
      int        a, b, len;
                   
      s.setStr(e->Value);
      a = b = 0;
                  
      while ( ! s.isNull() ) {
          a = s.find('"',b);
          if ( a == -1) break;
          b = s.find('"',a+1);
          if ( b == -1) break;
    
          len = b++ - ++a;
          ds->modes.append(s.mid(a,len));
      }
                  
      e  = xf86cfg->FindEntry(&(d->Data),"Virtual");
      if ( e != NULL ) {
          sscanf(e->Value,"%u %u",&(ds->virtualX),&(ds->virtualY));
      }
    
      reslist.append(ds);
  }

  return TRUE;
}

bool KVidData::setXF86ResolutionList(tXF86CfgDirectory* root,XF86DisplaySectionList &reslist)
{
    XF86DisplaySection *ds;
    tXF86CfgDirectory  *d;
    bool                b;
    QString             s;
    char                *ms;

    b = TRUE;

    for ( ds = reslist.first(); ds != NULL ; ds = reslist.next() ) {
        
	d = xf86cfg->GetSection(sectionScreen,"Display","Depth",ds->depth);

        if ( d == NULL ) {
	    d = new tXF86CfgDirectory("Display");
	    if ( d == NULL ) {
	        SetErrorText(klocale->translate("out of memory"));
		continue;
	    }
	    sectionScreen->Dir.append(d);
        }

        b = b && xf86cfg->SetValueInt(d,"Depth",ds->depth);
	if ( ds->virtualX == 0  || ds->virtualY == 0 ) {
	    b = b && xf86cfg->RemoveEntry(d,"Virtual");
	} else {
	    s.sprintf("%u %u",ds->virtualX,ds->virtualY);
	    b = b && xf86cfg->SetCfgEntry(d,"Virtual",s,FALSE);
	}

        s = "";
	
	for ( ms = ds->modes.first() ; ms != NULL ; ms = ds->modes.next() ) {
	    s = s + "\"" + ms + "\" ";
	}

	b = b && xf86cfg->SetCfgEntry(d,"Modes",s,FALSE);
    }
    
    return TRUE;
}

bool KVidData::setXF86Card(const QString &name)
{
   sectionDevice = xf86cfg->GetDeviceSection(name);

   if ( sectionDevice == NULL ) {
      SetErrorText(klocale->translate("Device section not found"));
      return FALSE;
   }

   doc->device.identifier = xf86cfg->GetValueStr(sectionDevice,"Identifier");
   doc->device.vendorName = xf86cfg->GetValueStr(sectionDevice,"VendorName");
   doc->device.boardName  = xf86cfg->GetValueStr(sectionDevice,"BoardName");
   doc->device.chipset    = xf86cfg->GetValueStr(sectionDevice,"Chipset");
   doc->device.videoRam   = xf86cfg->GetValueInt(sectionDevice,"VideoRam");

   return TRUE;
}

bool KVidData::setXF86Monitor(const QString &name)
{
   sectionMonitor = xf86cfg->GetMonitorSection(name);

   if ( sectionMonitor == NULL ) {
      SetErrorText(klocale->translate("monitor section not found"));
      return FALSE;
   }

   doc->monitor.identifier   = xf86cfg->GetValueStr(sectionMonitor,"Identifier");
   doc->monitor.vendorName   = xf86cfg->GetValueStr(sectionMonitor,"VendorName");
   doc->monitor.modelName    = xf86cfg->GetValueStr(sectionMonitor,"ModelName");
   doc->monitor.horizSync    = xf86cfg->GetValueRangeList(sectionMonitor,"HorizSync",1000);
   doc->monitor.vertRefresh  = xf86cfg->GetValueRangeList(sectionMonitor,"VertRefresh",1);
   doc->monitor.modeLines    = xf86cfg->GetValueModelines(sectionMonitor,"Modeline");

   return TRUE;
}


bool KVidData::syncRunningWithStartup()
{
    if ( xf86vid == NULL ) {
        SetErrorText(klocale->translate("the currently active configuration values are not available"));
        return FALSE;
    }
 
    QList<class  XF86VidModeLine> startup;
    XF86VidModeLine               *running, *ml, *lasthit;
    bool                          *done;
    unsigned                      cnt, i;
    char                          *name;
    unsigned                      hit;
    bool                          b;
    int                           exact;

    if (!xf86vid->GetModeLineList(running,cnt)) {
        SetErrorText(xf86vid->LastErrorText());
        return FALSE;
    }

    startup = doc->monitor.modeLines;
    doc->monitor.modeLines.setAutoDelete( FALSE );
    doc->monitor.modeLines.clear();

    done = new bool[cnt];
    for ( i = 0; i <cnt ; i++ ) done[i] = FALSE;

    // find exact matches
    for ( i = 0; i < cnt ; i++ ) {
        for ( ml = startup.first(); ml != NULL ; ml = startup.next() ) {
            if (   (ml->GetHorizontalResolution() == running[i].GetHorizontalResolution())
                && (ml->GetHorizontalSyncStart()  == running[i].GetHorizontalSyncStart())
                && (ml->GetHorizontalSyncEnd()    == running[i].GetHorizontalSyncEnd()) 
                && (ml->GetHorizontalSyncTotal()  == running[i].GetHorizontalSyncTotal()) 
                && (ml->GetVerticalResolution()   == running[i].GetVerticalResolution()) 
                && (ml->GetVerticalSyncStart()    == running[i].GetVerticalSyncStart()) 
                && (ml->GetVerticalSyncEnd()      == running[i].GetVerticalSyncEnd()) 
                && (ml->GetVerticalSyncTotal()    == running[i].GetVerticalSyncTotal()) 
                && (ml->GetDotClock()             == running[i].GetDotClock()) 
                && (ml->GetFlags()                == running[i].GetFlags()) 
            ) {
                name =  ml->GetName();
                running[i].SetName(name);
		free(name);
		startup.remove();
		done[i] = TRUE;
		break;
            }
	}
    }
	

    // find more relaxed matches
    for ( exact = 1 ; exact >= 0 ; exact-- ) {
        for ( i = 0; i < cnt ; i++ ) {
            if (done[i]) continue;
    
            name    = NULL;
            hit     = 0;
            lasthit = NULL;
                
            for ( ml = startup.first(); ml != NULL ; ml = startup.next() ) {
                if ( (ml->GetHorizontalResolution() == running[i].GetHorizontalResolution())
                  && (ml->GetVerticalResolution() == running[i].GetVerticalResolution()) 
                  && ( (exact == 0) || (ml->GetHorizontalSyncTotal() == running[i].GetHorizontalSyncTotal()) )
                  && ( (exact == 0) || (ml->GetVerticalSyncTotal() == running[i].GetVerticalSyncTotal()) ) 
                   ) 
                {
                    lasthit = ml;
                    if (name) free(name);
                    name =  ml->GetName();
                    ++hit;
                }
            }
        
            if ( hit == 1 ) {
                running[i].SetName(name);
                startup.remove(lasthit);
                done[i] = TRUE;
            }
            if (name) free(name);
        }
    } 

    b = TRUE;

    for ( i = 0; i < cnt ; i++ ) {
        b = b && done[i];
        doc->monitor.modeLines.append(&(running[i]));
    }

    for ( ml = startup.first(); ml != NULL ; ml = startup.next() ) 
        doc->monitor.modeLines.append(ml);
    
    delete done;

    return b;
}

QStrList KVidData::getVideoModeList()
{
    QStrList  list;
    char      *s;
    int       cnt, i;

    list.clear();

    if ( xf86vid == NULL ) {
        SetErrorText(klocale->translate("the currently active configuration values are not available"));
        return list;
    }
 
    cnt = vidCfg->cfgHideNonActiveModeLines ? xf86vid->GetModeCount() : doc->monitor.modeLines.count();

    for ( i = 0; i<cnt ; i++ ) {
        s = doc->monitor.modeLines.at(i)->GetName();
        list.append(s);
	free(s);
    }

    return list;
}


bool KVidData::updateXF86ConfigMonitor()
{
    bool b = TRUE;

    b = b && xf86cfg->SetValueStr(sectionMonitor,"Identifier",doc->monitor.identifier);
    b = b && xf86cfg->SetValueStr(sectionMonitor,"VendorName",doc->monitor.vendorName);
    b = b && xf86cfg->SetValueStr(sectionMonitor,"ModelName", doc->monitor.modelName);
    b = b && xf86cfg->SetValueRangeList(sectionMonitor,"HorizSync", doc->monitor.horizSync, 1000);
    b = b && xf86cfg->SetValueRangeList(sectionMonitor,"VertRefresh", doc->monitor.vertRefresh, 1);
    b = b && xf86cfg->SetValueModelines(sectionMonitor,"Modeline", doc->monitor.modeLines);
    
    return b;
}

bool KVidData::updateXF86ConfigCard()
{
    bool b = TRUE;

    b = b && xf86cfg->SetValueStr(sectionDevice,"Identifier",doc->device.identifier);
    b = b && xf86cfg->SetValueStr(sectionDevice,"VendorName",doc->device.vendorName);
    b = b && xf86cfg->SetValueStr(sectionDevice,"BoardName", doc->device.boardName);
    b = b && xf86cfg->SetValueStr(sectionDevice,"Chipset",   doc->device.chipset);
    if ( doc->device.videoRam == 0 ) {
        b = b && xf86cfg->RemoveEntry(sectionDevice,"VideoRam");
    } else {
        b = b && xf86cfg->SetValueInt(sectionDevice,"VideoRam", doc->device.videoRam);
    }
    
    return b;
}

bool KVidData::updateXF86ConfigDriver()
{
    bool b = TRUE;

    b = b && xf86cfg->SetValueStr(sectionScreen,"Driver",            doc->screen.driver);
    b = b && xf86cfg->SetValueStr(sectionScreen,"Device",            doc->screen.device);
    b = b && xf86cfg->SetValueStr(sectionScreen,"Monitor",           doc->screen.monitor);
    // TODO: blankTime
    // TODO: suspendTime
    // TODO: offTime
    if ( doc->screen.defaultColorDepth == 0 ) {
         b = b && xf86cfg->RemoveEntry(sectionScreen,"DefaultColorDepth");
    } else {
         b = b && xf86cfg->SetValueInt(sectionScreen,"DefaultColorDepth", doc->screen.defaultColorDepth);
    }
    b = b && setXF86ResolutionList(sectionScreen,doc->screen.resolutionList);
    
    return b;
}

bool KVidData::updateXF86Config()
{
    bool b = TRUE;

    b = b && updateXF86ConfigDriver();
    b = b && updateXF86ConfigCard();
    b = b && updateXF86ConfigMonitor();

    return b;
}

