/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <values.h>
#include <ctype.h>

#include <qfileinfo.h> 
#include <qmsgbox.h> 

#include <kapp.h>
#include <kfiledialog.h>

#include "../dialogmargin.h"
#include "printerdialog.h"

CPrinterDialog::CPrinterDialog( AdvancedPrinter *printer, const char *title, QWidget *parent,
			      char *name, bool modal )
   : CPageDialog( CPageDialog::TABS,  CPageDialog::HELP | CPageDialog::PREVIEW |
		 CPageDialog::PRINT | CPageDialog::CANCEL,
		 CPageDialog::PRINT, title, parent, name, modal )
{
	mPrinter = printer;

  mFrame[ page_destination ] = addPage( -1,i18n("Destination"),
					i18n("Destination"));
  mFrame[ page_option ] = addPage( -1, i18n("Options"), i18n("Options") );
  mFrame[ page_layout ] = addPage( -1, i18n("Page layout"), i18n("Layout") );

  setupDestinationPage();
  setupOptionPage();
  setupLayoutPage();

  adjustSize();
  showPage( page_destination );

  bool success = parsePrintcap();
  
  mConfig = 0;
  readConfiguration();

  if( success == false )
  {
    //
    // I postpone an error message. The reason is that I dont want to 
    // display the message before the dialog box becomes visible. 
    // 
    startTimer( 100 );
  }
}



CPrinterDialog::~CPrinterDialog( void )
{
  writeConfiguration();
  delete mConfig; mConfig = 0;
}


void CPrinterDialog::readConfiguration( void )
{
  if( mConfig != 0 ) {return; }

  QString configPath( kapp->localconfigdir() );
  QString configFile( "/bibletimeprinter" );

  mConfig = new KSimpleConfig( configPath + configFile );
  if( mConfig == 0 ) { return; }

  mConfig->setGroup( "Destination" );
  
  bool state = mConfig->readBoolEntry( "PrintToFile", false );
  mDestination.fileCheck->setChecked( state );
  printFileCheck();
  const char *text = mConfig->readEntry( "FileName" );
  mDestination.fileInput->setText( text == 0 ? "" :  text );
  QString qText( mConfig->readEntry( "PrinterName" ) );
  setSelectedPrinter( qText );
  state = mConfig->readBoolEntry( "Postscript", true );
  mDestination.postscriptRadio->setChecked( state );
  mDestination.textRadio->setChecked( state == true ? false : true );
  textFormatClicked( state == true ? 0 : 1 );

  mConfig->setGroup( "Option" );
  state = mConfig->readBoolEntry( "Portrait", true );
  mOption.portraitRadio->setChecked( state );
  mOption.landscapeRadio->setChecked( state == true ? false : true );
  text = mConfig->readEntry( "PaperType", "A4" );
//  comboMatchText( mOption.paperSelector, text );

  state = mConfig->readBoolEntry( "ScaleToFit", true );
  mOption.scaleCheck->setChecked( state );
  int val = mConfig->readNumEntry( "NumberOfCopies", 1 );
  mOption.pageSpin->setValue( val );
 
  state = mConfig->readBoolEntry( "FirstPageFirst", true );
  mOption.firstRadio->setChecked( state );
  mOption.lastRadio->setChecked( state == true ? false : true );
  text = mConfig->readEntry( "PreviewApplication", "kghostview" );
  mOption.previewInput->setText( text == 0 ? "" :  text );

  text = mConfig->readEntry( "Color" );
  if( text == 0 || strcmp( text, "color" ) == 0 )
  {
    mOption.colorGroup->setButton( 0 );
  }
  else if( strcmp( text, "gray" ) == 0 )
  {
    mOption.colorGroup->setButton( 1 );
  }
  else if( strcmp( text, "blackWhite" ) == 0 )
  {
    mOption.colorGroup->setButton( 2 );
  }
  else
  {
    mOption.colorGroup->setButton( 0 );
  }

  mConfig->setGroup( "PageLayout" );
  val = mConfig->readNumEntry( "MarginTop", 15 );
  mLayout.marginSpin[0]->setValue( val );
  val = mConfig->readNumEntry( "MarginBottom", 15 );
  mLayout.marginSpin[1]->setValue( val );
  val = mConfig->readNumEntry( "MarginLeft", 15 );
  mLayout.marginSpin[2]->setValue( val );
  val = mConfig->readNumEntry( "MarginRight", 15 );
  mLayout.marginSpin[3]->setValue( val );
  state = mConfig->readBoolEntry( "DrawHeader", true );
  mLayout.headerCheck->setChecked( state );
  slotDrawHeader( state );
  state = mConfig->readBoolEntry( "DrawFooter", true );
  mLayout.footerCheck->setChecked( state );
  slotDrawFooter( state );

  mLayout.headerCombo[0]->setCurrentItem(
    headerTextIndex( mConfig->readEntry("HeaderLeft","DateTime")));
  mLayout.headerCombo[1]->setCurrentItem(
    headerTextIndex( mConfig->readEntry("HeaderCenter")));
  mLayout.headerCombo[2]->setCurrentItem(
    headerTextIndex( mConfig->readEntry("HeaderRight","FreeString")));
  mLayout.headerCombo[3]->setCurrentItem(
    headerLineIndex( mConfig->readEntry("HeaderLine","SingleLine")));

  mLayout.footerCombo[0]->setCurrentItem(
    headerTextIndex( mConfig->readEntry("FooterLeft")));
  mLayout.footerCombo[1]->setCurrentItem(
    headerTextIndex( mConfig->readEntry("FooterCenter","PageNumber")));
  mLayout.footerCombo[2]->setCurrentItem(
    headerTextIndex( mConfig->readEntry("FooterRight")));
  mLayout.footerCombo[3]->setCurrentItem(
    headerLineIndex( mConfig->readEntry("FooterLine","SingleLine")));

}

void CPrinterDialog::writeConfiguration( void )
{
  if( mConfig == 0 )
  {
    return;
  }

  mConfig->setGroup( "Destination" );
  mConfig->writeEntry( "PrintToFile", 
    mDestination.fileCheck->isChecked() );
  mConfig->writeEntry( "FileName", 
    mDestination.fileInput->text() );
  QListViewItem *listItem = mDestination.printerList->currentItem();
  mConfig->writeEntry( "PrinterName", 
    listItem == 0 ? "" : listItem->text(0) );
  mConfig->writeEntry( "Postscript", 
    mDestination.postscriptRadio->isChecked() );

  mConfig->setGroup( "Option" );
  mConfig->writeEntry( "Portrait", 
    mOption.portraitRadio->isChecked() );
  mConfig->writeEntry( "PaperType", 
    mOption.paperSelector->currentText() );
  mConfig->writeEntry( "ScaleToFit", 
    mOption.scaleCheck->isChecked() );
  mConfig->writeEntry( "NumberOfCopies", 
    mOption.pageSpin->value() );
  mConfig->writeEntry( "FirstPageFirst", 
    mOption.firstRadio->isChecked() );
  mConfig->writeEntry( "PreviewApplication", 
    mOption.previewInput->text() );
  if( mOption.colorRadio->isChecked() == true )
  {
    mConfig->writeEntry( "Color", "color" );
  }
  else if( mOption.grayRadio->isChecked() == true )
  {
    mConfig->writeEntry( "Color", "gray" );
  }
  else
  {
    mConfig->writeEntry( "Color", "blackWhite" );
  }

  mConfig->setGroup( "PageLayout" );
  mConfig->writeEntry( "MarginTop", mLayout.marginSpin[0]->value() );
  mConfig->writeEntry( "MarginBottom", mLayout.marginSpin[1]->value() );
  mConfig->writeEntry( "MarginLeft", mLayout.marginSpin[2]->value() );
  mConfig->writeEntry( "MarginRight", mLayout.marginSpin[3]->value() );
  mConfig->writeEntry( "DrawHeader", mLayout.headerCheck->isChecked() );
  mConfig->writeEntry( "DrawFooter", mLayout.footerCheck->isChecked() );

  mConfig->writeEntry( "HeaderLeft", 
    headerText( mLayout.headerCombo[0]->currentItem() ) );
  mConfig->writeEntry( "HeaderCenter",
    headerText( mLayout.headerCombo[1]->currentItem() ) );
  mConfig->writeEntry( "HeaderRight",
    headerText( mLayout.headerCombo[2]->currentItem() ) );
  mConfig->writeEntry( "HeaderLine",
    headerLine( mLayout.headerCombo[3]->currentItem() ) );

  mConfig->writeEntry( "FooterLeft",
    headerText( mLayout.footerCombo[0]->currentItem() ) );
  mConfig->writeEntry( "FooterCenter",
    headerText( mLayout.footerCombo[1]->currentItem() ) );
  mConfig->writeEntry( "FooterRight",
    headerText( mLayout.footerCombo[2]->currentItem() ) );
  mConfig->writeEntry( "FooterLine",
    headerLine( mLayout.footerCombo[3]->currentItem() ) );

  mConfig->sync();
}


const char *CPrinterDialog::headerText( uint index )
{
  static const char *text[4] = {"None","DateTime","PageNumber","FreeString"};
  return( text[ index > 3 ? 0 : index ] );
}


int CPrinterDialog::headerTextIndex( const char *headerText )
{
  static const char *text[4] = {"None","DateTime","PageNumber","FreeString"};
  if( headerText != 0 )
  {
    for( int i=0; i<4; i++ )
    {
      if( strcmp( headerText, text[i] ) == 0 ) { return( i ); }
    }
  }  
  return( 0 );
}


const char *CPrinterDialog::headerLine( uint index )
{
  static const char *text[3] = {"None","SingleLine","Rectangle"};
  return( text[ index > 2 ? 0 : index ] );
}


int CPrinterDialog::headerLineIndex( const char *headerLine )
{
  static const char *text[3] = {"None","SingleLine","Rectangle"};
  if( headerLine != 0 )
  {
    for( int i=0; i<3; i++ )
    {
      if( strcmp( headerLine, text[i] ) == 0 ) { return( i ); }
    }
  }  
  return( 0 );
}
  
void CPrinterDialog::setupDestinationPage( void )
{
  const char *text;
  QFrame *page = mFrame[ page_destination ];

  QVBoxLayout *topLayout = new QVBoxLayout( page, OUTER_SEP, INNER_SEP );  
  if( topLayout == 0 ) { return; }

  mDestination.printerList = new QListView( page, "printerList" );
  mDestination.printerList->setAllColumnsShowFocus( TRUE );
  mDestination.printerList->addColumn( i18n("Printer"), 125 );
  mDestination.printerList->addColumn( i18n("Host"), 125 );
  mDestination.printerList->addColumn( i18n("Comment"), 125 );
  mDestination.printerList->setFrameStyle( QFrame::WinPanel + QFrame::Sunken );
  mDestination.printerList->setMinimumSize( 380, // (380 == 125 * 3 + 5) 
    mDestination.printerList->fontMetrics().height() * 5 + 4 );


  text = i18n("&Available printers"); 
  mDestination.printerLabel = new QLabel( mDestination.printerList,text,page );
  mDestination.printerLabel->setFixedHeight( 
    mDestination.printerLabel->sizeHint().height() );
  topLayout->addWidget( mDestination.printerLabel );
  topLayout->addWidget( mDestination.printerList, 10 );

  text = i18n("Print to fi&le");
  mDestination.fileCheck = new QCheckBox( text, page );
  mDestination.fileCheck->setFixedHeight( 
    mDestination.fileCheck->sizeHint().height() );
  topLayout->addWidget( mDestination.fileCheck );
  connect( mDestination.fileCheck, SIGNAL(clicked()), SLOT(printFileCheck()) );

 
  QHBoxLayout *hbox = new QHBoxLayout();
  if( hbox == 0 ) { return; }
  topLayout->addLayout( hbox );

  mDestination.fileInput = new QLineEdit( page );
  //Fixed height, see below
  hbox->addWidget( mDestination.fileInput , 0, AlignTop );

  text = i18n("&Browse");
  mDestination.browseButton = new QPushButton( text, page, "browse" );
  mDestination.browseButton->setFixedSize( 
    mDestination.browseButton->sizeHint() );
  mDestination.fileInput->setFixedHeight( 
    mDestination.browseButton->sizeHint().height() );
  hbox->addWidget( mDestination.browseButton, 0, AlignTop );
  connect( mDestination.browseButton, SIGNAL(clicked()),
	   SLOT(browserClicked()) );

  QGridLayout *gbox = new QGridLayout( 1, 2, INNER_SEP );
  if( gbox == 0 ) { return; }
  topLayout->addLayout( gbox );

  QButtonGroup *group = new QButtonGroup( i18n("Print range"), page );
  if( group == 0 ) { return; }
  gbox->addWidget( group, 0, 0 );

  QVBoxLayout *vbox = new QVBoxLayout( group, INNER_SEP, INNER_SEP );  
  vbox->addSpacing( group->fontMetrics().height() );
  
  mDestination.allRadio = new QRadioButton( i18n("&Everything"), group );
  mDestination.allRadio->setFixedSize( mDestination.allRadio->sizeHint() );
  group->insert( mDestination.allRadio, 0 );
  vbox->addWidget( mDestination.allRadio, 0, AlignLeft );

  mDestination.selectRadio = new QRadioButton( i18n("&Selection"), group);
  mDestination.selectRadio->setFixedSize(mDestination.selectRadio->sizeHint());
  group->insert( mDestination.selectRadio, 1 );
  vbox->addWidget( mDestination.selectRadio, 0, AlignLeft );

  mDestination.rangeRadio = new QRadioButton( i18n("&Range"), group );
  mDestination.rangeRadio->setFixedSize( mDestination.rangeRadio->sizeHint() );
  group->insert( mDestination.rangeRadio, 2 );
  vbox->addWidget( mDestination.rangeRadio, 0, AlignLeft );

  hbox = new QHBoxLayout();
  vbox->addLayout( hbox );
  mDestination.fromInput = new QLineEdit( group );
  mDestination.fromInput->setFixedHeight(
    mDestination.fromInput->sizeHint().height() );
  text = i18n("&From offset"); 
  mDestination.fromLabel = new QLabel( mDestination.fromInput, text, group );
  mDestination.fromLabel->setFixedHeight( 
    mDestination.fromLabel->sizeHint().height() );
  hbox->addWidget( mDestination.fromLabel );
  hbox->addWidget( mDestination.fromInput, 0, AlignLeft );

  hbox = new QHBoxLayout();
  vbox->addLayout( hbox );
  mDestination.toInput = new QLineEdit( group );
  mDestination.toInput->setFixedHeight(
    mDestination.toInput->sizeHint().height() );
  text = i18n("&To offset"); 
  mDestination.toLabel = new QLabel( mDestination.toInput, text, group );
  mDestination.toLabel->setFixedHeight( 
    mDestination.toLabel->sizeHint().height() );
  hbox->addWidget( mDestination.toLabel );
  hbox->addWidget( mDestination.toInput, 0, AlignLeft );
  vbox->addStretch();
  vbox->activate();
  connect( group, SIGNAL(clicked(int)), SLOT(printRangeClicked(int)) );

  group = new QButtonGroup( i18n("Text formatting"), page );
  if( group == 0 ) { return; }
  gbox->addWidget( group, 0, 1 );

  vbox = new QVBoxLayout( group, INNER_SEP, INNER_SEP );  
  vbox->addSpacing( group->fontMetrics().height() );
  
  text = i18n("Print as postscr&ipt");
  mDestination.postscriptRadio = new QRadioButton( text, group );
  mDestination.postscriptRadio->setFixedSize( 
    mDestination.postscriptRadio->sizeHint() );
  group->insert( mDestination.postscriptRadio, 0 );
  vbox->addWidget( mDestination.postscriptRadio, 0, AlignLeft );

  text = i18n("Print as plain &text");
  mDestination.textRadio = new QRadioButton( text, group );
  mDestination.textRadio->setFixedSize( mDestination.textRadio->sizeHint() );
  group->insert( mDestination.textRadio, 1 );
  vbox->addWidget( mDestination.textRadio, 0, AlignLeft );
  
  text = i18n("This choice will ignore\ncertain options. Refer to\nhelp for more information.");
  mDestination.helpLabel = new QLabel( text, group );
  mDestination.helpLabel->setFrameStyle( QFrame::Panel | QFrame::Sunken );
  mDestination.helpLabel->setFixedSize( mDestination.helpLabel->sizeHint() );
  mDestination.helpLabel->setAlignment( AlignBottom | AlignLeft );
  vbox->addWidget( mDestination.helpLabel, 0, AlignCenter );

  connect( group, SIGNAL(clicked(int)), SLOT(textFormatClicked(int)) );

  vbox->addStretch();
  vbox->activate();

  mDestination.fileCheck->setChecked( false );
  printFileCheck();
  mDestination.allRadio->setChecked( true );
  printRangeClicked( 0 );
  mDestination.postscriptRadio->setChecked( true );
  textFormatClicked( 0 );

  
  topLayout->activate();
}

void CPrinterDialog::setupOptionPage( void )
{
  const char *text;
  QFrame *page = mFrame[ page_option ];

  QVBoxLayout *topLayout = new QVBoxLayout( page, OUTER_SEP, INNER_SEP );  
  if( topLayout == 0 ) { return; }

  QGridLayout *gbox = new QGridLayout( 1, 2, INNER_SEP );
  if( gbox == 0 ) { return; }
  topLayout->addLayout( gbox );

  mOption.oriGroup = new QButtonGroup( i18n("Orientation"), page );
  if( mOption.oriGroup == 0 ) { return; }
  gbox->addWidget( mOption.oriGroup, 0, 0 );

  QVBoxLayout *vbox = new QVBoxLayout( mOption.oriGroup, INNER_SEP, INNER_SEP);
  vbox->addSpacing( mOption.oriGroup->fontMetrics().height() );
  
  mOption.portraitRadio = new QRadioButton( i18n("Por&trait"), 
					    mOption.oriGroup );
  mOption.portraitRadio->setFixedSize( mOption.portraitRadio->sizeHint() );
  mOption.oriGroup->insert( mOption.portraitRadio, 0 );
  vbox->addWidget( mOption.portraitRadio, 0, AlignLeft );

  mOption.landscapeRadio = new QRadioButton( i18n("L&andscape"), 
					     mOption.oriGroup);
  mOption.landscapeRadio->setFixedSize( mOption.landscapeRadio->sizeHint() );
  mOption.oriGroup->insert( mOption.landscapeRadio, 1 );
  vbox->addWidget( mOption.landscapeRadio, 0, AlignLeft );
  vbox->addStretch();
  vbox->activate();


  mOption.colorGroup = new QButtonGroup( i18n("Colors"), page );
  if( mOption.colorGroup == 0 ) { return; }
  gbox->addWidget( mOption.colorGroup, 0, 1 );

  vbox = new QVBoxLayout( mOption.colorGroup, INNER_SEP, INNER_SEP );  
  vbox->addSpacing( mOption.colorGroup->fontMetrics().height() );
  
  text = i18n("Print in c&olor");
  mOption.colorRadio = new QRadioButton( text, mOption.colorGroup );
  mOption.colorRadio->setFixedSize( mOption.colorRadio->sizeHint() );
  mOption.colorGroup->insert( mOption.colorRadio, 0 );
  vbox->addWidget( mOption.colorRadio, 0, AlignLeft );

  text = i18n("Print in &grayscale");
  mOption.grayRadio = new QRadioButton( text, mOption.colorGroup );
  mOption.grayRadio->setFixedSize( mOption.grayRadio->sizeHint() );
  mOption.colorGroup->insert( mOption.grayRadio, 1 );
  vbox->addWidget( mOption.grayRadio, 0, AlignLeft );

  text = i18n("Print in black and &white only");
  mOption.bwRadio = new QRadioButton( text, mOption.colorGroup );
  mOption.bwRadio->setFixedSize( mOption.bwRadio->sizeHint() );
  mOption.colorGroup->insert( mOption.bwRadio, 1 );
  vbox->addWidget( mOption.bwRadio, 0, AlignLeft );
  vbox->addStretch();
  vbox->activate();

  int addSpace = INNER_SEP;
  topLayout->addSpacing( addSpace );

  gbox = new QGridLayout( 4, 3, INNER_SEP+addSpace );
  if( gbox == 0 ) { return; }
  topLayout->addLayout( gbox );
  gbox->setColStretch( 2, 10 );

  mOption.paperSelector = new QComboBox( false, page );
  mOption.paperSelector->setFixedHeight( 
    mOption.paperSelector->sizeHint().height() );
  mOption.paperSelector->setMinimumWidth( 
    mOption.paperSelector->fontMetrics().width("M")*10 );
  connect( mOption.paperSelector, SIGNAL(activated(int)), 
	   SLOT(paperTypeChanged(int)) );

  text = i18n("Paper t&ype");
  QLabel *paperLabel = new QLabel( mOption.paperSelector, text, page );
  paperLabel->setFixedHeight( mOption.paperSelector->sizeHint().height() );
  paperLabel->setMinimumWidth( paperLabel->sizeHint().width() );

  text = i18n("Size: 8888mm x 8888mm");
  mOption.paperSizeLabel = new QLabel( text, page );
  mOption.paperSizeLabel->setFixedHeight( 
    mOption.paperSelector->sizeHint().height() );
  mOption.paperSizeLabel->setMinimumWidth( 
    mOption.paperSizeLabel->sizeHint().width() );
  mOption.paperSizeLabel->hide();
 
  gbox->addWidget( paperLabel, 0, 0 );
  gbox->addWidget( mOption.paperSelector, 0, 1 );
  gbox->addWidget( mOption.paperSizeLabel, 0, 2 );


  text = i18n("&Scale down printed document to fit selected paper type");
  mOption.scaleCheck = new QCheckBox( text, page );
  if( mOption.scaleCheck == 0 ) { return; }
  mOption.scaleCheck->setFixedSize( mOption.scaleCheck->sizeHint() );
  gbox->addMultiCellWidget( mOption.scaleCheck, 1, 1, 0, 2, AlignLeft );


  mOption.pageSpin = new QSpinBox( page );
  mOption.pageSpin->setFixedHeight( mOption.pageSpin->sizeHint().height() );
  mOption.pageSpin->setMinimumWidth( 
    mOption.pageSpin->fontMetrics().width("M")*10 );
  mOption.pageSpin->setRange( 1, MAXINT );

  text = i18n("&Number of copies");
  QLabel *pageLabel = new QLabel( mOption.pageSpin, text, page );
  pageLabel->setFixedHeight( mOption.pageSpin->sizeHint().height() );
  pageLabel->setMinimumWidth( pageLabel->sizeHint().width() );
  
  gbox->addWidget( pageLabel, 2, 0 );
  gbox->addWidget( mOption.pageSpin, 2, 1 );
 
  QButtonGroup *group = new QButtonGroup( page );
  if( group == 0 ) { return; }
  group->setFrameStyle( QFrame::NoFrame );
  gbox->addMultiCellWidget( group,  3, 3, 0, 2, AlignLeft );


  QHBoxLayout *hbox = new QHBoxLayout( group, 0, INNER_SEP );
  if( hbox == 0 ) { return; }

  text = i18n("Print &first page first");
  mOption.firstRadio = new QRadioButton( text, group );
  if( mOption.firstRadio == 0 ) { return; }
  mOption.firstRadio->setFixedSize( mOption.firstRadio->sizeHint() );
  group->insert( mOption.firstRadio, 0 );
  hbox->addWidget( mOption.firstRadio, 0, AlignLeft );

  text = i18n("Print &last page first");
  mOption.lastRadio = new QRadioButton( text, group );
  if( mOption.lastRadio == 0 ) { return; }
  mOption.lastRadio->setFixedSize( mOption.lastRadio->sizeHint() );
  group->insert( mOption.lastRadio, 0 );
  hbox->addWidget( mOption.lastRadio, 0, AlignLeft );

  hbox->addStretch(10);
  hbox->activate();

  QFrame *hline = new QFrame( page );
  if( hline == 0 ) { return; }
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  hline->setFixedHeight( hline->sizeHint().height() );
  topLayout->addWidget( hline );

  
  text = i18n("Pr&eview application");
  QLabel *previewLabel = new QLabel( text, page );
  if( previewLabel == 0 ) { return; }
  previewLabel->setFixedSize( previewLabel->sizeHint() );
  topLayout->addWidget( previewLabel, 0, AlignLeft );

  hbox = new QHBoxLayout();
  if( hbox == 0 ) { return; }
  topLayout->addLayout( hbox );

  mOption.previewInput = new QLineEdit( page );
  if( mOption.previewInput == 0 ) { return; }
  //Fixed height, see below
  hbox->addWidget( mOption.previewInput , 0, AlignTop );
  previewLabel->setBuddy( mOption.previewInput );

  text = i18n("&Browse");
  mOption.browseButton = new QPushButton( text, page, "browse" );
  mOption.browseButton->setFixedSize( mOption.browseButton->sizeHint() );
  mOption.previewInput->setFixedHeight( 
    mOption.browseButton->sizeHint().height() );
  hbox->addWidget( mOption.browseButton, 0, AlignTop );
  connect( mOption.browseButton, SIGNAL(clicked()),
	   SLOT(appBrowserClicked()) );
 

  QStrList list;
  paperType( list );
  mOption.paperSelector->insertStrList( &list );  

  mOption.paperSelector->setCurrentItem( 0 );
  paperTypeChanged( 0 );

  mOption.portraitRadio->setChecked( true );
  mOption.colorRadio->setChecked( true );
  mOption.firstRadio->setChecked( true );
  mOption.scaleCheck->setChecked( true );
  mOption.previewInput->setText( "kghostview" );

  topLayout->addStretch(10);
  topLayout->activate();
}





void CPrinterDialog::setupLayoutPage( void )
{
  QString text;
  QFrame *page = mFrame[ page_layout ];

  QVBoxLayout *topLayout = new QVBoxLayout( page, OUTER_SEP, INNER_SEP );  
  if( topLayout == 0 ) { return; }

  text = i18n("Margins (in millimeter)");
  QButtonGroup *group = new QButtonGroup( text, page );
  if( group == 0 ) { return; }
  topLayout->addWidget( group );
 
  QGridLayout *gbox = new QGridLayout( group, 3, 6, INNER_SEP, INNER_SEP );
  if( gbox == 0 ) { return; }
  gbox->addRowSpacing( 0, group->fontMetrics().height()-INNER_SEP );
  gbox->setColStretch( 5, 10 );
  
  QString name[4];
  int i;
  
  name[0] = i18n("&Top");
  name[1] = i18n("&Bottom");
  name[2] = i18n("&Left");
  name[3] = i18n("&Right");

  for( i=0; i<4; i++ )
  {
    mLayout.marginSpin[i] = new QSpinBox( group );
    mLayout.marginSpin[i]->setFixedHeight( 
      mLayout.marginSpin[i]->sizeHint().height() );
    mLayout.marginSpin[i]->setMinimumWidth( 
      mLayout.marginSpin[i]->fontMetrics().width("M")*10 );
    mLayout.marginSpin[i]->setRange( 0, MAXINT );

    QLabel *label = new QLabel( mLayout.marginSpin[i], name[i], group );
    label->setFixedHeight( mLayout.marginSpin[i]->sizeHint().height() );
    label->setFixedWidth( label->sizeHint().width() );

    if( i < 2 )
    {
      gbox->addWidget( label, i+1, 0, AlignLeft );
      gbox->addWidget( mLayout.marginSpin[i], i+1, 1, AlignLeft );
    }
    else
    {
      gbox->addWidget( label, i-1, 3, AlignLeft );
      gbox->addWidget( mLayout.marginSpin[i], i-1, 4, AlignLeft );
    }
  }
  gbox->activate();
  
  text = i18n("Draw h&eader above text");
  mLayout.headerCheck = new QCheckBox( text, page );
  mLayout.headerCheck->setFixedSize( mLayout.headerCheck->sizeHint() );
  connect( mLayout.headerCheck, SIGNAL( toggled(bool)),
	   SLOT( slotDrawHeader(bool)));
  topLayout->addWidget( mLayout.headerCheck, 0, AlignLeft );

  gbox = new QGridLayout( 5, 6, 0 );
  if( gbox == 0 ) { return; }
  topLayout->addLayout( gbox );

  gbox->setColStretch ( 5, 10 );
  gbox->addColSpacing( 1, INNER_SEP );
  gbox->addColSpacing( 3, INNER_SEP );

  gbox->addRowSpacing( 2, INNER_SEP );
  
  name[0] = i18n("Left");
  name[1] = i18n("Center");
  name[2] = i18n("Right");
  name[3] = i18n("Border");

  QStrList textList;
  textList.append(i18n("None"));
  textList.append(i18n("Date and time"));
  textList.append(i18n("Page number"));
  textList.append(i18n("FreeString"));

  QStrList lineList;
  lineList.append(i18n("None"));
  lineList.append(i18n("Single line"));
  lineList.append(i18n("Rectangle"));

  for( i=0; i<4; i++ )
  {
    mLayout.headerCombo[i] = new QComboBox( false, page );
    mLayout.headerCombo[i]->setFixedHeight( 
      mLayout.headerCombo[i]->sizeHint().height() );
    mLayout.headerCombo[i]->setMinimumWidth( 
      mLayout.headerCombo[i]->fontMetrics().width("M")*10 );
 
    mLayout.headerLabel[i] = new QLabel( mLayout.headerCombo[i], name[i], 
					 page );
    mLayout.headerLabel[i]->setFixedHeight( 
      mLayout.headerLabel[i]->sizeHint().height() );
    mLayout.headerLabel[i]->setFixedWidth( 
      mLayout.headerLabel[i]->sizeHint().width() );

    if( i<3 )
    {
      mLayout.headerCombo[i]->insertStrList( &textList );
      gbox->addWidget( mLayout.headerLabel[i], 0, i*2, AlignLeft );
      gbox->addWidget( mLayout.headerCombo[i], 1, i*2, AlignLeft );
    }
    else
    {
      mLayout.headerCombo[i]->insertStrList( &lineList );
      gbox->addWidget( mLayout.headerLabel[i], 3, 0, AlignLeft );
      gbox->addWidget( mLayout.headerCombo[i], 4, 0, AlignLeft );
    }
  }


  text = i18n("Draw &footer below text");
  mLayout.footerCheck = new QCheckBox( text, page );
  mLayout.footerCheck->setFixedSize( mLayout.footerCheck->sizeHint() );
  connect( mLayout.footerCheck, SIGNAL( toggled(bool)),
	   SLOT( slotDrawFooter(bool)));
  topLayout->addWidget( mLayout.footerCheck, 0, AlignLeft );

  gbox = new QGridLayout( 5, 6, 0 );
  if( gbox == 0 ) { return; }
  topLayout->addLayout( gbox );

  gbox->setColStretch ( 5, 10 );
  gbox->addColSpacing( 1, INNER_SEP );
  gbox->addColSpacing( 3, INNER_SEP );

  gbox->addRowSpacing( 2, INNER_SEP );

  for( i=0; i<4; i++ )
  {
    mLayout.footerCombo[i] = new QComboBox( false, page );
    mLayout.footerCombo[i]->setFixedHeight( 
      mLayout.footerCombo[i]->sizeHint().height() );
    mLayout.footerCombo[i]->setMinimumWidth( 
      mLayout.footerCombo[i]->fontMetrics().width("M")*10 );

    mLayout.footerLabel[i] = new QLabel( mLayout.footerCombo[i], name[i], 
					 page );
    mLayout.footerLabel[i]->setFixedHeight( 
      mLayout.footerLabel[i]->sizeHint().height() );
    mLayout.footerLabel[i]->setFixedWidth( 
      mLayout.footerLabel[i]->sizeHint().width() );

    if( i<3 )
    {
      mLayout.footerCombo[i]->insertStrList( &textList );
      gbox->addWidget( mLayout.footerLabel[i], 0, i*2, AlignLeft );
      gbox->addWidget( mLayout.footerCombo[i], 1, i*2, AlignLeft );
    }
    else
    {
      mLayout.footerCombo[i]->insertStrList( &lineList );
      gbox->addWidget( mLayout.footerLabel[i], 3, 0, AlignLeft );
      gbox->addWidget( mLayout.footerCombo[i], 4, 0, AlignLeft );
    }
  }

  for( i=0; i<4; i++ )
  {
    mLayout.marginSpin[i]->setValue( 15 );
  }
  mLayout.headerCheck->setChecked( true );
  slotDrawHeader( true );
  mLayout.footerCheck->setChecked( true );
  slotDrawFooter( true );

  topLayout->addStretch(10);
  topLayout->activate();

}




void CPrinterDialog::printFileCheck( void )
{
  bool state = mDestination.fileCheck->isChecked();
  mDestination.fileInput->setEnabled( state );
  mDestination.browseButton->setEnabled( state );
  state = state == true ? false : true;
  mDestination.printerLabel->setEnabled( state );
  mDestination.printerList->setEnabled( state );
}

void CPrinterDialog::browserClicked( void )
{
  QString url = KFileDialog::getSaveFileName( 0, "*", topLevelWidget() );
  if( url.isEmpty() == true )
  {
    return;
  }

  mDestination.fileInput->setText( url.data() );
}


void CPrinterDialog::appBrowserClicked( void )
{
  QString url = KFileDialog::getOpenFileName( 0, "*", topLevelWidget() );
  if( url.isEmpty() == true )
  {
    return;
  }

  mOption.browseButton->setText( url.data() );
}


void CPrinterDialog::printRangeClicked( int id )
{
  bool state = id == 2 ? true : false;
  mDestination.toLabel->setEnabled( state );
  mDestination.fromLabel->setEnabled( state );
  mDestination.toInput->setEnabled( state );
  mDestination.fromInput->setEnabled( state );
}

void CPrinterDialog::textFormatClicked( int id )
{
  mDestination.helpLabel->setEnabled( id == 1 ? true : false );
}


void CPrinterDialog::paperTypeChanged( int id )
{
  QString msg( i18n("Size: ") ); 
  char buf[30];

  const SPageSize &size = paperSize( (QPrinter::PageSize)id );
  sprintf( buf, "%umm x %umm", size.width, size.height );
  msg += buf;

  mOption.paperSizeLabel->setText( msg.data() ); 
  mOption.paperSizeLabel->show();
}



void CPrinterDialog::slotDrawHeader( bool state )
{
  for( int i=0; i<4; i++ )
  {
    mLayout.headerLabel[i]->setEnabled( state );
    mLayout.headerCombo[i]->setEnabled( state );
  }
}


void CPrinterDialog::slotDrawFooter( bool state )
{
  for( int i=0; i<4; i++ )
  {
    mLayout.footerLabel[i]->setEnabled( state );
    mLayout.footerCombo[i]->setEnabled( state );
  }
}



void CPrinterDialog::slotHelp( void )
{
  kapp->invokeHTMLHelp( QString("bibletime/index.html"), QString() );
}


void CPrinterDialog::slotPrint( void )
{
	setPrintParameter(false);
	accept();
}


void CPrinterDialog::slotPreview( void )
{
	setPrintParameter(true);
	accept();
}

bool CPrinterDialog::parsePrintcap( void )
{
  bool emptyPrintcap, noPrintcap;
  QString tmp, name;

  char *path[] =
  { 
    "/etc/printers.conf", 
    "/etc/printcap", 
    "/usr/local/etc/printcap",
    0
  };

  emptyPrintcap = noPrintcap = TRUE;

  for( uint i=0; path[i] != 0; i++ )
  {
    QFile file( path[i] );
    if( file.open( IO_ReadOnly ) == FALSE )
    {
      continue;
    }

    noPrintcap = FALSE;

    QTextStream st( (QIODevice *)&file );
    while( st.eof() == FALSE )
    {
      tmp = st.readLine();
      tmp = tmp.stripWhiteSpace();
      if( tmp[0] == '#' || tmp.isEmpty() )
      {
	continue;
      }
      name = tmp.left( tmp.find(':') );
      if( name.isEmpty() )
      {
	continue;
      }
      if( name.contains('|') )
      {
	name = name.left( name.find('|') );
      }
      name = name.stripWhiteSpace();
      if( name.isEmpty() )
      {
	continue;
      }
      else
      {
	emptyPrintcap = FALSE;
      }

      addPrinterName( name );
    }

    file.close();
  }

  QString empty;
  setSelectedPrinter( empty ); // Select the first

  return( emptyPrintcap == TRUE ? FALSE : TRUE );
}

void CPrinterDialog::addPrinterName( const QString &printer )
{
  const QListViewItem *entry = mDestination.printerList->firstChild();
  for( ; entry != 0; entry = entry->nextSibling() )
  {
    if( entry->text(0) == printer )
    {
      return;
    }
  }
  
  (void)new QListViewItem( mDestination.printerList, 
			   printer.data(), "", "" );


  /*
  (void)new QListViewItem( mDestination.printerList, 
			   printer.data(), "host", "comment" );
  */
}


void CPrinterDialog::setSelectedPrinter( const QString &name )
{
  QListViewItem *entry = mDestination.printerList->firstChild();
  for( ; entry != 0; entry = entry->nextSibling() )
  {
    if( entry->text(0) == name )
    {
      mDestination.printerList->setSelected ( entry, true );
      return;
    }
  }

  //
  // Use the first
  //
  entry = mDestination.printerList->firstChild();
  if( entry != 0 )
  {
    mDestination.printerList->setSelected ( entry, true );
  }
}



void CPrinterDialog::timerEvent( QTimerEvent *e )
{
  killTimers();
  QString msg = i18n("Unable to parse or read any printcap files" );

  QMessageBox::warning(
    this,
    i18n("Print"),
    msg,
    i18n("OK"),
    "",
    "",
    0,0);
}



void CPrinterDialog::paperType( QStrList &list )
{
  list.append(i18n("A4"));
  list.append(i18n("B5"));
  list.append(i18n("Letter"));
  list.append(i18n("Legal"));
  list.append(i18n("Executive"));
  #if 0
  list.append(i18n("A0"));
  list.append(i18n("A1"));
  list.append(i18n("A2"));
  list.append(i18n("A3"));
  list.append(i18n("A5"));
  list.append(i18n("A6"));
  list.append(i18n("A7"));
  list.append(i18n("A8"));
  list.append(i18n("A9"));
  list.append(i18n("B0"));
  list.append(i18n("B1"));
  list.append(i18n("B10"));
  list.append(i18n("B2"));
  list.append(i18n("B3"));
  list.append(i18n("B4"));
  list.append(i18n("B6"));
  list.append(i18n("B7"));
  list.append(i18n("B8"));
  list.append(i18n("B9"));
  list.append(i18n("C5E"));
  list.append(i18n("Comm10E"));
  list.append(i18n("DLE"));
  list.append(i18n("Folio"));
  list.append(i18n("Ledger"));
  list.append(i18n("Tabloid"));
  #endif
}



const CPrinterDialog::SPageSize CPrinterDialog::paperSize( 
			        QPrinter::PageSize pageSize )
{
  QPrinter printer;
  printer.setPageSize( pageSize );
  QPaintDeviceMetrics paintMetric( &printer );

  SPageSize size = { paintMetric.widthMM(), paintMetric.heightMM() };
  return( size );

  #if 0
  static SPageSize size[ QPrinter::Tabloid+2 ] = 
  {
    {210,  297},  // A4
    {182,  257},  // B5
    {216,  279},  // Letter
    {216,  356},  // Legal
    {191,  254},  // Executive
    {841,  1189}, // A0
    {594,  841},  // A1
    {420,  594},  // A2
    {297,  420},  // A3
    {148,  210},  // A5
    {105,  148},  // A6
    {74,   105},  // A7
    {52,   74},   // A8
    {37,   52},   // A9
    {1030, 1456}, // B0 
    {728,  1030}, // B1
    {32,   45},   // B10
    {515,  728},  // B2
    {364,  515},  // B3
    {257,  364},  // B4
    {128,  182},  // B6
    {91,   128},  // B7
    {64,   91},   // B8
    {45,   64},   // B9
    {163,  229},  // C5E
    {105,  241},  // Comm10E
    {110,  220},  // DLE
    {210,  330},  // Folio
    {432,  279},  // Ledger
    {279,  432},  // Tabloid 
    {0,    0},    // Unknown
  };

  return( size[ pageSize > QPrinter::Tabloid ? QPrinter::Tabloid : pageSize ]);
  #endif
}


/** set the print parameter */
bool CPrinterDialog::setPrintParameter(bool preview) {
/*  The params where set while dialog was shown. The user has clicked
		print or preview, we write the values back into the mPrinter object */
		
  mPrinter->setOutputToStdout( false );
  mPrinter->setAsText( mDestination.textRadio->isChecked() );

	if (preview) {
		// create temp file name
		QString path = "/tmp/bibletime.preview";
		mPrinter->setPreview(true);
    mPrinter->setOutputToFile( true );
    mPrinter->setOutputFileName( path );
    mPrinter->setPreviewApplication(mOption.previewInput->text());
	} else {
		mPrinter->setPreview(false);

	  if( mDestination.fileCheck->isChecked() == true )
  	{
    	QString path( mDestination.fileInput->text() );
	    if( path.isEmpty() == true )
  	  {
    	  QString msg(i18n("You must specify a destination file."));
      	QMessageBox::warning(this,i18n("Print"),msg,i18n("&OK"),"","",0,0 );
	      return false;
  	  }

    	QFileInfo info( path );
	    if( info.exists() == TRUE )
  	  {
    	  if( info.isDir() == TRUE )
      	{
					QString msg(i18n("You have specified an existing directory."));
  	      QMessageBox::warning(this,i18n("Print"),msg,i18n("&OK"),"","",0,0);
					return false;
      	}

	      if( info.isWritable() == FALSE )
  	    {
					QString msg = i18n("You do not have write permission to this file.");
					QMessageBox::warning(this,i18n("Print"),msg,i18n("&OK"),"","",0,0 );
					return false;
	      }

  	    QString msg;
    	  msg += i18n("You have specified an existing file.");
      	msg += "\n";
	      msg += i18n("Overwrite current file?");

  	    int reply = QMessageBox::warning(this,i18n("Print"),msg,i18n( "&Yes" ),
										i18n( "&No" ),"",0, 1 );
      	if( reply != 0 )
	      {
					return false;
    	  }
	    }

  	  mPrinter->setOutputToFile( true );
    	mPrinter->setOutputFileName( path );
	  }
  	else
	  {
  	  QListViewItem *listItem = mDestination.printerList->currentItem();
    	if( listItem == 0 )
	    {
  	    QString msg = i18n("You have not selected a printer.");
    	  QMessageBox::warning(this,i18n("Print"),msg,i18n("&OK"),"","",0,0 );
      	return false;
	    }

  	  mPrinter->setOutputToFile( false );
    	mPrinter->setPrinterName( listItem->text( 0 ) );
	  }
  }

  if( mOption.portraitRadio->isChecked() == true )
  {
    mPrinter->setOrientation( QPrinter::Portrait );
  }
  else
  {
    mPrinter->setOrientation( QPrinter::Landscape );
  }

  mPrinter->setPageSize((QPrinter::PageSize)mOption.paperSelector->currentItem());

  if( mOption.firstRadio->isChecked() == true )
  {
    mPrinter->setPageOrder( QPrinter::FirstPageFirst );
  }
  else
  {
    mPrinter->setPageOrder( QPrinter::LastPageFirst );
  }

  mPrinter->setNumCopies( mOption.pageSpin->value() );
  mPrinter->setScaleToFit( mOption.scaleCheck->isChecked() );

  mPrinter->setTopMarginMM( mLayout.marginSpin[0]->value() );
  mPrinter->setBottomMarginMM( mLayout.marginSpin[1]->value() );
  mPrinter->setLeftMarginMM( mLayout.marginSpin[2]->value() );
  mPrinter->setRightMarginMM( mLayout.marginSpin[3]->value() );

  mPrinter->setPageHeader(mLayout.headerCheck->isChecked(),
			  									mLayout.headerCombo[0]->currentItem(),
			  									mLayout.headerCombo[1]->currentItem(),
			  									mLayout.headerCombo[2]->currentItem(),
			  									mLayout.headerCombo[3]->currentItem() );

  mPrinter->setPageFooter( mLayout.footerCheck->isChecked(),
			  									mLayout.footerCombo[0]->currentItem(),
			  									mLayout.footerCombo[1]->currentItem(),
			  									mLayout.footerCombo[2]->currentItem(),
			  									mLayout.footerCombo[3]->currentItem() );

  if( mOption.colorRadio->isChecked() == true )
  {
    mPrinter->setColorMode( QPrinter::Color );
    mPrinter->setPrintBackWhite( false );
  }
  else if( mOption.grayRadio->isChecked() == true )
  {
    mPrinter->setColorMode( QPrinter::GrayScale );
    mPrinter->setPrintBackWhite( false );
  }
  else
  {
    mPrinter->setColorMode( QPrinter::Color );
    mPrinter->setPrintBackWhite( true );
  }

  if( mDestination.allRadio->isChecked() == true )
  {
    mPrinter->setAll( true );
    mPrinter->setSelection( false );
    mPrinter->setRange( false );
    mPrinter->setFromTo( mPrinter->minPage(), mPrinter->maxPage() );
  }
  else if( mDestination.selectRadio->isChecked() == true )
  {
    mPrinter->setAll( false );
    mPrinter->setSelection( true );
    mPrinter->setRange( false );
  }
  else
  {
    uint start, stop;
    bool ok1 = stringToOffset( mDestination.fromInput->text(), start );
    bool ok2 = stringToOffset( mDestination.toInput->text(), stop );

    if( ok1 == false || ok2 == false || start >= stop )
    {
      QMessageBox::warning(this,i18n("Print"),
      	i18n("Illegal format or missing range specification"),i18n("&OK"),
				"","",0,0 );
      return false;
    }

    mPrinter->setAll( false );
    mPrinter->setSelection( false );
    mPrinter->setRange( true, start, stop );
  }


  if( mDestination.textRadio->isChecked() == true )
  {
    if( mPrinter->outputToFile() == false )
    {
      QString msg = i18n("Not available yet!\nPrint plain text to printer" );
      QMessageBox::warning(this,i18n("Print"),msg,i18n("&OK"),"","",0, 0);
      return false;
    }
  }
  return true;
}

bool CPrinterDialog::stringToOffset( const char *text, uint &offset )
{
  if( text == 0 || strlen(text) == 0 )
    return false;

  const char *p = text;

  //
  // Skip any whitespaces in front of string
  //
  for( ; *p != 0 && isspace( *p ) ; p++ );

  int match = 0;
  bool space = false;
  if( strncmp( p, "0x", 2 ) == 0 || strncmp( p, "0X", 2 ) == 0 )
  {
    for( const char *q = p+2; *q != 0; q++ )
    {
      if( isxdigit( *q ) == 0 || space == true )
      {
				if( isspace( *q ) == 0 )
	  			return( false );
	  			
				space = true;
      }
    }
    match = sscanf( p+2, "%x", &offset );
  }
  else
  {
    for( const char *q = p; *q != 0; q++ )
    {
      if( isdigit( *q ) == 0 || space == true )
      {
				if( isspace( *q ) == 0 )
	  			return false;
	  			
				space = true;
      }
    }
    match = sscanf( p, "%u", &offset );
  }

  if( match == 0 )
    return false;

  return true;
}

