/***************************************************************************
     GrammerManager.cpp - manage grammer parts (articles, conjugation)
                        -------------------
    begin                : Sat Nov 27 09:50:53 MET 1999

    copyright            : (C) 1999,2000 by Ewald Arnold
    email                : ewald@ewald-arnold.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "GrammerManager.h"
#include "eadebug.h"

#ifndef i18n_noop
# define i18n_noop(x) (x)
#endif

Conjugation::conjug_name_t
Conjugation::names [] =
{
  { CONJ_SIMPLE_PRESENT,    i18n_noop("Simple preset") },
  { CONJ_PRESENT_PROGR,     i18n_noop("Preset progressive") },
  { CONJ_PRESENT_PERFECT,   i18n_noop("Preset perfect") },

  { CONJ_SIMPLE_PAST,       i18n_noop("Simple past") },
  { CONJ_PAST_PROGR,        i18n_noop("Past progressive") },
  { CONJ_PAST_PARTICIPLE,   i18n_noop("Past participle") },

  { CONJ_FUTURE,            i18n_noop("Future") }
};

Comparison::Comparison (
           const QString &l1,
           const QString &l2,
           const QString &l3
          )
{
   setL1 (l1);
   setL2 (l2);
   setL3 (l3);
}


bool Comparison::isEmpty() const
{
  return   ls1.stripWhiteSpace().isEmpty()
        && ls2.stripWhiteSpace().isEmpty()
        && ls3.stripWhiteSpace().isEmpty();
}


void Comparison::clear()
{
   ls1 = "";
   ls2 = "";
   ls3 = "";
}


QDataStream & operator<< (QDataStream &s, const Comparison &comp)
{
  s << comp.ls1
    << comp.ls2
    << comp.ls3;
  return s;
}


QDataStream & operator>> (QDataStream &s, Comparison &comp)
{
  s >> comp.ls1
    >> comp.ls2
    >> comp.ls3;
  return s;
}


//=================================================================


Article::Article
          (const QString &fem_def, const QString &fem_indef,
           const QString &mal_def, const QString &mal_indef,
           const QString &nat_def, const QString &nat_indef
          )
{
   setFemale  (fem_def, fem_indef);
   setMale    (mal_def, mal_indef);
   setNatural (nat_def, nat_indef);
}


void Article::setFemale
  (const QString &def, const QString &indef)
{
   fem_def = def;
   fem_indef = indef;
}


void Article::setMale
  (const QString &def, const QString &indef)
{
   mal_def = def;
   mal_indef = indef;
}


void Article::setNatural
  (const QString &def, const QString &indef)
{
   nat_def = def;
   nat_indef = indef;
}


void Article::female
  (QString &def, QString &indef) const
{
   def = fem_def;
   indef = fem_indef;
}


void Article::male
  (QString &def, QString &indef) const
{
   def = mal_def;
   indef = mal_indef;
}


void Article::natural
  (QString &def, QString &indef) const
{
   def = nat_def;
   indef = nat_indef;
}


QDataStream & operator<< (QDataStream &s, const Article &art)
{
  s << art.fem_def
    << art.fem_indef
    << art.mal_def
    << art.mal_indef
    << art.nat_def
    << art.nat_indef;
  return s;
}


QDataStream & operator>> (QDataStream &s, Article &art)
{
  s >> art.fem_def
    >> art.fem_indef
    >> art.mal_def
    >> art.mal_indef
    >> art.nat_def
    >> art.nat_indef;
  return s;
}

//==============================================================


int Conjugation::numEntries() const
{
   return conjugations.size();
}


QString Conjugation::getName (const QString &abbrev)
{
   for (int i = 0; i < (int) numNames(); i++)
     if (names[i].abbrev == abbrev)
       return names[i].name;
   return "";
}


QString Conjugation::getName (int idx)
{
   if (idx >= numNames() )
     return "";

   return names[idx].name;
}


QString Conjugation::getAbbrev (const QString &name)
{
   for (int i = 0; i < (int) numNames(); i++)
     if (names[i].name == name)
       return names[i].abbrev;
   return "";
}


QString Conjugation::getAbbrev (int idx)
{
   if (idx >= numNames() )
     return "";

   return names[idx].abbrev;
}


int Conjugation::numNames()
{
   return sizeof(names) / sizeof(names[0]);
}


QString Conjugation::getType (int idx)
{
  if (idx >= (int) conjugations.size() )
    return "";

  return conjugations[idx].type;
}


void Conjugation::cleanUp ()
{
  for (int i = (int)conjugations.size()-1; i >= 0; i--) {
    const conjug_t *ctp = &conjugations[i];
    if (   ctp->pers1_sing.stripWhiteSpace().isEmpty()
        && ctp->pers2_sing.stripWhiteSpace().isEmpty()
        && ctp->pers3_m_sing.stripWhiteSpace().isEmpty()
        && ctp->pers3_f_sing.stripWhiteSpace().isEmpty()
        && ctp->pers3_n_sing.stripWhiteSpace().isEmpty()
        && ctp->pers1_plur.stripWhiteSpace().isEmpty()
        && ctp->pers2_plur.stripWhiteSpace().isEmpty()
        && ctp->pers3_m_plur.stripWhiteSpace().isEmpty()
        && ctp->pers3_f_plur.stripWhiteSpace().isEmpty()
        && ctp->pers3_n_plur.stripWhiteSpace().isEmpty()
       )
     conjugations.erase(&conjugations[i], &conjugations[i+1]);
  }
}


#define _GET_CON_(elem, type, default) \
   if (type != CONJ_PREFIX) \
     KASSERT(!getName(type).isEmpty(), KDEBUG_ERROR, 0, (QString)"getting unknown type: <"+type+">"); \
   \
   for (int i = 0; i < (int) conjugations.size(); i++) \
     if (conjugations[i].type == type) \
        return conjugations[i].elem; \
   return default;


bool Conjugation::pers3SingularCommon(const QString &type) const
{
  _GET_CON_(s3common, type, false);
}


bool Conjugation::pers3PluralCommon(const QString &type) const
{
  _GET_CON_(p3common, type, false);
}


QString Conjugation::pers1Singular
  (const QString &type) const
{
  _GET_CON_(pers1_sing, type, "");
}


QString Conjugation::pers2Singular
  (const QString &type) const
{
  _GET_CON_(pers2_sing, type, "");
}


QString Conjugation::pers3FemaleSingular
  (const QString &type) const
{
  _GET_CON_(pers3_f_sing, type, "");
}


QString Conjugation::pers3MaleSingular
  (const QString &type) const
{
  _GET_CON_(pers3_m_sing, type, "");
}


QString Conjugation::pers3NaturalSingular
  (const QString &type) const
{
  _GET_CON_(pers3_n_sing, type, "");
}


QString Conjugation::pers1Plural
  (const QString &type) const
{
  _GET_CON_(pers1_plur, type, "");
}


QString Conjugation::pers2Plural
  (const QString &type) const
{
  _GET_CON_(pers2_plur, type, "");
}


QString Conjugation::pers3FemalePlural
  (const QString &type) const
{
  _GET_CON_(pers3_f_plur, type, "");
}


QString Conjugation::pers3MalePlural
  (const QString &type) const
{
  _GET_CON_(pers3_m_plur, type, "");
}


QString Conjugation::pers3NaturalPlural
  (const QString &type) const
{
  _GET_CON_(pers3_n_plur, type, "");
}


#undef _GET_CON_


#define _SET_CON_(elem, type, str) \
   if (type != CONJ_PREFIX) \
     KASSERT(!getName(type).isEmpty(), KDEBUG_ERROR, 0, (QString)"setting unknown type: "+type); \
   \
   for (int i = 0; i < (int) conjugations.size(); i++) \
     if (conjugations[i].type == type) { \
       conjugations[i].elem = str; \
       return; \
     } \
   conjug_t ct; \
   ct.type = type; \
   ct.elem = str; \
   conjugations.push_back(ct);


void Conjugation::setPers3PluralCommon(const QString &type, bool f)
{
  _SET_CON_(p3common, type, f);
}


void Conjugation::setPers3SingularCommon(const QString &type, bool f)
{
  _SET_CON_(s3common, type, f);
}


void Conjugation::setPers1Singular
  (const QString &type, const QString &str)
{
  _SET_CON_(pers1_sing, type, str);
}


void Conjugation::setPers2Singular
  (const QString &type, const QString &str)
{
  _SET_CON_(pers2_sing, type, str);
}


void Conjugation::setPers3FemaleSingular
  (const QString &type, const QString &str)
{
  _SET_CON_(pers3_f_sing, type, str);
}


void Conjugation::setPers3MaleSingular
  (const QString &type, const QString &str)
{
  _SET_CON_(pers3_m_sing, type, str);
}


void Conjugation::setPers3NaturalSingular
  (const QString &type, const QString &str)
{
  _SET_CON_(pers3_n_sing, type, str);
}


void Conjugation::setPers1Plural
  (const QString &type, const QString &str)
{
  _SET_CON_(pers1_plur, type, str);
}


void Conjugation::setPers2Plural
  (const QString &type, const QString &str)
{
  _SET_CON_(pers2_plur, type, str);
}


void Conjugation::setPers3FemalePlural
  (const QString &type, const QString &str)
{
  _SET_CON_(pers3_f_plur, type, str);
}


void Conjugation::setPers3MalePlural
  (const QString &type, const QString &str)
{
  _SET_CON_(pers3_m_plur, type, str);
}


void Conjugation::setPers3NaturalPlural
  (const QString &type, const QString &str)
{
  _SET_CON_(pers3_n_plur, type, str);
}

#undef _SET_CON_


QDataStream & operator<< (QDataStream &s, const Conjugation &cm)
{
  s << (int) cm.conjugations.size();
  for (int i = 0; i < (int)cm.conjugations.size(); i++) {
    const Conjugation::conjug_t *ctp = &cm.conjugations[i];
    s << ctp->type
      << ctp->pers1_sing
      << ctp->pers2_sing
      << (int)ctp->s3common
      << ctp->pers3_m_sing
      << ctp->pers3_f_sing
      << ctp->pers3_n_sing
      << ctp->pers1_plur
      << ctp->pers2_plur
      << (int)ctp->p3common
      << ctp->pers3_m_plur
      << ctp->pers3_f_plur
      << ctp->pers3_n_plur;
  }
  return s;
}


QDataStream & operator>> (QDataStream &s, Conjugation &cm)
{
  int l;
  int  s3c,
       p3c;

  s >> l;
  for (int i = 0; i < l; i++) {
    Conjugation::conjug_t ct;
    s >> ct.type
      >> ct.pers1_sing
      >> ct.pers2_sing
      >> s3c
      >> ct.pers3_m_sing
      >> ct.pers3_f_sing
      >> ct.pers3_n_sing
      >> ct.pers1_plur
      >> ct.pers2_plur
      >> p3c
      >> ct.pers3_m_plur
      >> ct.pers3_f_plur
      >> ct.pers3_n_plur;
    ct.p3common = p3c;
    ct.s3common = s3c;
    cm.conjugations.push_back(ct);
  }
  return s;
}

