
/***************************************************************************
                 kvd_kvtbin.cpp  -  maintain a binary kvtml document
                             -------------------                                         
    begin                : Sat Sep 04 11:00:53 MET 1999
                                           
    copyright            : (C) 1999,2000 by Ewald Arnold
    email                : ewald@ewald-arnold.de

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <kapp.h>

#include "compat_2x.h"
#include "kvoctraindoc.h"
#include "resource.h"

#include <qdatastream.h>
#include <qmessagebox.h>

//                 major:8, minor:8 count:16 min-hib FF=pre
#define KVD_BIN_REV    0x00060001    // make sure bin file matches EXACTLY


bool kvoctrainDoc::saveTypeNameKvtBin (QDataStream &os)
{
  os << type_descr.size();

  if (type_descr.size() == 0)
    return true;

  for (int lfn = 0; lfn < (int) type_descr.size(); lfn++) {
    os << lfn+1;
    os << type_descr[lfn];
  }

  return os.device()->status() == IO_Ok;
}


bool kvoctrainDoc::loadTypeNameKvtBin (QDataStream &is)
{
  int l;
  QString s;

  type_descr.clear();

  is >> l;
  if (l == 0)
    return true;

  for (int i = 0; i < l; i++) {
    int no;
    is >> no;
    is >> s;
    type_descr.push_back (s);
  }

  return is.device()->status() == IO_Ok;
}


bool kvoctrainDoc::saveOptionsKvtBin (QDataStream &os)
{
  os << (int) sort_allowed;

  return os.device()->status() == IO_Ok;
}


bool kvoctrainDoc::saveLessonKvtBin (QDataStream &os)
{
  os << getSizeHint(-1);

  os << lesson_descr.size();

  if (lesson_descr.size() == 0)
    return true;

  for (int lfn = 0; lfn < (int) lesson_descr.size(); lfn++) {
    os << lfn+1;
    os << (int) (getCurrentLesson() == lfn+1);
    os << lesson_descr[lfn];
  }

  return os.device()->status() == IO_Ok;
}


bool kvoctrainDoc::saveArticleKvtBin (QDataStream &os)
{
  os << (int) articles.size();
  for (int i = 0; i < (int) articles.size(); i++)
    os << articles[i];
  return os.device()->status() == IO_Ok;
}


bool kvoctrainDoc::saveConjugKvtBin (QDataStream &os)
{
  os << (int) conjugations.size();
  for (int i = 0; i < (int) conjugations.size(); i++)
    os << conjugations[i];
  return os.device()->status() == IO_Ok;
}


bool kvoctrainDoc::loadOptionsKvtBin (QDataStream &is)
{
  int l;
  is >> l;
  sort_allowed = (bool) l;

  return is.device()->status() == IO_Ok;
}


bool kvoctrainDoc::loadLessonKvtBin (QDataStream &is)
{
  int l;
  QString s;

  lesson_descr.clear();

  is >> l;
  setSizeHint (-1, l);

  is >> l;
  if (l == 0)
    return true;

  for (int i = 0; i < l; i++) {
    int no;
    int isCurr;

    is >> no;
    is >> isCurr;
    is >> s;

    if (isCurr != 0 && no != 0)
      setCurrentLesson (no);

    lesson_descr.push_back (s);
  }

  return is.device()->status() == IO_Ok;
}


bool kvoctrainDoc::loadArticleKvtBin (QDataStream &is)
{
  int l;
  Article art;
  articles.clear();
  is >> l;
  for (int i = 0; i < l; i++) {
    is >> art;
    articles.push_back(art);
  }
  return is.device()->status() == IO_Ok;
}


bool kvoctrainDoc::loadConjugKvtBin (QDataStream &is)
{
  int l;
  Conjugation con;
  conjugations.clear();
  is >> l;
  for (int i = 0; i < l; i++) {
    is >> con;
    conjugations.push_back(con);
  }
  return is.device()->status() == IO_Ok;
}


bool kvoctrainDoc::saveToKvtBin (QDataStream& os, QString title)
{
  os << 'k' <<  'v' << 'o' << 'c';
  os << (int) KVD_BIN_REV;

  os << (QString) "kvoctrain" KVD_VERS_PREFIX VERSION;
  setTitle (title);
  os << getTitle();
  os << getAuthor();
  os << numLangs();
  os << numEntries();

  if (!saveLessonKvtBin (os))
    return false;

  if (!saveArticleKvtBin (os))
    return false;

  if (!saveConjugKvtBin (os))
    return false;

  if (!saveOptionsKvtBin (os))
    return false;

  if (!saveTypeNameKvtBin (os))
    return false;

  QString q_org, q_trans;
  getQueryLang (q_org, q_trans);

  QString s;
  os << getSizeHint (0);
  s = getOriginalIdent();
  if (s.isEmpty() )
    s = "original";
  os << s;
  QString id;
  if (s == q_org)
    id = KV_O;
  else if (s == q_trans)
    id = KV_T;
  else
    id = "--";
  os << id;

  for (int i = 1; i < (int) numLangs(); i++) {
    os << (int) getSizeHint (i);
    s =  getIdent(i);
    if (s.isEmpty() ) {
      s.setNum (i);
      s.insert (0, "translation ");
    }
    os << s;
    if (s == q_org)
      id = KV_O;
    else if (s == q_trans)
      id = KV_T;
    else
      id = "--";
    os << id;
  }

  int ent_no = 0;
  int ent_percent = (int) vocabulary.size () / 100;
  float f_ent_percent = (int) vocabulary.size () / 100.0;
  emit progressChanged(this, 0);

  os << (int) vocabulary.size();
  vector<kvoctrainExpr>::const_iterator first =  vocabulary.begin ();
  while (first != vocabulary.end ()) {

    ent_no++;
    if (ent_percent != 0 && (ent_no % ent_percent) == 0 )
      emit progressChanged(this, ent_no / f_ent_percent);

    os << *first;
    first++;
  }

  setModified (false);
  return os.device()->status() == IO_Ok;
}


bool kvoctrainDoc::loadFromKvtBin (QDataStream& is)
{
  langs.clear();
  vocabulary.clear();
  Q_INT8 k,v,o,c;
  int rev;

  is >> k >> v >> o >> c;
  is >> rev;
  if (k != 'k' || v != 'v' || o != 'o' || c != 'c' || rev != KVD_BIN_REV) {
    errorKvtBin ("invalid file format");
    return false;
  }

  is >> generator;
  int pos = generator.findRev (KVD_VERS_PREFIX);
  if (pos >= 0) {
    doc_version = generator;
    doc_version.remove (0, pos+2);
  }
  is >> doctitle;
  if (doctitle == "")
    doctitle = i18n("untitled");
  is >> author;

  is >> cols;
  is >> lines;

  if (!loadLessonKvtBin (is))
    return false;

  if (!loadArticleKvtBin (is))
    return false;

  if (!loadConjugKvtBin (is))
    return false;

  if (!loadOptionsKvtBin (is))
    return false;

  if (!loadTypeNameKvtBin (is))
    return false;

  int val;
  QString lang;
  QString query_id;
  is >> val;
  setSizeHint (0, val);
  is >> lang;
  if (lang.isEmpty() )
    lang = "original";
  langs.push_back(lang);

  QString q_org, q_trans;
  is >> query_id;
  if (query_id == KV_O)
    q_org = lang;

  if (query_id == KV_T)
    q_trans = lang;

  for (int i = 1; i < cols; i++) {
    is >> val;
    setSizeHint (i, val);

    is >> lang;
    if (lang.isEmpty() ) {
       lang.setNum (i);
       lang.insert (0, "translation ");
    }
    langs.push_back(lang);

    is >> query_id;
    if (query_id == KV_O)
      q_org = lang;
  
    if (query_id == KV_T)
      q_trans = lang;
  }

  setQueryLang (q_org, q_trans);

  int ent_percent = (int) lines / 100;
  float f_ent_percent = (int) lines / 100.0;
  emit progressChanged(this, 0);

  int l;
  is >> l;
  for (int i = 0; i < l; i++) {
    if (ent_percent != 0 && (i % ent_percent) == 0 ) {
      emit progressChanged(this, i / f_ent_percent);
    }

    kvoctrainExpr kve;
    is >> kve;
    vocabulary.push_back(kve);

  }

  setModified (false);
  return is.device()->status() == IO_Ok;
}


void kvoctrainDoc::errorKvtBin (const QString &text )
{
   unknown_elem = true;
   QApplication::setOverrideCursor( arrowCursor, true );
   QString s = QString(KVOCTRAIN_TITLE)+QString(i18n(": error in binary file"));
   QString msg = text;
   QMessageBox mb( s,
       msg,
       QMessageBox::Critical,
       QMessageBox::Abort | QMessageBox::Default,
       0,
       0);
   mb.exec();
   QApplication::restoreOverrideCursor();
}
