#include <stdio.h>
#include <time.h>

#include <qmsgbox.h>
#include <qtabdlg.h>

#include "BlockOut.h"
#include "BlockOut.moc"

#include "QtDraw3D.h"

static QtDraw3D draw3D;

const char *version="BlockOut v0.1\nby Markus Pietrek\n(mpietrek@init-ka.de)";

BlockOut::BlockOut (const char *name) : KTopLevelWidget (name)
{
	readProperties (kapp->getConfig());

	view=0;
	
	obstacle=0;

	gamePlay = new PerformGamePlay ();

	try
	{
		char filename [256];
		
		sprintf (filename, "%s/BlockOut/BlockOut.figures",(const char *) KApplication::kde_datadir());
		gamePlay->readFiguresFromFile (filename);
	}
	catch (const char *c)
	{
		fprintf (stderr, "%s\n", c);
		exit (1);
	}
		
	view = new GameViewWidget (this);
	CHECK_PTR (view);
	setView (view);

	view->setPerformGamePlay (gamePlay);

	draw3D.connectToWidget (view);

	newGame();
	Draw3D::setGlobalInstance (&draw3D);


	createMenu();
	createStatusBar();
	createToolBar();

	resize (600, 600);
}

BlockOut::~BlockOut()
{
}

void BlockOut::saveProperties()
{
	saveProperties (kapp->getConfig());
	statusBar->changeItem (klocale->translate ("Settings saved"), SB_WHAT_DOING);
}

void BlockOut::saveProperties (KConfig *config)
{
	config->setGroup ("Settings");
	config->writeEntry ("SizeX", sizeX);
	config->writeEntry ("SizeY", sizeY);
	config->writeEntry ("SizeZ", sizeZ);
	config->writeEntry ("StartingLevel", startingLevel);
}


void BlockOut::readProperties (KConfig *config)
{
	config->setGroup ("Settings");
	sizeX=config->readNumEntry ("sizeX", 5);
	sizeY=config->readNumEntry ("sizeY", 5);
	sizeZ=config->readNumEntry ("sizeZ", 10);
	startingLevel=config->readNumEntry ("StartingLevel", 1);
}

void BlockOut::createMenu()
{
	QPopupMenu *game = new QPopupMenu;
	CHECK_PTR (game);

	game->insertItem (klocale->translate ("&New"), this, SLOT(newGame()), CTRL+Key_N);
	game->insertItem (klocale->translate ("E&xit"), qApp, SLOT(quit()), CTRL+Key_Q);

	QPopupMenu *_settings = new QPopupMenu;
	CHECK_PTR (_settings);
	_settings->insertItem (klocale->translate ("&Settings"), this, SLOT(settings()));
	_settings->insertItem (klocale->translate ("S&ave settings"), this, SLOT(saveProperties()));

	QPopupMenu *help = kapp->getHelpMenu (true, version);

	menu = new KMenuBar (this);
	CHECK_PTR (menu);
	menu->insertItem (klocale->translate ("&File"), game);
	menu->insertItem (klocale->translate ("&Settings"), _settings);
	menu->insertSeparator();
	menu->insertItem (klocale->translate ("&Help"), help);

	setMenu (menu);
}

void BlockOut::createStatusBar()
{
	statusBar=new KStatusBar (this);
	CHECK_PTR (statusBar);
	setStatusBar (statusBar);
	
	statusBar->insertItem ("Points:          ", SB_POINTS);
	statusBar->insertItem ("Level:    ", SB_LEVEL);
	statusBar->insertItem ("                      ", SB_WHAT_DOING);
}

void BlockOut::createToolBar()
{
	firstToolBar=new KToolBar (this);
	CHECK_PTR (firstToolBar);
	addToolBar (firstToolBar);
}

void BlockOut::newLevel (int i)
{
	char tmp[30];
	sprintf (tmp,"%s: %2i",klocale->translate ("Level"), i);
	statusBar->changeItem (tmp, SB_LEVEL);
}

void BlockOut::newPoints (int i)
{
	char tmp[30];

	sprintf (tmp,"%s: %i",klocale->translate ("Points"), i);
	statusBar->changeItem (tmp, SB_POINTS);
}

void BlockOut::gameEnded()
{
	QMessageBox::information (this, "BlockOut", klocale->translate ("Game ended"));
}

void BlockOut::newGame()
{
	if (obstacle)
		delete obstacle;
	
	obstacle = new Obstacle (sizeX, sizeY, sizeZ );


	draw3D.setPerspective (0, obstacle->getSizeX(), 0, obstacle->getSizeY(), obstacle->getSizeZ()+0.6, 0.85, false);

	gamePlay->setObstacle (obstacle);
	view->setTimeLevel (startingLevel);

	view->newGame();


	connect (view, SIGNAL (updatePoints (int)), SLOT (newPoints (int)));
	connect (view, SIGNAL (newLevel (int)), SLOT (newLevel (int)));
	connect (view, SIGNAL (gameEnded()), SLOT (gameEnded()));
}

void BlockOut::settings()
{
	QTabDialog *tabDialog=new QTabDialog (0,"", true);
	
	Settings *settings=new Settings (sizeX, sizeY, sizeZ, startingLevel, tabDialog);
	tabDialog->addTab (settings, klocale->translate ("Gameplay"));
	tabDialog->resize (350, 200);
	tabDialog->setOKButton();
	tabDialog->setCancelButton();
	if (tabDialog->exec()==QDialog::Accepted)
	{
		sizeX=settings->getSizeX();
		sizeY=settings->getSizeY();
		sizeZ=settings->getSizeZ();
		startingLevel=settings->getStartingLevel();
	}
	
	delete tabDialog;
}

int main (int argc, char *argv[])
{
	srand (time (NULL));

	KApplication app (argc, argv);
	BlockOut mainFrame ("BlockOut");

	app.setMainWidget (&mainFrame);
	mainFrame.show();
	return app.exec();
}

// Local Variables:
// compile-command: "make BlockOut"
// End:
