// plumber.C
//
// This program is free software. See the file COPYING for details.
// Author: Torsten Klein, 1998
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/types.h>

#include <kapp.h>
#include <kstring.h>
#include <qpainter.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qkeycode.h>
#include <kmsgbox.h>

#include "kplumber.h"

#include "pipe0000.xpm"
#include "pipe0001.xpm"
#include "pipe0010.xpm"
#include "pipe0011.xpm"
#include "pipe0100.xpm"
#include "pipe0101.xpm"
#include "pipe0110.xpm"
#include "pipe0111.xpm"
#include "pipe1000.xpm"
#include "pipe1001.xpm"
#include "pipe1010.xpm"
#include "pipe1011.xpm"
#include "pipe1100.xpm"
#include "pipe1101.xpm"
#include "pipe1110.xpm"
#include "pipe1111.xpm"
#include "icon.xpm"

#define PLUMBER_VERSION "v1.1"

#define TILE_W 32
#define TILE_H 32
#define DEFAULT_COLS 4
#define DEFAULT_ROWS 4

// state bits
#define DIR_UP 1
#define DIR_RIGHT 2
#define DIR_DOWN 4
#define DIR_LEFT 8

KPlumber::KPlumber(QWidget *parent)
    : QTableView(parent),
      states(0),
      playing(FALSE),
      marked(FALSE)
{
    setCellWidth(TILE_W);
    setCellHeight(TILE_H);
    setNumCols(DEFAULT_COLS);
    setNumRows(DEFAULT_ROWS);
    setTableFlags(Tbl_autoScrollBars | Tbl_smoothScrolling);
    make_table();
    char **xpms[] = {
	pipe0000_xpm, pipe0001_xpm, pipe0010_xpm, pipe0011_xpm,
	pipe0100_xpm, pipe0101_xpm, pipe0110_xpm, pipe0111_xpm,
	pipe1000_xpm, pipe1001_xpm, pipe1010_xpm, pipe1011_xpm,
	pipe1100_xpm, pipe1101_xpm, pipe1110_xpm, pipe1111_xpm
    };
    for(int i = 0; i < 16; i++)
	pm[i] = (const char**)xpms[i];
}

void KPlumber::make_table()
{
    if(states)
	delete states;
    states = new char[numCols() * numRows()];
    for(int i = 0; i < numCols(); i++)
	for(int j = 0; j < numRows(); j++)
	    state(j, i) = 0;
    endpairs = 0;
}

void KPlumber::randomize_table()
{
    for(int row = 0; row < numRows(); row++) {
	for(int col = 0; col < numCols() - 1; col++) {
	    int r = (rand() >> 10) & 1;		// low-order bits can be bad
	    endpairs += r;
	    if(r) {
		state(row, col) |= DIR_RIGHT;
		state(row, col + 1) |= DIR_LEFT;
	    }
	}
	if(row < numRows() - 1) 
	    for(int col = 0; col < numCols(); col++) {
		int r = (rand() >> 10) & 1;
		endpairs += r;
		if(r) {
		    state(row, col) |= DIR_DOWN;
		    state(row + 1, col) |= DIR_UP;
		}
	    }

    }
    // now spin them around a bit
    for(int row = 0; row < numRows(); row++)
	for(int col = 0; col < numCols(); col++) {
	    int r = (rand() >> 7) & 3; 		// 0..3
	    int s = state(row, col);
	    state(row, col) = ((s << r) | (s >> (4 - r))) & 0xf;
	}
    // and count the connected pairs in the spun configuration
    connpairs = 0;
    for(int row = 0; row < numRows(); row++)
	for(int col = 0; col < numCols(); col++)
	    connpairs += numconns(row, col);
    connpairs /= 2;		// each connection was counted twice
}

// return number of complete connections of a cell (0-4)
int KPlumber::numconns(int row, int col)
{
    int n = 0;
    int s = state(row, col);
    if(col > 0 && s & DIR_LEFT && state(row, col - 1) & DIR_RIGHT)
	n++;
    if(col < numCols() - 1 && s & DIR_RIGHT && state(row, col + 1) & DIR_LEFT)
	n++;
    if(row > 0 && s & DIR_UP && state(row - 1, col) & DIR_DOWN)
	n++;
    if(row < numRows() - 1 && s & DIR_DOWN && state(row + 1, col) & DIR_UP)
	n++;
    return n;
}

void KPlumber::paintCell(QPainter *p, int row, int col)
{
    p->drawPixmap(0, 0, pm[state(row, col)]);
}

void KPlumber::mousePressEvent(QMouseEvent *e)
{
    int row = findRow(e->pos().y());
    int col = findCol(e->pos().x());
    if(!playing || col == -1 || row == -1)
	return;
    if(marked) {
	repaint();
	marked = FALSE;
    }
    int s = state(row, col);
    int before = numconns(row, col);
    if(e->button() == LeftButton)
	state(row, col) = ((s >> 1) | (s << 3)) & 0xf;
    else if(e->button() == RightButton)
	state(row, col) = ((s << 1) | (s >> 3)) & 0xf;
    else
	return;
    updateCell(row, col);
    connpairs += numconns(row, col) - before;
    emit moved();
    if(connpairs == endpairs) {
	playing = FALSE;
	emit won();
    }
}

QSize KPlumber::sizeHint() const
{
    return QSize(numCols() * cellWidth(), numRows() * cellHeight());
}

void KPlumber::setSize(int rows, int cols)
{
    if(rows != numRows() || cols != numCols()) {
	setAutoUpdate(FALSE);
	setNumRows(rows);
	setNumCols(cols);
	setAutoUpdate(TRUE);
	make_table();
	adjustSize();
	setMaximumSize(sizeHint());
	repaint();
	playing = FALSE;
    }
}

void KPlumber::restart()
{
    make_table();
    randomize_table();
    repaint();
    playing = TRUE;
}

void KPlumber::mark(QPainter *p, int row, int col, int dir)
{
    int x, y;
    if(colXPos(col, &x) && rowYPos(row, &y)) {
	switch(dir) {
	case DIR_UP:
	    x += TILE_W / 2;
	    break;
	case DIR_DOWN:
	    x += TILE_W / 2;
	    y += TILE_H;
	    break;
	case DIR_LEFT:
	    y += TILE_H / 2;
	    break;
	case DIR_RIGHT:
	    y += TILE_H / 2;
	    x += TILE_W;
	    break;
	}
	int r = TILE_W / 3;
	p->drawEllipse(x - r, y - r, 2 * r, 2 * r);
    }
}

void KPlumber::markEnds()
{
    QPainter p(this);
    p.setPen(QPen(yellow, 3));
    for(int row = 0; row < numRows(); row++)
	for(int col = 0; col < numCols(); col++) {
	    if(state(row, col) & DIR_UP
	       && (row == 0 || !(state(row - 1, col) & DIR_DOWN)))
		mark(&p, row, col, DIR_UP);
	    if(state(row, col) & DIR_DOWN
	       && (row == numRows() - 1 || !(state(row + 1, col) & DIR_UP)))
		mark(&p, row, col, DIR_DOWN);
	    if(state(row, col) & DIR_LEFT
	       && (col == 0 || !(state(row, col - 1) & DIR_RIGHT)))
		mark(&p, row, col, DIR_LEFT);
	    if(state(row, col) & DIR_RIGHT
	       && (col == numCols() - 1 || !(state(row, col + 1) & DIR_LEFT)))
		mark(&p, row, col, DIR_RIGHT);
	}
    p.end();
    marked = TRUE;
}

Game::Game(QWidget *parent)
    : QWidget(parent)
{
    QPalette lcdpal;
    lcdpal.setNormal(QColorGroup(red, black, red, black, black, red, black));

    QVBoxLayout *tl = new QVBoxLayout(this, 8);

    KMenuBar* mb = new KMenuBar(this);

//    mb->insertSeparator();
//    tl->
//    setMenu(mb);

    QHBoxLayout *hl = new QHBoxLayout();
    tl->addLayout(hl);

    QLabel *tlbl = new QLabel("Time", this);
    tlbl->setFixedSize(tlbl->sizeHint());
    hl->addWidget(tlbl);

    timedisp = new QLCDNumber(4, this);
    timedisp->setSegmentStyle(QLCDNumber::Flat);
    timedisp->setPalette(lcdpal);
    timedisp->setFixedSize(100, 32);
    hl->addWidget(timedisp);

    hl->addStretch();

    restart = new QPushButton("Start", this);
    restart->setFixedSize(restart->sizeHint());
    hl->addWidget(restart);

    hl->addStretch();

    movedisp = new QLCDNumber(4, this);
    movedisp->setSegmentStyle(QLCDNumber::Flat);
    movedisp->setPalette(lcdpal);
    movedisp->setFixedSize(100, 32);
    hl->addWidget(movedisp);

    QLabel *mlbl = new QLabel("Moves", this);
    mlbl->setFixedSize(mlbl->sizeHint());
    hl->addWidget(mlbl);

    gb = new KPlumber(this);
    tl->addWidget(gb, 1);

    tl->activate();
    
    game_menu = new QPopupMenu;
    mb->insertItem("File", game_menu);
    game_menu->insertItem("New Game", this, SLOT(start()), ALT + Key_N);
    game_menu->insertItem("High Scores", this, SLOT(show_hiscores()));
    game_menu->insertSeparator();
    game_menu->connectItem(game_menu->insertItem("Mark Pipe Ends", MENU_MARK),
			   gb, SLOT(markEnds()));
    game_menu->setItemEnabled(MENU_MARK, FALSE);
    game_menu->setAccel(ALT + Key_M, MENU_MARK);
    game_menu->insertSeparator();
    game_menu->insertItem("Quit", kapp, SLOT(quit()), ALT + Key_Q);
    
    option_menu = new QPopupMenu;
    mb->insertItem("Options", option_menu);
    option_menu->setCheckable(TRUE);

    option_menu->connectItem(option_menu->insertItem("Small (4)", MENU_SMALL),
			     this, SLOT(set_small()));
    option_menu->connectItem(option_menu->insertItem("Medium (8)",
						     MENU_MEDIUM),
			     this, SLOT(set_medium()));
    option_menu->connectItem(option_menu->insertItem("Large (16)",
						     MENU_LARGE),
			     this, SLOT(set_large()));
    option_menu->insertSeparator();
    option_menu->connectItem(option_menu->insertItem("Custom Size...",
						     MENU_CUSTOM),
			     this, SLOT(set_custom()));

    QPopupMenu *help_menu = new QPopupMenu;
    mb->insertSeparator();
//    mb->insertItem("Help", help_menu);
    help_menu = kapp->getHelpMenu(true, i18n("KPlumber Version 1.1\n\n"
                                             "(c) 1998 The BerLinuX\n\n"
                                             "thanx to Kellina. U are the 1 4 ever"));
    mb->insertItem("Help", help_menu);

  

    connect(restart, SIGNAL(clicked()), this, SLOT(start()));
    connect(gb, SIGNAL(won()), this, SLOT(celebrate()));
    connect(gb, SIGNAL(moved()), this, SLOT(step()));

    setSize(DEFAULT_ROWS, DEFAULT_COLS);
    setIcon((const char**)icon_xpm);
    hiscorewindow = 0;
}

void Game::start()
{
    moves = seconds = 0;
    movedisp->display(moves);
    timedisp->display(seconds);
    killTimers();
    startTimer(1000);
    game_menu->setItemEnabled(MENU_MARK, TRUE);
    gb->restart();
}

void Game::celebrate()
{
    KApplication::beep();
    killTimers();
    game_menu->setItemEnabled(MENU_MARK, FALSE);
    if(!hiscorewindow)
	hiscorewindow = new HiScoreWindow();
    hiscorewindow->newScore(seconds, moves, gb->nrows(), gb->ncols());
}

void Game::step()
{
    moves++;
    movedisp->display(moves);
}

void Game::timerEvent(QTimerEvent *)
{
    seconds++;
    timedisp->display(seconds);
}

void Game::set_small()
{
    setSize(4, 4);
}

void Game::set_medium()
{
    setSize(8, 8);
}

void Game::set_large()
{
    setSize(16, 16);
}

void Game::set_custom()
{
    SizeDialog sd(this, gb->nrows(), gb->ncols());
    if(sd.exec()) {
	int r = sd.rows(), c = sd.cols();
	if(r >= 1 && c >= 1 && r / c < 4 && c / r < 4)
	    setSize(r, c);
    }
}

void Game::setSize(int rows, int cols)
{
    int id;
    if(rows == 4 && cols == 4)
	id = MENU_SMALL;
    else if(rows == 8 && cols == 8)
	id = MENU_MEDIUM;
    else if(rows == 16 && cols == 16)
	id = MENU_LARGE;
    else
	id = MENU_CUSTOM;
    for(int i = MENU_SMALL; i <= MENU_CUSTOM; i++)
	option_menu->setItemChecked(i, id == i);
    killTimers();
    moves = seconds = 0;
    movedisp->display(moves);
    timedisp->display(seconds);
    game_menu->setItemEnabled(MENU_MARK, FALSE);
    gb->setSize(rows, cols);
    int w = QMIN(TILE_W * cols + 16, kapp->desktop()->width() - 32);
    int h = QMIN(TILE_H * rows + 100, kapp->desktop()->height() - 32);
    resize(w, h);
}

void Game::show_hiscores()
{
    if(!hiscorewindow)
	hiscorewindow = new HiScoreWindow();
    hiscorewindow->readScores();
    hiscorewindow->show();
    hiscorewindow->raise();
}


SizeDialog::SizeDialog(QWidget *parent, int default_rows, int default_cols)
    : QDialog(parent, 0, TRUE)
{
    QVBoxLayout *tl = new QVBoxLayout(this, 16);
    QHBoxLayout *rl = new QHBoxLayout;
    tl->addLayout(rl);

    QLabel *rlbl = new QLabel("Rows", this);
    rlbl->setFixedSize(rlbl->sizeHint());
    rl->addWidget(rlbl);
    rl->addStretch(1);

    rows_ed = new KLined(this);
    rows_ed->setFixedSize(40, rows_ed->sizeHint().height());
    QString s;
    s.setNum(default_rows);
    rows_ed->setText(s);
    rows_ed->setFocus();
    rl->addWidget(rows_ed);

    QHBoxLayout *cl = new QHBoxLayout();
    tl->addLayout(cl);

    QLabel *clbl = new QLabel("Columns", this);
    clbl->setFixedSize(clbl->sizeHint());
    cl->addWidget(clbl);
    cl->addStretch(1);

    cols_ed = new KLined(this);
    cols_ed->setFixedSize(40, cols_ed->sizeHint().height());
    s.setNum(default_cols);
    cols_ed->setText(s);
    cl->addWidget(cols_ed);

    tl->addStretch(1);

    QHBoxLayout *bl = new QHBoxLayout;
    tl->addLayout(bl);

    QPushButton *ok = new QPushButton("OK", this);
    ok->setDefault(TRUE);
    ok->setFixedSize(64, 28);
    bl->addWidget(ok);
    bl->addStretch(1);
    QPushButton *cancel = new QPushButton("Cancel", this);
    cancel->setFixedSize(64, 28);
    bl->addWidget(cancel);

    tl->freeze();

    connect(ok, SIGNAL(clicked()), SLOT(accept()));
    connect(cancel, SIGNAL(clicked()), SLOT(reject()));
}

int SizeDialog::rows()
{
    return atoi(rows_ed->text());
}

int SizeDialog::cols()
{
    return atoi(cols_ed->text());
}

HiScoreWindow::HiScoreWindow(QWidget *parent)
    : QWidget(parent)
{
    setCaption("Plumber High Scores");
    char *cat_name[categories] = {"44", "88", "1616"};

    QGridLayout *gl = new QGridLayout(this, 1 + categories + 1, 5, 16, 8);

    QLabel *fl = new QLabel("Fastest", this);
    fl->setFixedSize(fl->sizeHint());
    gl->addMultiCellWidget(fl, 0, 0, 1, 2);
    QLabel *ll = new QLabel("Least Moves", this);
    ll->setFixedSize(ll->sizeHint());
    gl->addMultiCellWidget(ll, 0, 0, 3, 4);

    gl->addColSpacing(0, 64);
    gl->addColSpacing(1, 120);
    gl->addColSpacing(2, 60);
    gl->addColSpacing(3, 120);
    gl->addColSpacing(4, 60);
    for(int i = 0; i < categories; i++) {
	QLabel *l = new QLabel(cat_name[i], this);
	l->setFixedHeight(l->sizeHint().height());
	gl->addWidget(l, 1 + i, 0);
	
	lbl_fast_name[i] = new QLabel(this);
	lbl_fast_name[i]->setBackgroundColor(white);
	gl->addWidget(lbl_fast_name[i], 1 + i, 1);
	lbl_fast_time[i] = new QLabel(this);
	lbl_fast_time[i]->setAlignment(AlignRight | AlignVCenter);
	lbl_fast_time[i]->setBackgroundColor(white);
	gl->addWidget(lbl_fast_time[i], 1 + i, 2);
	lbl_least_name[i] = new QLabel(this);
	lbl_least_name[i]->setBackgroundColor(white);
	gl->addWidget(lbl_least_name[i], 1 + i, 3);
	lbl_least_moves[i] = new QLabel(this);
	lbl_least_moves[i]->setAlignment(AlignRight | AlignVCenter);
	lbl_least_moves[i]->setBackgroundColor(white);
	gl->addWidget(lbl_least_moves[i], 1 + i, 4);
    }
    QLabel *la = new QLabel("Largest", this);
    la->setFixedHeight(la->sizeHint().height());
    gl->addWidget(la, categories + 1, 0);
    lbl_largest = new QLabel(this);
    lbl_largest->setBackgroundColor(white);
    gl->addMultiCellWidget(lbl_largest, categories + 1, categories + 1, 1, 4);
    gl->activate();

    struct passwd *p = getpwuid(getuid());
    strtok(p->pw_gecos, ", ");	// use first name
    last_name = p->pw_gecos;
}

QString HiScoreWindow::getline(FILE *f)
{
    static char buf[512];
    fgets(buf, sizeof(buf), f);
    strtok(buf, "\n");
    return buf;
}

int HiScoreWindow::getint(FILE *f)
{
    int x;
    fscanf(f, "%d\n", &x);
    return x;
}


#define STRINGIZE(x) STRING2(x)
#define STRING2(x) #x

QString HiScoreWindow::scorefile()
{
#ifdef HISCOREFILE
    return STRINGIZE(HISCOREFILE);
#else
    QString file;
    file.sprintf("%s/.plumberscores", getenv("HOME"));
    return file;
#endif
}

void HiScoreWindow::readScores()
{
    FILE *f = fopen(scorefile(), "r");
    if(!f) {
	// generate default table
	for(int i = 0; i < categories; i++) {
	    fast_name[i] = "Nobody";
	    fast_time[i] = 999999;
	    least_name[i] = "Nobody";
	    least_moves[i] = 999999;
	}
	largest_name = "Nobody";
	largest_rows = largest_cols = 1;
	largest_time = largest_moves = 999999;
    } else {
	for(int i = 0; i < categories; i++) {
	    fast_name[i] = getline(f);
	    fast_time[i] = getint(f);
	    least_name[i] = getline(f);
	    least_moves[i] = getint(f);
	}
	largest_name = getline(f);
	largest_rows = getint(f);
	largest_cols = getint(f);
	largest_time = getint(f);
	largest_moves = getint(f);
	fclose(f);
    }
    setScores();
}

void HiScoreWindow::writeScores()
{
    int fd = creat(scorefile(), 0600);
    if(fd < 0) {
	fprintf(stderr, "Plumber: Can't write to high score file %s\n",
		(const char *)scorefile());
	return;
    }
    FILE *f = fdopen(fd, "w");
    for(int i = 0; i < categories; i++)
	fprintf(f, "%s\n%d\n%s\n%d\n",
		(const char*)fast_name[i], fast_time[i],
		(const char*)least_name[i], least_moves[i]);
    fprintf(f, "%s\n%d\n%d\n%d\n%d\n",
	    (const char*)largest_name, largest_rows, largest_cols,
	    largest_time, largest_moves);
    fclose(f);
}

void HiScoreWindow::newScore(int time, int moves, int rows, int cols)
{
    readScores();
    int size[categories] = {4, 8, 16};

    int i;
    for(i = 0; i < categories; i++) {
	if(rows == size[i] && cols == size[i]
	   && (time < fast_time[i] || moves < least_moves[i]))
	    break;
    }
    if(i < categories || rows * cols > largest_rows * largest_cols) {
	NewScoreDialog nd(kapp->mainWidget(), last_name);
	nd.exec();
	last_name = nd.playerName();
	for(i = 0; i < categories; i++)
	    if(rows == size[i] && cols == size[i]) {
		if(time < fast_time[i]) {
		    fast_name[i] = nd.playerName();
		    fast_time[i] = time;
		}
		if(moves < least_moves[i]) {
		    least_name[i] = nd.playerName();
		    least_moves[i] = moves;
		}
	    }
	if(rows * cols > largest_rows * largest_cols) {
	    largest_name = nd.playerName();
	    largest_rows = rows;
	    largest_cols = cols;
	    largest_time = time;
	    largest_moves = moves;
	}
	writeScores();
	setScores();
	show();
	raise();
    }
}

void HiScoreWindow::setScores()
{
    QString s;
    for(int i = 0; i < categories; i++) {
	lbl_fast_name[i]->setText(fast_name[i]);
	s.sprintf("%d s", fast_time[i]);
	lbl_fast_time[i]->setText(s);
	lbl_least_name[i]->setText(least_name[i]);
	s.setNum(least_moves[i]);
	lbl_least_moves[i]->setText(s);
    }
    s.sprintf("%s   (%d%d, %d s, %d moves)",
	      (const char*)largest_name, largest_rows, largest_cols,
	      largest_time, largest_moves);
    lbl_largest->setText(s);
}

NewScoreDialog::NewScoreDialog(QWidget *parent, QString default_name)
    : QDialog(parent, 0, TRUE)
{
    setCaption("Hiscore!");
    QVBoxLayout *tl = new QVBoxLayout(this, 16);
    QLabel *l1 = new QLabel("You just got a Great Score!", this);
    l1->setFixedSize(l1->sizeHint());
    tl->addWidget(l1);

    QHBoxLayout *hl = new QHBoxLayout;
    tl->addLayout(hl);

    QLabel *l2 = new QLabel("Your name:", this);
    l2->setFixedSize(l2->sizeHint());
    hl->addWidget(l2);

    name_ed = new QLineEdit(this);
    name_ed->setFixedSize(120, name_ed->sizeHint().height());
    name_ed->setText(default_name);
    name_ed->selectAll();
    hl->addWidget(name_ed);

    QPushButton *ok = new QPushButton("OK", this);
    ok->setDefault(TRUE);
    ok->setFixedSize(64, 28);
    tl->addWidget(ok);
    tl->freeze();
    connect(ok, SIGNAL(clicked()), SLOT(accept()));
}

QString NewScoreDialog::playerName()
{
    QString s = name_ed->text();
    s.stripWhiteSpace();
    if(s.isEmpty())
	return "Anonymous";
    else
	return s;
}

int main(int argc, char **argv)
{
    srand(time(0));
    KApplication app(argc, argv);
    app.setFont("Helvetica", 12);
    Game g;
    app.setMainWidget(&g);
    g.show();
    return app.exec();
}

