/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** graph.cpp
*/

#include "graph.h"
#include <stdio.h>

knode::knode(graph *g,Vector3 v)
{
	edgelist.empty();
	edgelist.setUnique(1);

	flags = 0;

	if((parent = g))
		parent->add(this);
	
	v3 = v;

//	printf("created knode <%g,%g,%g>\n",v3[0],v3[1],v3[2]);
}

knode::knode(graph *g,knode *k)
{
	edgelist.empty();
	edgelist.setUnique(1);

	flags = k->flags;

	if((parent = g))
		parent->add(this);
	
	v3 = k->vector();

//	printf("created knode <%g,%g,%g>\n",v3[0],v3[1],v3[2]);
}

knode::~knode()
{
	while(!edgelist.isEmpty())
		delete edgelist[0];

	if(parent) parent->remove(this);

//	printf("deleted knode <%g,%g,%g>\n",v3[0],v3[1],v3[2]);
}

const char	*knode::id()
{
	static char	idstr[50];

	sprintf(idstr,"<%g,%g,%g>",v3[0],v3[1],v3[2]);

	return idstr;
}

Vector3	knode::vector()
{
	return v3;
}

void	knode::clearFlags(int f)
{
	flags = f;
}

void	knode::add(edge *e)
{
	edgelist += e;
}

edge	*knode::getEdge(int i)
{
	if(i >= edgelist.length())
		return 0;

	return edgelist[i];
}

int	knode::numEdges()
{
	return edgelist.length();
}

void	knode::removeEdge(edge *t)
{
	int	i;

	i = edgelist.find(t);
	if(i >= 0)
	{
//		printf("removed edge %i from knode\n",t->id());
		edgelist.deleteAt(i);
	}
}

void	knode::replace(knode *k)
{
	int	i;

	for(i = k->numEdges() - 1;i >= 0;i--)
	{
		k->getEdge(i)->replace(k,this);
	}
}

int	knode::connected(knode *k)
{
	int	i;

	for(i = 0;i < k->numEdges();i++)
	{
		if(this == k->getEdge(i)->knode1())
			return 1;
		if(this == k->getEdge(i)->knode1())
			return 1;
	}

	return 0;
}

#define	MIN(a,b)	(a < b?a:b)
#define	MAX(a,b)	(a > b?a:b)

int	knode::distance(knode *k)
{
	int	i,d;


	if(this == k)
		return 0;

	if(flags == 2)
		return 2000000000;

	flags = 2;

	d = 2000000000;

	for(i = 0;i < edgelist.length();i++)
	{
		if(edgelist[i]->knode1() != this)
			d = MIN(d,edgelist[i]->knode1()->distance(k) + 1);
		if(edgelist[i]->knode2() != this)
			d = MIN(d,edgelist[i]->knode2()->distance(k) + 1);
	}

//	printf("knode %i distance to knode %i is %i\n",id(),k->id(),d);

	return MIN(d,2000000000);	
}

edge::edge(graph *g,knode *kp1,knode *kp2,int id)
{
	k1 = kp1;
	k2 = kp2;
	flags = 0;
	num_id = id;

	k1->add(this);
	k2->add(this);

	if((parent = g))
		parent->add(this);

	printf("created edge %i between ",num_id);
	printf("knode %s and ",k1->id());
	printf("knode %s\n",k2->id());
}

edge::~edge()
{
	if(k1) k1->removeEdge(this);
	if(k2) k2->removeEdge(this);

	if(parent) parent->remove(this);

//	printf("deleted edge %i\n",num_id);
}

knode	*edge::knode1()
{
	return k1;
}

knode	*edge::knode2()
{
	return k2;
}

void	edge::replace(knode *kold,knode *knew)
{
//	printf("Replace %p by %p\n",kold,knew);
	if(knew == 0)
	{
		printf("Attention! Try to replace by NULL.\n");
		return;
	}

	if(k1 == kold)
	{
		kold->removeEdge(this);
		knew->add(this);
		k1 = knew;
	}
	if(k2 == kold)
	{
		kold->removeEdge(this);
		knew->add(this);
		k2 = knew;
	}
}

int	edge::id()
{
	return num_id;
}

// return 0 if is close, else error
int	edge::closed()
{
	// the edge is close, if the edge is part of exactly two triangles
	// if the graph is small (less than 14 knodes) the edges are not
	// close if the number of triangles is less than two
	int		i,t,n;

	// count the triangles

	n = 0;
	for(i = 0;i < k2->numEdges();i++)
	{
		if(this == k2->getEdge(i))
			continue;
		for(t = 0;t < k2->getEdge(i)->knode1()->numEdges();t++)
		{
			if(this == k2->getEdge(i)->knode1()->getEdge(t))
				continue;
			if(k1 == k2->getEdge(i)->knode1()->getEdge(t)->knode1())
				n++;
			if(k1 == k2->getEdge(i)->knode1()->getEdge(t)->knode2())
				n++;
		}
		for(t = 0;t < k2->getEdge(i)->knode2()->numEdges();t++)
		{
			if(this == k2->getEdge(i)->knode2()->getEdge(t))
				continue;
			if(k1 == k2->getEdge(i)->knode2()->getEdge(t)->knode1())
				n++;
			if(k1 == k2->getEdge(i)->knode2()->getEdge(t)->knode2())
				n++;
		}
	}

	return (n - 2);
}

int	edge::unique()
{
	int	i;

	for(i = 0;i < k1->numEdges();i++)
	{
		if(this != k1->getEdge(i))
		{
			if(k1 == k1->getEdge(i)->knode1() && k2 == k1->getEdge(i)->knode2())
				return 0;
			if(k2 == k1->getEdge(i)->knode1() && k1 == k1->getEdge(i)->knode2())
				return 0;
		}
	}
	for(i = 0;i < k2->numEdges();i++)
	{
		if(this != k2->getEdge(i))
		{
			if(k1 == k2->getEdge(i)->knode1() && k2 == k2->getEdge(i)->knode2())
				return 0;
			if(k2 == k2->getEdge(i)->knode1() && k1 == k2->getEdge(i)->knode2())
				return 0;
		}
	}

	return 1;
}

edge	*edge::swap()
{
	knode*	k;

	k = k1;
	k1 = k2;
	k2 = k;

	return this;
}

graph::graph()
{
	knodelist.setUnique(1);
	edgelist.setUnique(1);
}

graph::~graph()
{
	while(!edgelist.isEmpty())
	{
		delete edgelist[0];
	}

	while(!knodelist.isEmpty())
	{
		delete knodelist[0];
	}
}

int	graph::add(knode *k)
{
	int		i;

	if(!k)
		return 0;
	if(knodelist.find(k) >= 0)
		return 0;

	if(k->parent != this)
	{
		knodelist += new knode(this,k);
	}
	else
	{
		knodelist += k;

		for(i = 0;i < k->numEdges();i++)
		{
			add(k->getEdge(i));
		}
	}

	return 0;
}

int	graph::add(edge *e)
{
	if(!e)
		return 0;
	if(edgelist.find(e) >= 0)
		return 0;

	if(e->parent != this)
	{
		edge	*t;

		edgelist += t = new edge(this,
					new knode(this,e->knode1()),
					new knode(this,e->knode2()),
					e->id());

		add(t->knode1());
		add(t->knode2());
	}
	else
	{
		edgelist += e;

		add(e->knode1());
		add(e->knode2());
	}


	return 0;
}

int	graph::remove(knode *k)
{
	k->parent = 0;

	while(k->numEdges())
	{
		remove(k->getEdge(0));
	}

	knodelist.deleteAt(knodelist.find(k));

	return 0;
}

int	graph::remove(edge *e)
{
	e->parent = 0;

	edgelist.deleteAt(edgelist.find(e));

	if(e->k1) e->k1->removeEdge(e);
	if(e->k2) e->k2->removeEdge(e);
	
	return 0;
}

int	graph::numKnodes()
{
	return knodelist.length();
}

int	graph::numEdges()
{
	return edgelist.length();
}

knode	*graph::getKnode(int i)
{
	return knodelist[i];
}

edge	*graph::getEdge(int i)
{
	return edgelist[i];
}

int	graph::distance(knode *k1,knode *k2)
{
	int		i,r1,r2;

	for(i = 0;i < knodelist.length();i++)
	{
		knodelist[i]->clearFlags();
	}

	r1 = k1->distance(k2);

	for(i = 0;i < knodelist.length();i++)
	{
		knodelist[i]->clearFlags();
	}

	r2 = k2->distance(k1);

	i = MIN(r1,r2);

//	printf("distance between %i %i %i\n",k1->id(),k2->id(),i);

	if(i == 2000000000)
		return -1;

	return i;
}

int	graph::isClose(list<edge*> *l,int grade)
{
	int		i,n;

	if(l)
		l->empty();

	removeDoubleKnodes();
	removeDoubleEdges();

	for(i = 0;i < edgelist.length();i++)
	{
		if((n = edgelist[i]->closed()) < grade)
		{
			if(l)
				l->append(edgelist[i]);
			else
				return 0;
		}
//		printf("Edge %i is %s close (%i)\n",edgelist[i]->id(),(n?"not":""),n);
	}

	if(l)
		return l->isEmpty();
	else
		return 1;
}

list<edge*>	graph::getOpenPolygone(list<edge*> ncelist)
{
// 1. take the first edge
// 2. take the second edge
// 3. take the next edge closest in the plane discribed by 1. + 2.
// 4. repeate 3. until the polygone is close or there is no match
// 5. is the polygone not close return an empty one

	list<edge*>	rlist;
	int		i,t;
	edge		*best;

	// 1. take the first edge
	rlist += ncelist[0];

	// 2. take the second edge
	for(i = 1;i < ncelist.length();i++)
	{
		if(rlist[0]->knode2() == ncelist[i]->knode1())
		{
			rlist += ncelist[i];
			break;
		}
		if(rlist[0]->knode2() == ncelist[i]->knode2())
		{
			rlist += ncelist[i]->swap();
			break;
		}
	}

	// 4. repeat 3. until the polygone is close or there is no match
	for(t = 0;t < ncelist.length() && rlist.getLast()->knode2() != rlist[0]->knode1();t++)
	{
		best = 0;
		// 3. take the next edge closest in the plane discribed by 1. + 2.
		for(i = 1;i < ncelist.length();i++)
		{
			// is the edge in rlist you cannot insert it again
			if(rlist.find(ncelist[i]) >= 0)
				continue;
			if(rlist.getLast()->knode2() == ncelist[i]->knode1())
			{
				best = ncelist[i];
				break;
			}
			if(rlist.getLast()->knode2() == ncelist[i]->knode2())
			{
				best = ncelist[i]->swap();
				break;
			}
			
		}
		for(;i < ncelist.length();i++)
		{
			// is the edge in rlist you cannot insert it again
			if(rlist.find(ncelist[i]) >= 0)
				continue;
			if(rlist.getLast()->knode2() == ncelist[i]->knode1())
			{
				double	d1,d2;
				Vector3	v1,v2,v3;

				v1 = rlist[0]->knode1()->vector();
				v2 = rlist[0]->knode2()->vector();
				v3 = rlist[1]->knode2()->vector();
				d1 = best->knode1()->vector().distance(v1,v2,v3);
				v1 = rlist[0]->knode1()->vector();
				v2 = rlist[0]->knode2()->vector();
				v3 = rlist[1]->knode2()->vector();
				d2 = ncelist[i]->knode2()->vector().distance(v1,v2,v3);
				if(d2 < d1)
					 best = ncelist[i];
			}
			if(rlist.getLast()->knode2() == ncelist[i]->knode2())
			{
				double	d1,d2;
				Vector3	v1,v2,v3;

				v1 = rlist[0]->knode1()->vector();
				v2 = rlist[0]->knode2()->vector();
				v3 = rlist[1]->knode2()->vector();
				d1 = best->knode1()->vector().distance(v1,v2,v3);
				v1 = rlist[0]->knode1()->vector();
				v2 = rlist[0]->knode2()->vector();
				v3 = rlist[1]->knode2()->vector();
				d2 = ncelist[i]->knode1()->vector().distance(v1,v2,v3);

				if(d2 < d1)
					 best = ncelist[i]->swap();
			}
		}
		if(!best)
			break;
		rlist +=  best;
	}

	// 5. is the polygone not close return an empty one
	if(rlist.getLast()->knode2() != rlist[0]->knode1())
		rlist.empty();

	return rlist;
}

int	graph::close()
{
	list<edge*>	enclist;
	int		c,g;

	c = 0;
	g = -2;
	for(g = -2;g <= 0;g++)
	{
		while(!isClose(&enclist,g))
		{
			list<edge*>	polygone;
			edge		*e;
			int		i;

			for(i = 0;i < c;i++)
			{
				e = enclist[0];
				if(!e->closed() > -2)
				{
					enclist.deleteAt(0);
					enclist += e;
				}
			}
		
			polygone.setUnique(1);
			polygone.empty();
			polygone = getOpenPolygone(enclist);

			printf("found a %i point polygone\n",polygone.length());

			if(polygone.length() == 0 || polygone[0]->knode1() != polygone.getLast()->knode2())
				break;

			for(i = 0;i < polygone.length();i++)
			{
				printf("%i. <%g,%g,%g> <%g,%g,%g>\n",i,
					polygone[i]->knode1()->vector()[0],
					polygone[i]->knode1()->vector()[1],
					polygone[i]->knode1()->vector()[2],
					polygone[i]->knode2()->vector()[0],
					polygone[i]->knode2()->vector()[1],
					polygone[i]->knode2()->vector()[2]);
			}


			// this is not the best solution
			// make a geometrytest if we have to use polygone[1] and polygon[2] or later pairs,
			// because they may be concave
			//new edge(this,polygone[0]->knode1(),polygone[1]->knode2(),numEdges() + 1);

			int		t,n,b1,b2;
			Vector3		v1,v2,vb1,vb2;

			b1 = 0;
			b2 = 2;
			vb1 = polygone[b1]->knode1()->vector();
			vb2 = polygone[b2]->knode1()->vector();
			for(i = 0;i < polygone.length() - 1;i++)
			{
				for(t = i + polygone.length() / 2;t < polygone.length() / 2;t++)
				{
					n = t % polygone.length();
					if(abs(n - i)  < 2)
						continue;
					v1 = polygone[i]->knode1()->vector();
					v2 = polygone[n]->knode1()->vector();
					if(vb1.distance(vb2) > v1.distance(v2))
					{
						b1 = i;
						b2 = n;
						vb1 = v1;
						vb2 = v2;
					}
				}
			}

			new edge(this,polygone[b1]->knode1(),polygone[b2]->knode1(),numEdges() + 1);

			removeDoubleKnodes();
			removeDoubleEdges();

			c++;
		}
	}

	return isClose();
}

int	graph::removeDoubleEdges()
{
	int		i;

	for(i = edgelist.length() - 1;i >= 0;i--)
	{
		if(!edgelist[i]->unique())
		{
			printf("delete edge %i\n",edgelist[i]->id());
			delete edgelist[i];
		}
	}

	return 0;
}

int	graph::removeDoubleKnodes()
{
	int		i,t;

	for(i = knodelist.length() - 2;i >= 0;i--)
	{
		for(t = knodelist.length() - 1;t > i;t--)
		{
			if((knodelist[i] != knodelist[t]) &&
			  (knodelist[i]->vector() == knodelist[t]->vector()))
			{
				printf("delete knode <%g,%g,%g>\n",knodelist[i]->vector()[0],knodelist[i]->vector()[1],knodelist[i]->vector()[2]);
				knodelist[i]->replace(knodelist[t]);
				delete knodelist[t];
			}
		}
	}

	return 0;
}

knode	*graph::findKnode(Vector3 v)
{
	int	i;

	for(i = 0;i < knodelist.length();i++)
	{
		if(knodelist[i]->vector() == v)
			return knodelist[i];
	}

	return 0;
}



