/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** torus.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "torus.h"
#include "num.h"
#include "view.h" 

torus::torus(base *p,char *n,double r1,double r2,texture *t) :
	csgobj(p,n,t)
{
	dragvector	*dv;

	numtype = NUM_TORUS;
	major = r1;
	minor = r2;
	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,0,-minor),0,0,&minor,-1);
	dv->setPosition(Vector3(0,0,-1));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

torus::torus(base *p,char *n,double r1,double r2,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,vs,vr,vt)
{
	dragvector	*dv;

	numtype = NUM_TORUS;
	major = r1;
	minor = r2;
	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,0,-minor),0,0,&minor,-1);
	dv->setPosition(Vector3(0,0,-1));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

torus::torus(base *p,char *n,double r1,double r2,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,t,vs,vr,vt)
{
	dragvector	*dv;

	numtype = NUM_TORUS;
	major = r1;
	minor = r2;
	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,0,-minor),0,0,&minor,-1);
	dv->setPosition(Vector3(0,0,-1));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

torus::torus(base *p,torus *tc) :
	csgobj(p,(csgobj*)tc)
{
	dragvector	*dv;

	numtype = NUM_TORUS;
	major = tc->major;
	minor = tc->minor;
	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,0,-minor),0,0,&minor,-1);
	dv->setPosition(Vector3(0,0,-1));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

torus::~torus()
{
	// workaround
	removeFromParent();
}

base	*torus::copy(base *p)
{
	return new torus(p,this);
}


int	torus::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int	torus::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

void	torus::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("torus: %s\n",name);
}

int	torus::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{ 
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab); 
	fprintf(fp,"// Objectname = %s\n",name); 
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = torus\n");
	printTab(fp,tab);
	fprintf(fp,"torus\n");
	printTab(fp,tab); 
	fprintf(fp,"{\n"); 
	printTab(fp,tab + tabsize); 
	fprintf(fp,"%f, %f\n",major,minor); 
	
	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	transform::exportPOV(fp,tab + tabsize,tabsize,anim);

	if(isFlag(HOLLOW_ON))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow on\n");
	}
	else if(isFlag(HOLLOW_OFF))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow off\n");
	}
	
	printTab(fp,tab);
	fprintf(fp,"}\n");

	return 0;
}

int	torus::save(media *m,int ver)
{
	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);
			writeChunk("TRUS");
			writeNameChunk(name);

			saveFlags(m);

			writeDouble(major);
			writeDouble(minor);

			anim::save(m);
			transform::save(m,ver);

			saveTexture(m);
			writeChunkLen();
 		break;
		default:
			return -2;
	}


	return 0;
}

int	torus::load(media *m,int l,int ver)
{
	int	pos;

	if(!m) return -1;

	pos = m->tell();
	
	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);

			loadFlags(m,l);

			major = readDouble();
			minor = readDouble();

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);

 			if(l > (m->tell() - pos))
			{
				loadTexture(m,l - (m->tell() - pos));
			}
 		break;
		default:
			return -2;
	}

	return 0;
}

int	torus::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p,vx,vy,vz;

	//printf("Torus\n");

	if(isFlag(HIDE)) return 0;

	p = Vector3(0,0,0);
	vx = Vector3(major,0,0);
	vy = Vector3(0,minor,0);
	vz = Vector3(0,0,major);

	transformMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
		v->drawTorus(p,vx,vy,vz,minor,m,anim);

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	torus::calculate(int)
{
	setMin(Vector3(-major,-minor,-major));
	setMax(Vector3(major,minor,major));

	return 0;
}

void	torus::setMinorRadius(double nmr)
{
	minor = nmr;
}

void	torus::setMajorRadius(double nmr)
{
	major = nmr;
}

double	torus::volume(int anim)
{
	Vector3		v;

	v = getVScale();

	return 2 * PI * PI * minor * minor * major * v[0] * v[1] * v[2];
}



