/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** patternmodifier_bak.cpp
*/
#include "patternmodifier.h"
#include <chunk.h>

patternmodifier::patternmodifier():chunk()
{
	scale=Vector3(1,1,1);
	rotate=Vector3(0,0,0);
	translate=Vector3(0,0,0);
	turbulence=Vector3(0,0,0);
	octaves=0.0;
	omega=0.0;
	lambda=0.0;
	blackhole=Vector3(0,0,0);
	fall_off=0;
	strength=0;
	repeat=Vector3(0,0,0);
	inverse=0;
	repeatwarp=Vector3(0,0,0);
	offsetwarp=Vector3(0,0,0);
	flipwarp=Vector3(0,0,0);
	blackholeradius=0;
	turbulencewarp=Vector3(0,0,0);
	mode=0;
}

patternmodifier::~patternmodifier()
{
}

void patternmodifier::saveVector(FILE *fp, Vector3 v)
{
	fwrite(&v[0],1,sizeof(double),fp);
	fwrite(&v[1],1,sizeof(double),fp);
	fwrite(&v[2],1,sizeof(double),fp);
}

void patternmodifier::readVector(FILE*fp, Vector3 v)
{
	fread(&v[0],1,sizeof(double),fp);
	fread(&v[1],1,sizeof(double),fp);
	fread(&v[2],1,sizeof(double),fp);
}

void patternmodifier::exportPOVVector(FILE *fp, Vector3 v)
{
	if (v[0]==v[1]) {
		if (v[1]==v[2]) {
	      	  	fprintf(fp,"%g",v[0]);
			return;
		}
	}
	fprintf(fp,"<%g, %g, %g>",v[0],v[1],v[2] );
}

void patternmodifier::setScale(Vector3 v)
{
	scale=v;
}

void patternmodifier::setMode(int m)
{
	mode=m;
}


int patternmodifier::getMode()
{
	return mode;
}
void patternmodifier::setTurbulenceWarp(Vector3 v)
{
	turbulencewarp=v;
}

void patternmodifier::setRotate(Vector3 v)
{
	rotate=v;
}
void patternmodifier::setTranslate(Vector3 v)
{
	translate=v;
}
void patternmodifier::setTurbulence(Vector3 v)
{
	turbulence=v;
}
void patternmodifier::setBlackHole(Vector3 v)
{
	blackhole=v;
}
void patternmodifier::setBlackHoleRadius(double v)
{
	blackholeradius=v;
}

void patternmodifier::setRepeat(Vector3 v)
{
	repeat=v;
}
void patternmodifier::setRepeatWarp(Vector3 v)
{
	repeatwarp=v;
}
void patternmodifier::setOffsetWarp(Vector3 v)
{
	offsetwarp=v;
}
void patternmodifier::setFlipWarp(Vector3 v)
{
	flipwarp=v;
}

void patternmodifier::setOctaves(double v)
{
	octaves=v;
}
void patternmodifier::setOmega(double v)
{
	omega=v;
}
void patternmodifier::setLambda(double v)
{
	lambda=v;
}
void patternmodifier::setFallOff(double v)
{
	fall_off=v;
}
void patternmodifier::setStrength(double v)
{
	strength=v;
}
void patternmodifier::setInverse(double v)
{
	inverse=v;
}

Vector3 patternmodifier::getScale()
{
	return scale;
}

Vector3 patternmodifier::getTurbulenceWarp()
{
	return turbulencewarp;
}

Vector3 patternmodifier::getRotate()
{
	return rotate;
}
Vector3 patternmodifier::getTranslate()
{
	return translate;
}
Vector3 patternmodifier::getTurbulence()
{
	return turbulence;
}
Vector3 patternmodifier::getBlackHole()
{
	return blackhole;
}
double patternmodifier::getBlackHoleRadius()
{
	return blackholeradius;
}

Vector3 patternmodifier::getRepeat()
{
	return repeat;
}
Vector3 patternmodifier::getRepeatWarp()
{
	return repeatwarp;
}
Vector3 patternmodifier::getOffsetWarp()
{
	return offsetwarp;
}
Vector3 patternmodifier::getFlipWarp()
{
	return flipwarp;
}
double patternmodifier::getOctaves()
{
	return octaves;
}
double patternmodifier::getOmega()
{
	return omega;
}
double patternmodifier::getLambda()
{
	return lambda;
}
double patternmodifier::getFallOff()
{
	return fall_off;
}
double patternmodifier::getStrength()
{
	return strength;
}
double patternmodifier::getInverse()
{
	return inverse;
}


int	patternmodifier::save(FILE *fp)
{
	if(!fp)	return -1;
	#ifdef DEBUG
		printf("saving PatternModifier\n");
	#endif
	setFile(fp);

	writeChunk("PATM");
	saveVector(fp,scale);
	saveVector(fp,rotate);
	saveVector(fp,translate);
	saveVector(fp,turbulence);
	saveVector(fp,blackhole);
	saveVector(fp,repeat);
	saveVector(fp,repeatwarp);
	saveVector(fp,offsetwarp);
	saveVector(fp,flipwarp);
	SD(octaves);
	SD(omega);
	SD(lambda);
	SD(fall_off);
	SD(strength);
	SD(inverse);
	SD(blackholeradius);
	SI(mode);
	writeChunkLen();
	#ifdef DEBUG
		printf("PatternMOdifier saved\n");
	#endif
	return 0;
}

int	patternmodifier::load(FILE *fp,int l)
{
	int	pos = ftell(fp);
	#ifdef DEBUG
		printf("Loading Patternmodifier\n");
	#endif
	if(!fp) return -1;
	readVector(fp,scale);
	readVector(fp,rotate);
	readVector(fp,translate);
	readVector(fp,turbulence);
	readVector(fp,blackhole);
	readVector(fp,repeat);
	readVector(fp,repeatwarp);
	readVector(fp,offsetwarp);
	readVector(fp,flipwarp);
	fread(&octaves, 1, sizeof(double),fp);
	fread(&omega, 1, sizeof(double),fp);
	fread(&lambda, 1, sizeof(double),fp);
	fread(&fall_off, 1, sizeof(double),fp);
	fread(&strength, 1, sizeof(double),fp);
	fread(&inverse, 1, sizeof(double),fp);
	fread(&blackholeradius, 1, sizeof(double),fp);
	fread(&mode, 1, sizeof(int),fp);
	
	fseek(fp,pos+l,SEEK_SET); // Man weiss ja nie...
	#ifdef DEBUG
		printf("Patternmodifier loaded\n");
	#endif
	return 0;
}

int patternmodifier::exportPOV(FILE *fp,int tab,int tabsize, int anim)
{
	if(!fp) return -1;
	#ifdef DEBUG
		printf("exporting Patternmodifer (POV)\n");
	#endif


	if ((scale[0]!=1)||(scale[1]!=1)||(scale[2]!=1)) {
		printTab(fp,tab);
		fprintf(fp,"scale ");
		exportPOVVector(fp,scale);
		fprintf(fp,"\n");
	}
	if ((translate[0]!=0)||(translate[1]!=0)||(translate[2]!=0)) {
		printTab(fp,tab);
		fprintf(fp,"translate ");
		exportPOVVector(fp,translate);
		fprintf(fp,"\n");
	}
	if ((rotate[0]!=0)||(rotate[1]!=0)||(rotate[2]!=0)) {
		printTab(fp,tab);
		fprintf(fp,"rotate ");
		exportPOVVector(fp,rotate);
		fprintf(fp,"\n");
	}

	switch(mode) {
		case 0:
		break;
		case 1:
			if ((turbulence[0]!=0)||(turbulence[1]!=0)||(turbulence[2]!=0)) {
				printTab(fp,tab);
				fprintf(fp,"turbulence ");
				exportPOVVector(fp,turbulence);
				fprintf(fp,"\n");
			}
			if (octaves!=0) {
				printTab(fp,tab+tabsize);
				fprintf(fp,"octaves %g\n",octaves);	
			}
			if (omega!=0) {
				printTab(fp,tab+tabsize);
				fprintf(fp,"omega %g\n",omega);	
			}
			if (lambda!=0) {
				printTab(fp,tab+tabsize);
				fprintf(fp,"lambda %g\n",lambda);	
			}
		break;
		case 2://simple warp
			if ((repeatwarp[0]!=0)||(repeatwarp[1]!=0)||(repeatwarp[2]!=0)) {
				printTab(fp,tab);
				fprintf(fp,"warp { ");			
				printTab(fp,tab+tabsize);
				fprintf(fp,"repeat ");
				exportPOVVector(fp,repeatwarp);
				fprintf(fp,"\n");	
				if ((offsetwarp[0]!=0)||(offsetwarp[1]!=0)||(offsetwarp[2]!=0)) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"offset ");
					exportPOVVector(fp,offsetwarp);
					fprintf(fp,"\n");
				}
				if ((flipwarp[0]!=0)||(flipwarp[1]!=0)||(flipwarp[2]!=0)) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"flip ");
					exportPOVVector(fp,flipwarp);
					fprintf(fp,"\n");
				}	
				if ((turbulencewarp[0]!=0)||(turbulencewarp[1]!=0)||(turbulencewarp[2]!=0)) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"turbulence ");
					exportPOVVector(fp,turbulencewarp);
					fprintf(fp,"\n");
				}
				if (octaves!=0) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"octaves %g\n",octaves);	
				}
				if (omega!=0) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"omega %g\n",omega);	
				}
				if (lambda!=0) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"lambda %g\n",lambda);	
				}
				printTab(fp,tab);
				fprintf(fp,"} ");			
			}
		break;
		case 3:
			if (blackholeradius!=0) {
				printTab(fp,tab);
				fprintf(fp,"warp { ");		
				printTab(fp,tab+tabsize);
				fprintf(fp,"black_hole <%g, %g, %g> %g\n",blackhole[0],blackhole[1],blackhole[2],blackholeradius);
				if (fall_off!=0) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"falloff %g\n",fall_off);
				}
				if (strength!=0) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"strength %g\n",strength);
				}
				if ((repeat[0]!=0)||(repeat[1]!=0)||(repeat[2]!=0)) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"repeat ");
					exportPOVVector(fp,repeat);
					fprintf(fp,"\n");
				}	
				if ((turbulencewarp[0]!=0)||(turbulencewarp[1]!=0)||(turbulencewarp[2]!=0)) {
					printTab(fp,tab+tabsize);
					fprintf(fp,"turbulence ");
					exportPOVVector(fp,turbulencewarp);
					fprintf(fp,"\n");
				}	
				if (inverse!=0) {
					printTab(fp,tab);
					fprintf(fp,"inverse\n");
				}
				printTab(fp,tab);
				fprintf(fp,"} ");		
			}
		break;

	}

	#ifdef DEBUG
		printf("PatternModifier exported (POV) saved\n");
	#endif
	return 0;
}

