/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** viewwidget.cpp
*/
#include "viewwidget.h"
#include "viewwidget.moc"


#include <kapp.h>

#include <math.h>
#include <misc.h>

ViewWidget::ViewWidget(QWidget *p,const char *n,world *wptr,bool autodel) :
	QFrame(p,n), view(wptr,n,autodel)
{
#if (USE_RTTI == 0)
	bv = 0;
#endif
#if (USE_PVM == 1)
	pvm_on=false;
#endif

	setFrameStyle( Box | Sunken );
        setLineWidth(1);

	qpm = new QPixmap(width() - 4,height() - 4);
	qp = new QPainter(qpm);

	peye = Vector3(0,0,200);
	vview = Vector3(0,0,-1);
	vup = Vector3(0,1,0);
	vright = Vector3(1,0,0);
/*
        translate(Vector3(0,0,-200));
        rotate(Vector3(PI,0,0));
*/

	setFOV(0.1);
	
	xo = 0;yo = 0;
	viewmode = 1;
	axemode = 7;

	ppmfp = 0;
	tcounter = -1;

	configure();
}

ViewWidget::~ViewWidget()
{
	emit destroyed();
	
	qp->end();
	
	delete qp;
	
	delete qpm;
}

int	ViewWidget::drawLine(Vector3 a,Vector3 b,Matrix44 m,int anim)
{
	if(drawselected)
		return drawLine(a,b,m,anim,selcolor);

	return drawLine(a,b,m,anim,drawcolor);
}

int	ViewWidget::drawLine(Vector3 a,Vector3 b,Matrix44 m,int anim,QColor color)
{
	Vector2	a2,b2,c2;

	if(project(a,a2,m,anim)) return -1;
	if(project(b,b2,m,anim)) return -1;

	qp->setPen(color);

	if(a2[0] > 32000.0)
	{
		if(a2[0] == b2[0]) return -2;
		c2[0] = 32000;
		c2[1] = ((a2[1] - b2[1]) * (c2[0] - b2[0])) / (a2[0] - b2[0]) - b2[1];
		a2 = c2;
	}
	if(a2[0] < -32000.0)
	{
		if(a2[0] == b2[0]) return -2;
		c2[0] = -32000;
		c2[1] = ((a2[1] - b2[1]) * (c2[0] - b2[0])) / (a2[0] - b2[0]) - b2[1];
		a2 = c2;
	}
	if(a2[1] > 32000.0)
	{
		if(a2[1] == b2[1]) return -2;
		c2[1] = 32000;
		c2[0] = ((a2[0] - b2[0]) * (c2[1] - b2[1])) / (a2[1] - b2[1]) - b2[0];
		a2 = c2;
	}
	if(a2[1] < -32000.0)
	{
		if(a2[1] == b2[1]) return -2;
		c2[1] = -32000;
		c2[0] = ((a2[0] - b2[0]) * (c2[1] - b2[1])) / (a2[1] - b2[1]) - b2[0];
		a2 = c2;
	}

	if(b2[0] > 32000.0)
	{
		if(b2[0] == a2[0]) return -2;
		c2[0] = 32000;
		c2[1] = ((b2[1] - a2[1]) * (c2[0] - a2[0])) / (b2[0] - a2[0]) - a2[1];
		b2 = c2;
	}
	if(b2[0] < -32000.0)
	{
		if(b2[0] == a2[0]) return -2;
		c2[0] = -32000;
		c2[1] = ((b2[1] - a2[1]) * (c2[0] - a2[0])) / (b2[0] - a2[0]) - a2[1];
		b2 = c2;
	}
	if(b2[1] > 32000.0)
	{
		if(b2[1] == a2[1]) return -2;
		c2[1] = 32000;
		c2[0] = ((b2[0] - a2[0]) * (c2[1] - a2[1])) / (b2[1] - a2[1]) - a2[0];
		b2 = c2;
	}
	if(b2[1] < -32000.0)
	{
		if(b2[1] == a2[1]) return -2;
		c2[1] = -32000;
		c2[0] = ((b2[0] - a2[0]) * (c2[1] - a2[1])) / (b2[1] - a2[1]) - a2[0];
		b2 = c2;
	}

	qp->drawLine((int)a2[0],(int)a2[1],(int)b2[0],(int)b2[1]);

	return 0;
}

int	ViewWidget::drawCross(Vector3 a,Matrix44 m,int anim)
{
	Vector2	a2;

	if(project(a,a2,m,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2[0] - 10,(int)a2[1] - 10,(int)a2[0] + 10,(int)a2[1] + 10);
	qp->drawLine((int)a2[0] + 10,(int)a2[1] - 10,(int)a2[0] - 10,(int)a2[1] + 10);

	return 0;
}

int		ViewWidget::drawTriangle(Vector3 a,Vector3 b,Vector3 c,Matrix44 m,int anim)
{
	Vector2	a2,b2,c2;

	if(project(a,a2,m,anim)) return -1;
	if(project(b,b2,m,anim)) return -1;
	if(project(c,c2,m,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2[0],(int)a2[1],(int)b2[0],(int)b2[1]);
	qp->drawLine((int)b2[0],(int)b2[1],(int)c2[0],(int)c2[1]);
	qp->drawLine((int)c2[0],(int)c2[1],(int)a2[0],(int)a2[1]);

	return 0;
}

int	ViewWidget::drawTrapezium(Vector3 a,Vector3 b,Vector3 c,Matrix44 m,int anim)
{
	Vector3		d;
	Vector2	a2,b2,c2,d2;

	d = b;
	d -= a;
	d += c;

	if(project(a,a2,m,anim)) return -1;
	if(project(b,b2,m,anim)) return -1;
	if(project(c,c2,m,anim)) return -1;
	if(project(d,d2,m,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2(0),(int)a2(1),(int)b2(0),(int)b2(1));
	qp->drawLine((int)b2(0),(int)b2(1),(int)c2(0),(int)c2(1));
	qp->drawLine((int)c2(0),(int)c2(1),(int)d2(0),(int)d2(1));
	qp->drawLine((int)d2(0),(int)d2(1),(int)a2(0),(int)a2(1));

	return 0;
}

int	ViewWidget::drawCircle(Vector3 m,Vector3 a,Vector3 b,Matrix44 ma,int anim)
{
	Vector3		d,n;
	Vector2	n2,d2;
	int		t;
	double		alpha;

	alpha = (double)-1 / 16 * PI;
	n = m;
	n += a * cos(alpha) + b * sin(alpha);
	if(project(n,n2,ma,anim)) return -1;

/*
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);
*/
	for(t = 0;t < 32;t++)
	{
		alpha = (double)t / 16 * PI;
		d = m;
		d += a * cos(alpha) + b * sin(alpha);

		if(project(d,d2,ma,anim)) return -1;

		qp->drawLine((int)n2(0),(int)n2(1),(int)d2(0),(int)d2(1));

		n2 = d2;
	}

	return 0;
}

int		ViewWidget::drawSemiCircle(Vector3 m,Vector3 a,Vector3 b,Matrix44 ma,int anim)
{
	Vector3		d,n;
	Vector2		n2,d2;
	int		t;
	double		alpha;

	alpha = (double)-1 / 16 * PI;
	n = m;
	n += a * cos(alpha) + b * sin(alpha);
	if(project(n,n2,ma,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	for(t = 0;t < 16;t++)
	{
		alpha = (double)t / 16 * PI;
		d = m;
		d += a * cos(alpha) + b * sin(alpha);

		if(project(d,d2,ma,anim)) return -1;

		qp->drawLine((int)n2(0),(int)n2(1),(int)d2(0),(int)d2(1));

		n2 = d2;
	}

	return 0;
}

int		ViewWidget::drawSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m,a,b,ma,anim);
	drawCircle(m,b,c,ma,anim);
	drawCircle(m,c,a,ma,anim);

	return 0;
}

int		ViewWidget::drawSemiSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m,a,b,ma,anim);
	drawSemiCircle(m,b,c,ma,anim);
	drawSemiCircle(m,c,a,ma,anim);

	return 0;
}

int		ViewWidget::drawBox(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] -= (a + b + c);
	v[1] -= (a - b + c);
	v[2] += (a + b - c);
	v[3] += (a - b - c);
	v[4] -= (a + b - c);
	v[5] -= (a - b - c);
	v[6] += (a + b + c);
	v[7] += (a - b + c);

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[1],ma,anim);
	drawLine(v[1],v[2],ma,anim);
	drawLine(v[2],v[3],ma,anim);
	drawLine(v[3],v[0],ma,anim);
	drawLine(v[4],v[5],ma,anim);
	drawLine(v[5],v[6],ma,anim);
	drawLine(v[6],v[7],ma,anim);
	drawLine(v[7],v[4],ma,anim);
	drawLine(v[0],v[4],ma,anim);
	drawLine(v[1],v[5],ma,anim);
	drawLine(v[2],v[6],ma,anim);
	drawLine(v[3],v[7],ma,anim);

	return 0;
}

int		ViewWidget::drawCone(Vector3 m,Vector3 a,Vector3 b,Vector3 c,double d,Matrix44 ma,int anim)
{
	Vector3		v[8];

	d *= 2;

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a + c;
	v[1] += b + c;
	v[2] += a + c;
	v[3] += -b + c;
	v[4] += -a * d - c;
	v[5] += b * d - c;
	v[6] += a * d - c;
	v[7] += -b * d - c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[4],ma,anim);
	drawLine(v[1],v[5],ma,anim);
	drawLine(v[2],v[6],ma,anim);
	drawLine(v[3],v[7],ma,anim);

	drawCircle(m + c,a,b,ma,anim);
	drawCircle(m,a * (1 + d) / 2,b * (1 + d) / 2,ma,anim);
	drawCircle(m - c,a * d,b * d,ma,anim);

	return 0;
}

int		ViewWidget::drawCylinder(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a - c;
	v[1] += b - c;
	v[2] += a - c;
	v[3] += -b - c;
	v[4] += -a + c;
	v[5] += b + c;
	v[6] += a + c;
	v[7] += -b + c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[4],ma,anim);
	drawLine(v[1],v[5],ma,anim);
	drawLine(v[2],v[6],ma,anim);
	drawLine(v[3],v[7],ma,anim);

	drawCircle(-c,a,b,ma,anim);
	drawCircle(m,a,b,ma,anim);
	drawCircle(c,a,b,ma,anim);

	return 0;
}

int		ViewWidget::drawTorus(Vector3 m,Vector3 a,Vector3 b,Vector3 c,double d,Matrix44 ma,int anim)
{
	if(drawselected)
		qp->setPen(selcolor);
	else
		qp->setPen(drawcolor);

	// top
	drawCircle(m + b,a,c,ma,anim);
	// bottom
	drawCircle(m - b,a,c,ma,anim);
	// inner
	drawCircle(m,a / a.length() * (a.length() - d),c / c.length() * (c.length() - d),ma,anim);
	// outer
	drawCircle(m,a / a.length() * (a.length() + d),c / c.length() * (c.length() + d),ma,anim);

	// west
	drawCircle(m + a,a / a.length() * d,b,ma,anim);
	// east
	drawCircle(m - a,a / a.length() * d,b,ma,anim);
	// north
	drawCircle(m + c,c / c.length() * d,b,ma,anim);
	// south
	drawCircle(m - c,c / c.length() * d,b,ma,anim);

	return 0;
}

int		ViewWidget::drawBlobSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	qp->setPen(blobcolor);

	drawCircle(m,a,b,ma,anim);
	drawCircle(m,b,c,ma,anim);
	drawCircle(m,c,a,ma,anim);

	return 0;
}

int		ViewWidget::drawBlobCylinder(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a - c;
	v[1] += b - c;
	v[2] += a - c;
	v[3] += -b - c;
	v[4] += -a + c;
	v[5] += b + c;
	v[6] += a + c;
	v[7] += -b + c;

	drawLine(v[0],v[4],ma,anim,blobcolor);
	drawLine(v[1],v[5],ma,anim,blobcolor);
	drawLine(v[2],v[6],ma,anim,blobcolor);
	drawLine(v[3],v[7],ma,anim,blobcolor);

	qp->setPen(blobcolor);

	drawCircle(-c,a,b,ma,anim);
	drawCircle(m,a,b,ma,anim);
	drawCircle(c,a,b,ma,anim);

	return 0;
}

int		ViewWidget::drawIcon(Vector3 v,QPixmap p,int x,int y,Matrix44 m,int anim)
{
	Vector2	v2;

	if(project(v,v2,m,anim)) return -1;

	qp->drawPixmap((int)v2[0] + x,(int)v2[1] + y,p);

	return 0;
}

int		ViewWidget::drawSymbol(Vector3 v,const char *name,Matrix44 m,int anim)
{
#include "icons/tree/lightpoint.xpm"
#include "icons/tree/spotlight.xpm"
//#include "icons/tree/arealight.xpm"

	QPixmap			pm_lp((const char**)lightpoint_xpm);
	QPixmap			pm_sl((const char**)spotlight_xpm);

	if(strcmp(name,"lightpoint") == 0)
		return drawIcon(v,pm_lp,12,12,m,anim);
	if(strcmp(name,"spotlight") == 0)
		return drawIcon(v,pm_sl,12,12,m,anim);

	return -1;
}

int		ViewWidget::drawAxis(Matrix44 m,int anim)
{
	Vector3		vm(0,0,0),vx(1,0,0),vy(0,1,0),vz(0,0,1);

	drawLine(vm,vx,m,anim,xaxecolor);//QColor(200,0,0)

	drawLine(vm,vy,m,anim,yaxecolor);//QColor(0,200,0)

	drawLine(vm,vz,m,anim,zaxecolor);//QColor(100,0,200)

	return 0;
}

int		ViewWidget::drawDragvector(Matrix44 m,Vector3 p,Vector3 d,int anim)
{
	Vector2	a;

	if(anim) return 0;
	
	drawLine(p,p + d,m,0,QColor(128,0,64));

	if(project(p + d,a,m,0)) return -1;

	qp->setPen(QColor(128,0,64));
	qp->drawEllipse((int)a[0] - 3,(int)a[1] - 3,7,7);

	return 0;
}

void		ViewWidget::drawanim()
{
	draw(1);
}

void		ViewWidget::draw()
{
	draw(0);
}

void		ViewWidget::draw(int anim)
{
	Matrix44	m;
	Vector3		vm(0,0,0),vx(1,0,0),vy(0,1,0),vz(0,0,1),a,b;
	int		t;

	stopRendering();

	m.unify();

	if(getSubObject())
	{
		vm = -((transform*)subobject)->getVTranslate();
		m.translateVector(vm);
	}
	else
	{
		vm = Vector3(0,0,0);
	}
			
	qp->fillRect(0,0,width(),height(),backcolor);

/*
	qp->setPen(QColor(0,0,0));
	qp->drawLine(0,0,width(),0);
	qp->drawLine(0,0,0,height());
	qp->drawLine(1,1,width() - 2,1);
	qp->setPen(QColor(255,255,255));
	qp->drawLine(width() - 1,0,width() - 1,height() - 1);
	qp->drawLine(0,height() - 1,width() - 1,height() - 1);
*/

	qp->setPen(gridcolor);

	switch(projmode)
	{
		case PERSPECTIVE:
			for(t = -gridlines_xz;grid_xz && t <= gridlines_xz;t++)
			{
				a = vm + (vx * (double)t + vz * -gridlines_xz) * gridspace_xz;
				b = vm + (vx * (double)t + vz * gridlines_xz) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
				a = vm + (vz * (double)t + vx * -gridlines_xz) * gridspace_xz;
				b = vm + (vz * (double)t + vx * gridlines_xz) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
			}
			for(t = -gridlines_xy;grid_xy && t <= gridlines_xy;t++)
			{
				a = vm + (vx * (double)t + vy * -gridlines_xy) * gridspace_xy;
				b = vm + (vx * (double)t + vy * gridlines_xy) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
				a = vm + (vy * (double)t + vx * -gridlines_xy) * gridspace_xy;
				b = vm + (vy * (double)t + vx * gridlines_xy) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
			}
			for(t = -gridlines_zy;grid_zy && t <= gridlines_zy;t++)
			{
				a = vm + (vz * (double)t + vy * -gridlines_zy) * gridspace_zy;
				b = vm + (vz * (double)t + vy * gridlines_zy) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
				a = vm + (vy * (double)t + vz * -gridlines_zy) * gridspace_zy;
				b = vm + (vy * (double)t + vz * gridlines_zy) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vm + vx * 10,m,anim,xaxecolor);//QColor(200,0,0)

			drawLine(vm + vx * 10 + vx + vz,vm + vx * 10 - vz,m,anim,letterxcolor);//QColor(100,0,0));
			drawLine(vm + vx * 10 + vx - vz,vm + vx * 10 + vz,m,anim,letterxcolor);

			drawLine(vm,vm + vy * 10,m,anim,yaxecolor);//QColor(0,200,0)

			drawLine(vm + vy * 10 + vy + vx,vm + vy * 10 + vy * 0.5,m,anim,letterycolor);//QColor(0,100,0));
			drawLine(vm + vy * 10 +      vx,vm + vy * 10 + vy * 0.5,m,anim,letterycolor);
			drawLine(vm + vy * 10 + vy * 0.5,vm + vy * 10 + vy * 0.5 - vx,m,anim,letterycolor);

			drawLine(vm,vm + vz * 10,m,anim,zaxecolor);//QColor(100,0,200)

			drawLine(vm + vz * 10 + vz + vx,vm + vz * 10 + vx,m,anim,letterzcolor);//QColor(50,0,100));
			drawLine(vm + vz * 10 + vz - vx,vm + vz * 10 - vx,m,anim,letterzcolor);
			drawLine(vm + vz * 10 + vz - vx,vm + vz * 10 + vx,m,anim,letterzcolor);
		break;
		case PARALLEL_XY:
			for(t = -gridlines_xy;t <= gridlines_xy;t++)
			{
				a = vm + (vx * (double)t + vy * -10) * gridspace_xy;
				b = vm + (vx * (double)t + vy * 10) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
				a = vm + (vy * (double)t + vx * -10) * gridspace_xy;
				b = vm + (vy * (double)t + vx * 10) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vm + vx * 10,m,anim,xaxecolor);
			drawLine(vm,vm + vy * 10,m,anim,yaxecolor);
		break;
		case PARALLEL_ZY:
			for(t = -gridlines_zy;t <= gridlines_zy;t++)
			{
				a = vm + (vz * (double)t + vy * -10) * gridspace_zy;
				b = vm + (vz * (double)t + vy * 10) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
				a = vm + (vy * (double)t + vz * -10) * gridspace_zy;
				b = vm + (vy * (double)t + vz * 10) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vm + vz * 10,m,anim,zaxecolor);
			drawLine(vm,vm + vy * 10,m,anim,yaxecolor);
		break;
		case PARALLEL_XZ:
			for(t = -gridlines_xz;t <= gridlines_xz;t++)
			{
				a = vm + (vx * (double)t + vz * -10) * gridspace_xz;
				b = vm + (vx * (double)t + vz * 10) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
				a = vm + (vz * (double)t + vx * -10) * gridspace_xz;
				b = vm + (vz * (double)t + vx * 10) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vm + vz * 10,m,anim,zaxecolor);
			drawLine(vm,vm + vx * 10,m,anim,xaxecolor);
		break;
	}

	qp->setPen(drawcolor);
	
	if(worldptr)
	{
		if(getSubObject())
			subobject->draw(this,m,anim);
		else
			worldptr->draw(this,m,anim);
	}
	else
		printf("no world!\n");

	flush();
}

void		ViewWidget::setWorld(world *ww)
{
	view	*v;
        int	i,error;

	// is this world new for me?
	if(ww == getWorld())
		return; // no, nothing to do

	// lets search a view without widget
	for(i = 0;i < ww->getNumViews();i++)
	{			
		v = ww->getView(i);
	
		// lets have a look at v
		// if v is a widget
#if (USE_RTTI == 1)
		if(dynamic_cast<ViewWidget*>(v) != 0)
#else
		if(v->baseview() == 0)
#endif
			continue; // take the next one
			
		// Do not remove myself !!!
		// (we should not reach here, if v is widget)
		if((view*)this != v)
		{
			ww->removeView(v);
			take(v);
			
			delete v;
			break;
		}
	}
	
	while((error = view::setWorld(ww)) != 0)
	{
		switch(error)
		{
#ifdef DEBUG		
			case -1:
				printf("Unknown error in %s at line %i\n",__FILE__,__LINE__);
			break;
#endif
			case -2:
#ifdef DEBUG		
				printf("view has no name in %s at line %i\n",__FILE__,__LINE__);
#endif
			break;
#ifdef DEBUG		
			case -3:
				printf("view exist in %s at line %i\n",__FILE__,__LINE__);
			break;
#endif
			case -4:
#ifdef DEBUG		
				printf("view has existing name in %s at line %i\n",__FILE__,__LINE__);
#endif
				{
					char	buffer[256];
				
				        strncpy(buffer,getName(),254);
				        strcat(buffer,"1");
				        setName(buffer);
				
				        emit changedViewName();
				 }
			break;
		}
	}	
	
//	resize();
		
	draw();
}

void		ViewWidget::rotateView(double x,double y,double z)
{
//	rotate(Vector3(x,y,z));

	Vector3		vr(x,y,z);
	Matrix44	m;
	Vector4		v,u,r;

	m.rotateVector(vr);
	v = vview;
	u = vup;
	r = vright;

	v *= m;
	u *= m;
	r *= m;

	vview = v;
	vup = u;
	vright = r;

	calcVectors(0);
}

void		ViewWidget::rotate0View(double x,double y,double z)
{
//	rotateView(x,y,z);
	
	Vector3		vr(x,y,z);
	Matrix44	m;
	Vector4		v4;

	m.rotateVector(vr);
	v4 = peye;
	v4 *= m;
	peye = v4;

	rotateView(x,y,z);
}

void		ViewWidget::moveView(double x,double y,double z)
{
	translate(Vector3(x,y,z));
	calcVectors(0);
}

void		ViewWidget::changeFOV(double d)
{
	setFOV(fov + d);
}

void		ViewWidget::setViewMode(int vm)
{
	viewmode = vm;
}

void		ViewWidget::setProjectionMode(int pm)
{
	projmode = pm;
        draw();
}

int		ViewWidget::projectionMode()
{
	return projmode;
}

void		ViewWidget::setAxeMode(int am)
{
	axemode = am;
}

void		ViewWidget::setSelected(base *bp)
{
	selectedptr = bp;
	draw();
}

void		ViewWidget::mouseMoveEvent(QMouseEvent *me)
{
	int		dx,dy;
	Vector3		v(1,1,1);

	if(projmode == PERSPECTIVE && selectedptr)
	{
		if(selectedptr->changeDragvector(this,selectedptr->getMatrix(),xo,yo,me->x(),me->y()) == 0)
		{
			xo = me->x();
			yo = me->y();

			emit changedModel();
			emit redraw();
			
			return;
		}
	}

	dx = me->x() - xo;
	dy = me->y() - yo;

	switch(viewmode)
	{
		case 1:
			changeFOV(((double)(dx + dy)) / 5000.0);
		break;
		case 2:
			if(me->state() == LeftButton)
				moveView((double)dx / 10.0,(double)dy / 10.0,0);
			else
				moveView(0,(double)dy / 10.0,(double)dx / 10.0);
		break;
		case 3:
			if(me->state() == LeftButton)
				rotate0View((double)dx / 1800 * PI,0,0);
			else if(me->state() == RightButton)
				rotate0View(0,0,(double)dx / 1800 * PI);
			else
				rotate0View(0,(double)dx / 1800 * PI,0);
		break;
		case 4: // Object scale
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;

			if(axemode & 1)
				v(0) = exp((double)dx / -100);
			if(axemode & 2)
				v(1) = exp((double)dx / -100);
			if(axemode & 4)
				v(2) = exp((double)dx / -100);
			((transform*)selectedptr)->scale(v);
			draw();
			emit changedModel();
		break;
		case 5: // Object translate
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;
			if(me->state() == LeftButton)
				((transform*)selectedptr)->translate(Vector3((double)dx / 10.0,(double)dy / 10.0,0));
			else
				((transform*)selectedptr)->translate(Vector3(0,(double)dy / 10.0,(double)dx / 10.0));
			draw();
			emit changedModel();
		break;
		case 6: // Object rotate
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;
			if(axemode & 1)
				((transform*)selectedptr)->rotate(Vector3((double)dx / 1800 * PI,0,0));
			if(axemode & 2)
				((transform*)selectedptr)->rotate(Vector3(0,0,(double)dx / 1800 * PI));
			if(axemode & 4)
				((transform*)selectedptr)->rotate(Vector3(0,(double)dx / 1800 * PI,0));
			draw();
			emit changedModel();
		break;
	}
	draw();

	xo = me->x();
	yo = me->y();
}

void		ViewWidget::mousePressEvent(QMouseEvent *me)
{
	xo = me->x();
	yo = me->y();
}


void		ViewWidget::resizeEvent(QResizeEvent *re)
{
	setScreenSize(width() - 4,height() - 4);
	createScreen(width() - 4,height() - 4);
	
	qp->end();
	qpm->resize(width() - 4,height() - 4);
	qp->begin(qpm);

	draw();
	
	emit resizedView(width() - 4,height() - 4);
	
	QFrame::resizeEvent(re);
}

void		ViewWidget::paintEvent(QPaintEvent *pe)
{
	flush();
	
	QFrame::paintEvent(pe);
}


void		ViewWidget::flush()
{
	QPainter	paint(this);

	qp->end();
	paint.drawPixmap(2,2,*qpm);
	qp->begin(qpm);
}


#if (USE_PVM == 0)
// NON-PVM-Version !!


void		ViewWidget::startRendering(int quality)
{
	char		str[16];
	QString		qstr;
	int		viewnum;
	Preferences	prefs;

	// now wait a quarter second
	if(tcounter != -1 && tcounter < 5)
		return;
		
	printf("Start Rendering\n");
	
	// stop all the timer of this widget
	killTimers();
	proc.kill();
	if(ppmfp)
		fclose(ppmfp);
	ppmfp = 0;
	flush();

	tmpdir = prefs.getTempPath();

	tmpname.sprintf("tmp%i",rand());
	
	qstr = tmpdir + "/" + tmpname + ".pov";
	
	viewnum = worldptr->getViewPosition(this);
	if(viewnum < 0)
		return;
			
	emit report(i18n("Writing renderfile"));
	emit exportPOV(qstr,viewnum);

	proc.clearArguments();

	proc << prefs.getRenderProgram();
	sprintf(str,"+W%i",width());
	proc << str;
	sprintf(str,"+H%i",height());
	proc << str;
//	proc << "-GA";
	qstr = "-O" + tmpdir + "/" + tmpname + ".ppm";
	proc << qstr;
	qstr = "-I" + tmpdir + "/" + tmpname + ".pov";
	proc << qstr << "+FP";
	qstr.sprintf("-Q%i",quality);
	proc << qstr;
	
	proc.start(KProcess::DontCare,KProcess::Stdout);

	x = y = 0;

	tcounter = 0;
	ppmfp = 0;
	lc = 0;

	startTimer(150);
	
	qstr = QString(i18n("Start rendering with")) + " ";
	qstr += prefs.getRenderProgram();
	emit report(qstr);
}

void		ViewWidget::stopRendering()
{
	QString		qstr;
	
	if(!ppmfp)
		return;
		
	qstr = i18n("Ready");
	emit report(qstr);
	
	killTimers();
	proc.kill();
	if(ppmfp)
		fclose(ppmfp);
	ppmfp = 0;
	flush();
	
	qstr = "rm -f " + tmpdir + "/" + tmpname + ".ppm";
	system(qstr);
	qstr = "rm -f " + tmpdir + "/" + tmpname + ".pov";
	system(qstr);
	
	tcounter = -1;
	
	emit stoppedRendering();
}

void		ViewWidget::timerEvent(QTimerEvent*)
{
	QString		tmpstr,qstr;

	if(!proc.isRunning())
	{
		if(proc.normalExit())
		{
#ifdef DEBUG
			printf("Renderer exitstatus:%i\n",proc.exitStatus());
#endif
		}
		else
		{
			QString		errormsg;

			errormsg = i18n("Renderer killed by foreign process!");
			emit report(errormsg);
		}
		killTimers();
		proc.kill();
		tcounter = -1;

		if(ppmfp != 0)
			readImage();
		// Attention! readImage could close the file
		if(ppmfp != 0)
		{
			fclose(ppmfp);
			ppmfp = 0;
		}
		flush();
			
		return;	
	}
	if(ppmfp == 0)
	{

		tcounter++;
		// 33 * 150ms = 4950ms = ~5sec
		if(tcounter == 33)
		{
			emit report(i18n("Renderer is parsing"));
		}
		// 800 * 150ms = 120000ms = 120sec timeout
		if(tcounter > 800)
		{
			emit report(i18n("Stopped rendering becauseof timeout"));

			killTimers();
			proc.kill();

			return;
		}
		
		ppmfp = fopen(tmpdir + "/" + tmpname + ".ppm","rb");
		
		if(ppmfp == 0)
			return;
		fseek(ppmfp,0,SEEK_SET);
	}
	readImage();
}

void	ViewWidget::readImage()
{
	unsigned char	buffer[3 * 1024];
	int		i,t,r,g,b,p;
	QString		tmpstr,qstr;
	
	if(lc < 3)
	{
		p = ftell(ppmfp);
		fseek(ppmfp,0,SEEK_END);
		if(ftell(ppmfp) <= p)
		{
			fseek(ppmfp,p,SEEK_SET);
			return;
		}
		fseek(ppmfp,p,SEEK_SET);
		
		do
		{
			fread(buffer,1,1,ppmfp);
		}
		while(buffer[0] != '\n');
		lc++;
		return;
	}

	p = ftell(ppmfp);
	fseek(ppmfp,0,SEEK_END);
	if(ftell(ppmfp) <= p)
	{
		fseek(ppmfp,p,SEEK_SET);
		return;
	}
	fseek(ppmfp,p,SEEK_SET);
	
	while((i = fread(buffer,3,1024,ppmfp)) > 0)
	{

		for(t = 0;t < i;t++)
		{
			r = buffer[t * 3];
			g = buffer[t * 3 + 1];
			b = buffer[t * 3 + 2];
			qp->setPen(QColor(r,g,b));
			qp->drawPoint(x++,y);
			if(x >= width())
			{
				x = 0;
				y++;
				if(y >= height())
				{
					emit finishedRendering();
					stopRendering();
					
					return;
				}
			}
		}
	}
	
	flush();

	tmpstr.setNum(((x + y * width()) * 100) / (width() * height()));
	qstr = (QString)i18n("Rendered") + " ";
	qstr += tmpstr + "%";
	emit report(qstr);
}

#endif

#if (USE_PVM == 1)
// PVM-Version !!
void		ViewWidget::startRendering(int quality)
{
	QString		qstr,tmp;
	int		viewnum;


	int length,i,j,k;
	FILE 	*file;
	char *data;

	ClusterInfo *ci;
	Preferences prefs;

	if (pvm_on==true) stopRendering();
	pvm_on=true;

	// Is PVM running on the localhost ???

	if ((mytid=pvm_mytid())<0) {
		qstr = i18n("Couldn't start PVM....Daemon not running ?");
		emit report(qstr);
		return;
	}

	// export the POV-File
	
	tmp = prefs.getTempPath();
	tmpname.sprintf("tmp%i",rand());
	qstr = tmp + "/" + tmpname + ".pov";
	viewnum = worldptr->getViewPosition(this);
	if(viewnum < 0) return;
	emit exportPOV(qstr,viewnum);
	file=fopen(qstr,"r");
	if (file) {
		fseek(file,0,SEEK_END);
		length=ftell(file);
		fseek(file,0,SEEK_SET);
		data=(char *)malloc(length);
		if (data) {
			fread(data,1,length,file);
		} else return;
		fclose(file);
	} else return;

	// Add hosts to the PVM-Config

	for (i=0;i<prefs.numClusterHosts();i++) {
		char *host;
		host=strdup(prefs.recentClusterHost(i));

		j=pvm_addhosts(&host,1,&k);				
		if (host) free(host);
	}

        // Get info
	
	pvm_config(&nprocessors,(int*)0,&pi);
	

	for (i=0;i<nprocessors;i++) {
	     	ci=new ClusterInfo;
		ci->pinfo=&pi[i];   // Clustername, speed(?), id
		ci->x1=0;
		ci->y1=0;
		ci->x2=0;
		ci->y2=0;
		ci->timePline=0.0;
		clusters.append(ci);
	}

	pixel=0;

	
	for (i=0;i<nprocessors;i++) {
		int xywhq[9];
		int tile=height()/nprocessors;
		pvmhostinfo *hi=clusters[i]->pinfo;
		printf("%d %s\n",i,hi->hi_name);
		pvm_spawn("k4ded",(char**)0,0,"",1,&hi->hi_tid );

		xywhq[0]=length;		
		clusters[i]->x1=xywhq[1]=0;
		clusters[i]->y1=xywhq[2]=tile*i;  // start-y
		clusters[i]->x2=xywhq[3]=width();
		clusters[i]->y2=xywhq[4]=tile*i+tile;  // end-y
		xywhq[5]=width();
		xywhq[6]=height();
		xywhq[7]=quality;
		pvm_initsend(PvmDataDefault);
		pvm_pkint(xywhq,8,1);
		pvm_send(hi->hi_tid,1);

		pvm_initsend(PvmDataDefault);
		pvm_packf("%+ %*c", PvmDataDefault,length,data);
		pvm_send(hi->hi_tid,2);
	}
	pixel=width()*height();
	pixel_togo=pixel;

	if (data) free(data);
	
	killTimers();

	startTimer(150);

	tmp.setNum(nprocessors);
	qstr = "PVM-Rendering started on ";
	qstr += tmp + " Processors";
	emit report(qstr);
}

void		ViewWidget::stopRendering()
{
	QString		qstr;
        Preferences	prefs;
	int i,j,k;	
	
	emit report(qstr);

	killTimers();
	flush();

	killTimers();
	
	tcounter = -1;
	if (pvm_on==true) {
		for (i=0;i<clusters.length();i++) {
			char *host=clusters[i]->pinfo->hi_name;
			j=pvm_delhosts(&host,1,&k);				
			printf("Removed %s  %d\n",host,j );
		}
		pvm_exit();
		qstr = "PVM-Rendering stopped.";
		emit report(qstr);
		while (clusters.length()>0) {
			delete clusters[0];
			clusters.deleteAt(0);
		}
		nprocessors=0;
		pvm_on=false;
	}

	emit stoppedRendering();
}

void		ViewWidget::timerEvent(QTimerEvent*)
{
	QString		qstr,tmpstr;
	int id,msg_tag,msg_tid;
	int x1,y1,x2,y2,length,time,j,i;
	int t,r,g,b,lc;
	int tid;	
	char *data;
	char *hostname;
	unsigned char *buffer;


	if (!(id=pvm_probe(-1,20))) return;

	//killTimers();

	pvm_bufinfo(id,&j,&msg_tag,&msg_tid);

	pvm_recv(msg_tid,20);
	//pvm_bufinfo(mid,(int*)0,(int*)0,&slaveid);
	pvm_unpackf("%d %d %d %d %d %d",&length,&time,&x1,&y1,&x2,&y2);

	for (i=0;i<nprocessors;i++) {
		if (msg_tid==clusters[i]->pinfo->hi_tid) hostname=clusters[i]->pinfo->hi_name;
		//printf("%d\n",clusters[i]->pinfo->hi_tid);
	}

	printf("recv from %s |%d: %d %d %d %d  (time:%d sec)\n",hostname,msg_tid,x1,y1,x2,y2,time);

	data=(char *)malloc(length);
	//printf("rec %d bytes from %d\n",length,slaveid);

	pvm_recv(msg_tid,21);
	pvm_bufinfo(pvm_getrbuf(),&j,(int*)0,&tid);
	pvm_upkbyte(data,length,1);			


	//Overread PPM-Header
	lc=0;i=0;
	while ((lc<3)&&(i<length)) {
		if (((*(data+i))==10)||((*(data+i))==13)) lc++;
		i++;
	}
	if (lc<3) {return;}
	buffer=(unsigned char*)(data+i);
	
	x=x1;
	y=y1;
        pixel_togo-=(x2-x1)*(y2-y1);

	t=0;
	for (y=y1;y<y2;y++) {
		for (x=x1;x<x2;x++) {
			r = buffer[t*3 ];
			g = buffer[t*3 + 1];
			b = buffer[t*3 + 2];
			qp->setPen(QColor(r,g,b));
			qp->drawPoint(x,y);
			t++;
		}
	}

	flush();
	//startTimer(150);

	if (pixel_togo<=0) {
		emit finishedRendering();
		stopRendering();
	}

	tmpstr.setNum((( (pixel-pixel_togo) * 100) / (pixel)));
	qstr = "Rendered ";
	qstr += tmpstr + "%";
	emit report(qstr);
}


#endif


int		ViewWidget::setName(const char *n)
{
	return view::setName(n);
}

void		ViewWidget::saveViewImage(QString fname,QString fmt)
{
	QString		qstr;
	
	if(qpm->save(fname,fmt))
	{
		qstr = "Saved " + fname + " as " + fmt;
		emit report(qstr);
	}
	else
	{
		qstr = "Failed saving " + fname + " as " + fmt;
		emit report(qstr);
	}
}

void		ViewWidget::reconfigure()
{
	configure();
	draw();
}

void		ViewWidget::configure()
{
	Preferences prefs;

	backcolor = prefs.getBackgroundColor(); 
	drawcolor = prefs.getLinesColor(); 
	selcolor = prefs.getSelectedLinesColor(); 
	boxcolor = prefs.getBoxColor(); 

	gridcolor = prefs.getGridColor(); 

	xaxecolor = prefs.getXAxisColor();
	yaxecolor = prefs.getYAxisColor();
	zaxecolor = prefs.getZAxisColor();

	letterxcolor = prefs.getLetterXColor();
	letterycolor = prefs.getLetterYColor();
	letterzcolor = prefs.getLetterZColor();

	blobcolor = prefs.getBlobColor();

	setAspectRatio(prefs.getAspectRatio());

	gridlines_xy = prefs.getGridLinesXY();
	gridlines_xz = prefs.getGridLinesXZ();
	gridlines_zy = prefs.getGridLinesZY();
	gridpoints_xy = prefs.getGridPointsXY();
	gridpoints_xz = prefs.getGridPointsXZ();
	gridpoints_zy = prefs.getGridPointsZY();
	gridspace_xy = prefs.getGridSpaceXY();
	gridspace_xz = prefs.getGridSpaceXZ();
	gridspace_zy = prefs.getGridSpaceZY();
	grid_xy = prefs.getGridXY();
	grid_xz = prefs.getGridXZ();
	grid_zy = prefs.getGridZY();
}

void		ViewWidget::setViewObject(base *b)
{
	setSubObject(b);
	draw();
}

void		ViewWidget::resize()
{
	resize(w,h);
}

void		ViewWidget::resize(int w,int h)
{
	QFrame::resize(w,h);
}


