/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** avi_load.c
*/
#include <sys/stat.h>

#include <stdlib.h>
#include <stdio.h>
#include <jpeglib.h>
#include <gtk/gtk.h>

#include "global.h"
#include "avi_load.h"

#define FCC(ch4) (ch4[0] | ch4[1]<<8 | ch4[2]<<16 | ch4[3] << 24)

/* Function Prototypes */

void my_jpeg_stdio_dest                (j_compress_ptr cinfo, guchar *outbuffer);

/* The number of bytes written to the file by the JPEG Compresser */

gint numbytes, use_quant_table;

/* Define a Huffman table */

void add_huff_table (j_decompress_ptr dinfo, JHUFF_TBL **htblptr, const UINT8 *bits, const UINT8 *val) {
  if (*htblptr == NULL)
    *htblptr = jpeg_alloc_huff_table((j_common_ptr) dinfo);

  memcpy((*htblptr)->bits, bits, sizeof((*htblptr)->bits));
  memcpy((*htblptr)->huffval, val, sizeof((*htblptr)->huffval));

  /* Initialize sent_table FALSE so table will be written to JPEG file. */
  (*htblptr)->sent_table = FALSE;
}

/* Set up the standard Huffman tables (cf. JPEG standard section K.3) */
/* IMPORTANT: these are only valid for 8-bit data precision! */

void std_huff_tables (j_decompress_ptr dinfo) {
  static const UINT8 bits_dc_luminance[17] =
    { /* 0-base */ 0, 0, 1, 5, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0 };
  static const UINT8 val_dc_luminance[] =
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 };
  
  static const UINT8 bits_dc_chrominance[17] =
    { /* 0-base */ 0, 0, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0 };
  static const UINT8 val_dc_chrominance[] =
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 };
  
  static const UINT8 bits_ac_luminance[17] =
    { /* 0-base */ 0, 0, 2, 1, 3, 3, 2, 4, 3, 5, 5, 4, 4, 0, 0, 1, 0x7d };
  static const UINT8 val_ac_luminance[] =
    { 0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12,
      0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07,
      0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
      0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0,
      0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16,
      0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
      0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
      0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
      0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
      0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
      0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
      0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
      0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
      0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
      0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
      0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4,
      0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
      0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea,
      0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
      0xf9, 0xfa };
  static const UINT8 bits_ac_chrominance[17] =
    { /* 0-base */ 0, 0, 2, 1, 2, 4, 4, 3, 4, 7, 5, 4, 4, 0, 1, 2, 0x77 };
  static const UINT8 val_ac_chrominance[] =
    { 0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21,
      0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71,
      0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
      0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0,
      0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34,
      0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26,
      0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
      0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
      0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
      0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
      0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
      0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96,
      0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5,
      0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
      0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
      0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2,
      0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
      0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
      0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
      0xf9, 0xfa };
  
  add_huff_table(dinfo, &dinfo->dc_huff_tbl_ptrs[0],
                 bits_dc_luminance, val_dc_luminance);
  add_huff_table(dinfo, &dinfo->ac_huff_tbl_ptrs[0],
                 bits_ac_luminance, val_ac_luminance);
  add_huff_table(dinfo, &dinfo->dc_huff_tbl_ptrs[1],
                 bits_dc_chrominance, val_dc_chrominance);
  add_huff_table(dinfo, &dinfo->ac_huff_tbl_ptrs[1],
                 bits_ac_chrominance, val_ac_chrominance);
}

void Decode_JPEG(FILE *inFile, guchar *outBuffer){
  int row_stride;
  int y;
  struct jpeg_decompress_struct dinfo;
  struct jpeg_error_mgr jerr;

  dinfo.err = jpeg_std_error(&jerr);
  jpeg_create_decompress(&dinfo);
  jpeg_stdio_src(&dinfo, inFile);
  jpeg_read_header(&dinfo, TRUE);
  if (dinfo.dc_huff_tbl_ptrs[0] == NULL){
    std_huff_tables(&dinfo);
  }
  /*  dinfo.out_color_space = JCS_YCbCr;*/
  dinfo.out_color_space = JCS_RGB;
  /* not as acurate as the original one but much faster */
  dinfo.dct_method = JDCT_IFAST;
  jpeg_start_decompress(&dinfo);
  row_stride= dinfo.output_width*dinfo.output_components;
  for (y = 0; y < dinfo.output_height; y++){
    jpeg_read_scanlines(&dinfo, (JSAMPARRAY) &outBuffer, 1);
    outBuffer += row_stride;
  }
  jpeg_destroy_decompress(&dinfo);
}


void Compress_JPEG(guchar *outbuffer, guchar *inBuffer, gint width, gint height) {
  int row_stride;
  int y;
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  unsigned char marker[60];
  int i;

  static const unsigned int qtable_0[] = {
          16,  11,  10,  16,  24,  40,  51,  61,
          12,  12,  14,  19,  26,  58,  60,  55,
          14,  13,  16,  24,  40,  57,  69,  56,
          14,  17,  22,  29,  51,  87,  80,  62,
          18,  22,  37,  56,  68, 109, 103,  77,
          24,  35,  55,  64,  81, 104, 113,  92,
          49,  64,  78,  87, 103, 121, 120, 101,
          72,  92,  95,  98, 112, 100, 103,  99};

  static const unsigned int qtable_1[] = {
          17,  18,  24,  47,  99,  99,  99,  99,
          18,  21,  26,  66,  99,  99,  99,  99,
          24,  26,  56,  99,  99,  99,  99,  99,
          47,  66,  99,  99,  99,  99,  99,  99,
          99,  99,  99,  99,  99,  99,  99,  99,
          99,  99,  99,  99,  99,  99,  99,  99,
          99,  99,  99,  99,  99,  99,  99,  99,
          99,  99,  99,  99,  99,  99,  99,  99};

  static const unsigned int qtable_blank[] = {
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1,
          1,  1,  1,  1,  1,  1,  1,  1};

  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&cinfo);
  my_jpeg_stdio_dest (&cinfo, outbuffer);

  cinfo.image_width = width;
  cinfo.image_height = height;
  cinfo.input_components = 3;
  cinfo.in_color_space = JCS_RGB;

  jpeg_set_defaults(&cinfo);
  jpeg_set_colorspace (&cinfo, JCS_YCbCr);
  jpeg_set_quality (&cinfo, 75, TRUE);

  cinfo.dc_huff_tbl_ptrs[0]->sent_table = TRUE;
  cinfo.dc_huff_tbl_ptrs[1]->sent_table = TRUE;
  cinfo.ac_huff_tbl_ptrs[0]->sent_table = TRUE;
  cinfo.ac_huff_tbl_ptrs[1]->sent_table = TRUE;

  cinfo.comp_info[0].component_id = 0;
  cinfo.comp_info[0].v_samp_factor = 1;
  cinfo.comp_info[1].component_id = 1;
  cinfo.comp_info[2].component_id = 2;

  cinfo.write_JFIF_header = FALSE;

  if (use_quant_table) {
    jpeg_add_quant_table (&cinfo, 0, qtable_0, 100, TRUE);
    jpeg_add_quant_table (&cinfo, 1, qtable_1, 100, TRUE);
  } else {
    jpeg_add_quant_table (&cinfo, 0, qtable_blank, 100, TRUE);
    jpeg_add_quant_table (&cinfo, 1, qtable_blank, 100, TRUE);
  }

  jpeg_start_compress(&cinfo, FALSE);

  i=0;
  marker[i++] = 'A';
  marker[i++] = 'V';
  marker[i++] = 'I';
  marker[i++] = '1';
  marker[i++] = 0;
  while (i<60)
    marker[i++] = 32;
  
  jpeg_write_marker (&cinfo, JPEG_APP0, marker, 60);

  i=0;
  while (i<60)
    marker[i++] = 0;

  jpeg_write_marker (&cinfo, JPEG_COM, marker, 60);

  row_stride= cinfo.image_width*cinfo.input_components;
  for (y = 0; y < cinfo.image_height; y++){
    jpeg_write_scanlines(&cinfo, (JSAMPARRAY) &inBuffer, 1);
    inBuffer += row_stride;
  }
  jpeg_finish_compress(&cinfo);
  jpeg_destroy_compress(&cinfo);
}

int get_fcc (FILE *fp) {
  int fourcca[4];
  
  fourcca[0] = getc(fp);
  fourcca[1] = getc(fp);
  fourcca[2] = getc(fp);
  fourcca[3] = getc(fp);
  
  return FCC(fourcca);
}

void block_handle (FILE *fp, gint fourcc, gint movnum) {
  if (fourcc == FCC("RIFF")) {
    fseek (fp,8,1);
  } else if (fourcc == FCC("LIST")) {
    fseek (fp,8,1);
  } else {
    fseek(fp,get_fcc(fp),1);
  }
}

gint avi_initialize_movie (gint movnum) {
  FILE *fp;
  gchar fourcca[4];
  gint exitval=0; 

  fp = fopen (movie_db[movnum].filename,"rb");

  if (fp != NULL) {
    while ((fourcca[0]=getc(fp)) != EOF) {
      fourcca[1] = getc(fp);
      fourcca[2] = getc(fp);
      fourcca[3] = getc(fp);
      
      if (FCC(fourcca) == FCC("avih")) {
	fseek (fp, 20, 1);
	movie_db[movnum].frames=get_fcc(fp);
	fseek (fp,12,1);
	movie_db[movnum].width=get_fcc(fp);
	movie_db[movnum].height=get_fcc(fp);
	fseek (fp,16,1);
      } else if (FCC(fourcca) == FCC("strh")) {
	fseek (fp, 8, 1);
	
	movie_db[movnum].compression[0]=getc(fp);
	movie_db[movnum].compression[1]=getc(fp);
	movie_db[movnum].compression[2]=getc(fp);
	movie_db[movnum].compression[3]=getc(fp);
	
	fseek (fp,48,1);
      } else {
	block_handle(fp,FCC(fourcca),movnum);
      }
    }
    
    fclose (fp);

    exitval = 1;
  } else {
    printf ("Unable to open movie file: %s\n",movie_db[movnum].filename);
  }

  return exitval;
}

void avi_load_frame (gint movnum, guchar *buffer, gint framenum) {
  FILE *fp;
  gint temp,i,x,y;

  fp = fopen (movie_db[movnum].filename,"rb");

  if (fp != NULL) {
    /* This (should) put us to the beginning of the image lists */
    
    while (framenum > 0) {
      if ((temp = get_fcc(fp)) == FCC("00dc") || temp == FCC("00db")) {
	framenum--;
      if (framenum > 0)
	fseek (fp, get_fcc(fp), 1);
      }
    }
    
    temp = get_fcc(fp);

    if (strcmp(movie_db[movnum].compression,"DIB ")==0 || 
	strcmp(movie_db[movnum].compression,"rgb ")==0) {
      for (y=movie_db[movnum].height-1; y>=0;y--) {
	for (x=0; x < movie_db[movnum].width; x++) {
	  for (i=0; i<3; i++) {
	    buffer[((y)*movie_db[movnum].width+x)*3+(2-i)] = getc(fp);
	}
	}
      }
    } else if (strcmp(movie_db[movnum].compression,"mjpg")==0) {
      Decode_JPEG(fp,buffer);
    }
  } else {
    printf ("Unable to open movie file: %s\n",movie_db[movnum].filename);
  }
}

void bfcc (FILE *fp, gint num) {
  gint i;

  for (i=0; i < num; i++)
    putc('0',fp);
}

/* These typedefs code is pretty much all gleaned from aviriff.h, do a net search if you want it */

typedef struct _riffchunk {
  gchar fcc[4];
  gint cb;
} RIFFCHUNK;

typedef struct _rifflist {
  gchar fcc[4];
  gint cb;
  gchar ListType[4];
} RIFFLIST;

#define ckidMAINAVIHEADER FCC('avih')
typedef struct _avimainheader {
  gchar fcc[4];
  gint cb;
  gint MicoSecPerFrame;
  gint MaxBytesPerSec;
  gint PaddingGranularity;
  gint Flags;
  
#define AVIF_HASINDEX 0x00000010
#define AVIF_MUSTUSEINDEX 0x00000020
#define AVIF_ISINTERLEAVED 0x00000100
#define AVIF_TRUSTCKTYPE 0x00000800
#define AVIF_WASCAPTUREFILE 0x00010000
#define AVIF_COPYRIGHTED 0x00020000

  gint TotalFrames;
  gint InitialFrames;
  gint Streams;
  gint SuggestedBufferSize;
  gint Widget;
  gint Height;
  gint Reserved[4];
} AVIMAINHEADER;

#define ckidSTREAMLIST FCC('strl')
#define ckidSTREAMHEADER FCC('strh')
typedef struct _avistreamheader {
  gchar fcc[4];
  gint cb;
  gchar Type[4];

#define streamtypeVIDEO FCC('vids')
#define streamtypeAUDIO FCC('auds')
#define streamtypeMIDI  FCC('mids')
#define streamtypeTEXT  FCC('txts')

  gchar Handler[4];
  gint Flags;

#define AVISF_DISABLED 0x00000001
#define AVISF_VIDEO_PALCHANGES 0x00010000

  gshort Priority;
  gshort Language;
  gint InitialFrames;
  gint Scale;
  gint Rate;
  gint Start;
  gint Length;
  gint SuggestedBufferSize;
  gint Quality;
  gint SampleSize;
  struct {
    gshort left;
    gshort top;
    gshort right;
    gshort bottom;
  } Frame;
} AVISTREAMHEADER;

#define ckidSTREAMFORMAT FCC('strf')

#define ckidAVIOLDINDEX FCC('idx1')

void put_fcc (FILE *fp, gint num) {
  putc (num,fp);
  putc (num >> 8,fp);
  putc (num >> 16,fp);
  putc (num >> 24,fp);
}

void put_tcc (FILE *fp, gint num) {
  putc (num,fp);
  putc (num >> 8,fp);
}

void avi_write_movie (FILE *fp, gint width_out, gint height_out, gint frames_out) {
  struct stat stbuf;
  int i;

  fp = fopen ("output.avi", "wb");

  if (fp != NULL) {
    stat ("tmp.avi",&stbuf);
    
    printf ("size was %d\n", (int) stbuf.st_size);
    
    fprintf (fp,"RIFF");
    put_fcc (fp, 12 + stbuf.st_size + 12 + sizeof(AVIMAINHEADER) + sizeof(AVISTREAMHEADER) + 
	     sizeof(RIFFLIST) + 68 + 8 + 4 + 
	     (2*1024 - (sizeof(AVIMAINHEADER) + sizeof(AVISTREAMHEADER)
			+ 68 + 8 /* For Stream Format */
			+ sizeof (RIFFLIST) /* For Stream List */
			+ sizeof (RIFFLIST) /* For Header list */
			+ 8 /* For JUNK fcc & size */
			+ sizeof (RIFFLIST) /* for File header (RIFF (size) AVI) */
			+ 12)) + 8 );
    fprintf (fp,"AVI ");
    
    fprintf (fp,"LIST");
    put_fcc (fp, sizeof(AVIMAINHEADER) + sizeof(AVISTREAMHEADER) + sizeof(RIFFLIST) + 68 + 8 + 4);
    fprintf (fp,"hdrl");
    
    fprintf (fp, "avih");
    put_fcc (fp, sizeof(AVIMAINHEADER)-8);
    put_fcc (fp, 66667);
    put_fcc (fp, 1719296);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    put_fcc (fp, frames_out);
    put_fcc (fp, 0);
    put_fcc (fp, 1);
    put_fcc (fp, 114620);
    put_fcc (fp, width_out);
    put_fcc (fp, height_out);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    
    fprintf (fp,"LIST");
    put_fcc (fp, sizeof(AVISTREAMHEADER)+68+8+4);
    fprintf (fp,"strl");
    
    fprintf (fp, "strh");
    put_fcc (fp, sizeof(AVISTREAMHEADER)-8);
    put_fcc (fp, FCC("vids"));
    put_fcc (fp, FCC("mjpg"));
    put_fcc (fp, 0);
    put_tcc (fp, 0);
    put_tcc (fp, 0);
    put_fcc (fp, 0);
    put_fcc (fp, 66667);
    put_fcc (fp, 1000000);
    put_fcc (fp, 0);
    put_fcc (fp, frames_out);
    put_fcc (fp, width_out*height_out*3);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    put_tcc (fp,0);
    put_tcc (fp,0);
    put_tcc (fp,0);
    put_tcc (fp,0);
    
    fprintf (fp, "strf");
    put_fcc (fp, 68);
    put_fcc (fp, 68);
    put_fcc (fp, width_out);
    put_fcc (fp, height_out);
    put_tcc (fp, 1);  /* Planes */
    put_tcc (fp, 24); /* Depth */
    put_fcc (fp, FCC("MJPG"));
    put_fcc (fp, width_out*height_out*3);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    put_fcc (fp, 0);
    
    put_fcc (fp, 44);
    put_fcc (fp, 24);
    put_fcc (fp, 0);
    put_fcc (fp, 2);
    put_fcc (fp, 8);
    put_fcc (fp, 2);
    put_fcc (fp, 1);
    
    fprintf (fp, "JUNK");
    put_fcc (fp, 2*1024 - (sizeof(AVIMAINHEADER) + sizeof(AVISTREAMHEADER)
			   + 68 + 8 /* For Stream Format */
			   + sizeof (RIFFLIST) /* For Stream List */
			   + sizeof (RIFFLIST) /* For Header list */
			   + 8 /* For JUNK fcc & size */
			   + sizeof (RIFFLIST) /* for File header (RIFF (size) AVI) */
			   + 12 /* To get the first frame aligned */));
    for (i=0; i < (2*1024 - (sizeof(AVIMAINHEADER) + sizeof(AVISTREAMHEADER)
			     + 68 + 8 /* For Stream Format */
			     + sizeof (RIFFLIST) /* For Stream List */
			     + sizeof (RIFFLIST) /* For Header list */
			     + 8 /* For JUNK fcc & size */
			     + sizeof (RIFFLIST) /* for File header (RIFF (size) AVI) */
			     + 12)); i++)
      putc(0,fp);
    
    fprintf (fp, "LIST");
    put_fcc (fp, stbuf.st_size+4);
    fprintf (fp, "movi");

    fclose(fp);

    system ("cat tmp.avi >> output.avi");
  } else {
    printf ("Unable to open movie file: output.avi\n");
  }
}

void avi_init () {
  use_quant_table =1;
}

void avi_write_frame (FILE *fp, guchar *buffer, gint width_out, gint height_out) {
  static guchar *outbuffer;
  int i;

  numbytes=0;

  outbuffer = g_malloc (width_out * height_out * 3 * sizeof(guchar));

  /*  printf ("(1)Buffer is %x and outbuffer is %x\n",buffer,outbuffer);*/

  Compress_JPEG (outbuffer, buffer, width_out, height_out);

  use_quant_table=0;

  /*  printf ("Bytes written was %d\n",numbytes+(numbytes%4));*/

  fprintf (fp, "00db");
  put_fcc (fp, numbytes+(numbytes%4));

  fwrite (outbuffer, 1, numbytes, fp);
  for (i=0; i < numbytes%4; i++)
    putc(0,fp);

  /*  printf ("(2)Buffer is %x and outbuffer is %x\n",buffer,outbuffer);*/
  g_free(outbuffer);
}

/* The following code shamelessly ripped from the IJG (independant jpeg groups)
 * libjpeg implementation.... it's basically jdatadst.c hacked up to return the
 * number of bytes of data it wrote */

typedef struct {
  struct jpeg_destination_mgr pub; /* public fields */

  guchar * outbuffer;               /* target stream */
  JOCTET * buffer;              /* start of buffer */
} my_destination_mgr;

typedef my_destination_mgr * my_dest_ptr;

#define OUTPUT_BUF_SIZE  4096   /* choose an efficiently fwrite'able size */
#define JFWRITE(file,buf,sizeofbuf) ((size_t) fwrite((const void *) (buf), (size_t) 1, (size_t) (sizeofbuf), (file)))

void init_destination (j_compress_ptr cinfo)
{
  my_dest_ptr dest = (my_dest_ptr) cinfo->dest;

  /* Allocate the output buffer --- it will be released when done with image */
  dest->buffer = (JOCTET *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_IMAGE,
                                  OUTPUT_BUF_SIZE * sizeof(JOCTET));
  dest->pub.next_output_byte = dest->buffer;
  dest->pub.free_in_buffer = OUTPUT_BUF_SIZE;
}

boolean empty_output_buffer (j_compress_ptr cinfo) {
  my_dest_ptr dest = (my_dest_ptr) cinfo->dest;

  /*  if (JFWRITE(dest->outfile, dest->buffer, OUTPUT_BUF_SIZE) != (size_t) OUTPUT_BUF_SIZE);*/
  memcpy ((dest->outbuffer+numbytes),dest->buffer,OUTPUT_BUF_SIZE);

  dest->pub.next_output_byte = dest->buffer;
  dest->pub.free_in_buffer = OUTPUT_BUF_SIZE;
  
  numbytes = numbytes + OUTPUT_BUF_SIZE;
  return TRUE;
}


void term_destination (j_compress_ptr cinfo) {
  my_dest_ptr dest = (my_dest_ptr) cinfo->dest;
  size_t datacount = OUTPUT_BUF_SIZE - dest->pub.free_in_buffer;

  if (datacount > 0) {
    memcpy ((dest->outbuffer+numbytes),dest->buffer,datacount);
    /*    if (JFWRITE(dest->outfile, dest->buffer, datacount) != datacount);*/
  }
  numbytes = numbytes + datacount;
}

void my_jpeg_stdio_dest (j_compress_ptr cinfo, guchar *outbuffer) {
  my_dest_ptr dest;

  /* The destination object is made permanent so that multiple JPEG images
   * can be written to the same file without re-executing jpeg_stdio_dest.
   * This makes it dangerous to use this manager and a different destination
   * manager serially with the same JPEG object, because their private object
   * sizes may be different.  Caveat programmer.
   */
  if (cinfo->dest == NULL) {    /* first time for this JPEG object? */
    cinfo->dest = (struct jpeg_destination_mgr *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT,
                                  sizeof(my_destination_mgr));
  }

  dest = (my_dest_ptr) cinfo->dest;
  dest->pub.init_destination = init_destination;
  dest->pub.empty_output_buffer = empty_output_buffer;
  dest->pub.term_destination = term_destination;
  dest->outbuffer = outbuffer;
}
