/***************************************************************************
                          hierarchyview.cpp  -  description                              
                             -------------------                                         
    begin                : Tue Mar 16 1999                                           
    copyright            : (C) 1999 by Richard Moore                         
    email                : rich@kde.org                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <qwhatsthis.h>
#include <kapp.h>
#include <kiconloader.h>
#include "hierarchyview.h"
#include "kguidoc.h"
#include "component.h"
#include "global.h"

#define MINIICON(x) kapp->getIconLoader()->loadMiniIcon(x)

const int ID_POPUP_INSERT= 1;
const int ID_POPUP_DELETE= 2;
const int ID_POPUP_REFRESH= 3;
const int ID_POPUP_PROPERTIES= 4;

HierarchyView::HierarchyView(QWidget *parent, const char *name)
	: QListView(parent, name),
		map()
{
	addColumn("Widget");
	addColumn("Type");
	setSorting(-1); // Don't sort
	
	connect( this,	SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int) ),
									SLOT(showPopup(QListViewItem *, const QPoint &) ) );

	setRootIsDecorated(true);
	setFocusPolicy(QWidget::StrongFocus);

	QWhatsThis::add(this, i18n("Provides a tree view of the widget hierarchy of the forms in your application.") );
}

HierarchyView::~HierarchyView()
{

}

void HierarchyView::setDocument(KGUIDoc *doc)
{
	ASSERT(doc);
	
	document_= doc;
	setRoot(doc->rootInstance());
}

void HierarchyView::setRoot(Instance *root)
{
	ASSERT(root);

	this->root= root;
	refresh();
}

// SLOT
void HierarchyView::refresh()
{
	clear();

	QListViewItem *allForms= new QListViewItem(this);
	CHECK_PTR(allForms);
	
	allForms->setText(0, "All forms");
	allForms->setText(1, "Contains every form in the current project");
	
	QListViewItem *item= new QListViewItem(allForms);
	item->setText(0, root->name());
	item->setPixmap(0, root->component()->pixmap());

	item->setText(1, root->component()->name());

	map.clear();
	map.insert(item, root);
	buildTree(root, item);
	
	allForms->setOpen(true);
}

// SLOT
void HierarchyView::refresh(Instance *root)
{
	// Delete the old item for root
	QPtrDictIterator<Instance> it( map );
	
	QListViewItem *old= 0;
	while ( it.current() ) {
		if (it.current() == root) {
			old= (QListViewItem *) it.currentKey();
		}
		++it;
	}

	if (old != 0) {
		QListViewItem *parent= (QListViewItem *) old->parent();
		
		if (parent == 0) {
			refresh();
		}
		else {
			bool open= old->isOpen();
			QListViewItem *preceding= old->itemAbove();
			map.remove(old);
			delete old;
			
			// Create a new item for root
			QListViewItem *item;
			if (preceding) {
				item= new QListViewItem(parent, preceding);
			}
			else {
				item= new QListViewItem(parent);			
			}
			item->setText(0, root->name());
			item->setPixmap(0, root->component()->pixmap());

			item->setText(1, root->component()->name());

			// Replace old view of root with a new one
			buildTree(root, item);
			
			item->setOpen(open);
			map.insert(item, root);
		}
	}
	else {
		warning("Could not find view item for root %p", root);
		refresh();
	}
}

void HierarchyView::buildTree(Instance *root, QListViewItem *parent)
{
	QListViewItem *preceding= 0;

	for (uint i= 0; i < root->childCount(); i++) {
		// Handle the top child
		Instance *child= root->getChild(i);
		QListViewItem *item= new QListViewItem(parent, preceding);
		item->setText(0, child->name());

		QPixmap p= child->component()->pixmap();
		item->setPixmap(0, p);

		item->setText(1, child->component()->name());

		// Add to the map
		map.insert(item, child);

		// Then recurse
		buildTree(child, item);
		
		preceding= item;
	}
}

Instance *HierarchyView::findInstance(QListViewItem *i)
{
	return map[i];
}

// SLOT
void HierarchyView::showPopup(QListViewItem *item, const QPoint &pos)
{
	if (item && map[item]) {
		item->listView()->setSelected(item, true);
	
		QPopupMenu *rmbMenu= new QPopupMenu();
		CHECK_PTR(rmbMenu);

		QPopupMenu *newWidgetMenu= Global::getComponentManager()->createInsertPopup(rmbMenu);
	 	rmbMenu->insertItem(i18n("&Insert"), newWidgetMenu, ID_POPUP_INSERT );
	 	
	 	rmbMenu->insertItem(i18n("&Remove..."), ID_POPUP_DELETE );
	 	
	 	// You can't delete the root item
	 	if ( findInstance(item) == root ) {
	 		rmbMenu->setItemEnabled(ID_POPUP_DELETE, false);
		}
		else {
	 		rmbMenu->setItemEnabled(ID_POPUP_DELETE, true);
		}
		
	 	rmbMenu->insertItem(i18n("Re&fresh"), ID_POPUP_REFRESH );
	 	rmbMenu->insertItem(i18n("&Properties..."), ID_POPUP_PROPERTIES );
	 	
	 	int result= rmbMenu->exec(pos);
	 		
	 	warning("Selected popup item %d", result);

	 	// Process the result	 	
	 	if (result == -1) {
 			// Cancelled 	
	 	}
		else if (result == ID_POPUP_DELETE) {
	 		deleteInstance(findInstance(item));
	 	}
	 	else if (result == ID_POPUP_REFRESH) {
			document_->refreshViews();
	 	}
	 	else if (result == ID_POPUP_PROPERTIES) {
 			emit showProperties(findInstance(item)); 	
	 	}
	 	else {
 			// Might be an insert command
 			Component *comp= Global::getComponentManager()->component(result);
 			if (comp) {
				document_->insert(comp);
 			}
	 	}
	 		
		delete rmbMenu;
	}
}

void HierarchyView::deleteInstance(Instance *toDie)
{
	ASSERT(toDie != root);
	
	warning("Instance %p must die!", toDie);
	
	Instance *parent= toDie->parent();
	int i= parent->findChild(toDie);
	
	warning("Find result %d", i);
		
	// Ensure the properties dlg isn't looking this way, this should
	// move into the document class
	emit showProperties(parent);
	
	if (i != -1)
		document_->remove(parent, i);	
}







































