/*
 * kmoledit.cpp
 *
 * Copyright (C) 1999 Tomislav Gountchev <tomi@socrates.berkeley.edu>
 */

/**
 * The built in editor used to define / undefine groups and elements.
 */


#include "kmoledit.h"

KMolEdit::KMolEdit (QWidget *parent, const char* name, KMolCalc* kmolcalc) 
  : QDialog (0, name, TRUE) {

  // Error messages:
  NAMES = QString(i18n("Names must start with a letter,\n followed by any number of lowercase letters."));

  this->kmolcalc = kmolcalc;
  setCaption (i18n("Kmol Group Definitions"));
  setFixedSize (220, 190);

  // formula entry field
  formula = new QLineEdit (this);
  formula->setGeometry (10, 30, 200, 30);
  formula->setFont(kapp->fixedFont);
  formula->setFocus();
  
  QLabel* formula_l = new QLabel (i18n("Symbol"), this, "grouplabel");
  formula_l->setFont(kapp->generalFont);
  formula_l->setGeometry (formula->x(), formula->y()-20, 100, 18);

  // molecular weight or group definition entry field
  definition = new QLineEdit (this);
  definition->setGeometry (10, 90, 200, 30);
  definition->setFont(kapp->fixedFont);

  
  QLabel* definition_l = new QLabel (i18n("Formula or Weight"), 
				     this, "deflabel");
  definition_l->setFont(kapp->generalFont);
  definition_l->setGeometry (definition->x(), definition->y()-20, 200, 18);

  QPushButton* define = new QPushButton (i18n ("Define"), this, "define");
  define->setGeometry(10, 150, 60, 30);
  define->setFont(kapp->generalFont);
  connect (define, SIGNAL(clicked()), this, SLOT(edit()));
  connect (definition, SIGNAL(returnPressed()), define, SLOT(animateClick()));

  QPushButton* ok = new QPushButton (i18n ("Done"), this, "done");
  ok->setGeometry(80, 150, 60, 30);
  ok->setFont(kapp->generalFont);
  connect (ok, SIGNAL(clicked()), this, SLOT(save()));

  QPushButton* cancel = new QPushButton (i18n ("Cancel"), this, "cancel");
  cancel->setGeometry (150, 150, 60, 30);
  cancel->setFont(kapp->generalFont);
  connect (cancel, SIGNAL(clicked()), this, SLOT(undo()));

}
  
/**
 * Parse the user input from this->formula and this->definition. Check syntax,
 * check definition and call the corresponding kmolcalc function to define element
 * or group.
 */
QString* KMolEdit::edit() {
  QString name(formula->text());
  if (name.isEmpty()) return 0;
  if ((name[0] < 'A' || name[0] > 'Z') && (name[0] < 'a' || name[0] > 'z')) 
    { // Not a valid syntax
      error(NAMES);
      return new QString (i18n("Bad formula."));
    }
  for (uint i = 1; i < name.length(); i++)
    if ((name[i] < 'a' || name[i] > 'z') && name[i] != '*' && name[i] != '\'')
      {
	error(NAMES);
	return new QString (i18n("Bad formula."));
      }
  QString def(definition->text());
  if (def.isEmpty()) {
    kmolcalc->undefineGroup(name); // empty definition field used to erase records!!
    formula->setFocus();
    return 0;
  }
  bool ok = TRUE;
  double weight = def.toDouble(&ok);
  if (ok) { // element definition requested, weight entered
    kmolcalc->defineElement(name, weight);
  }
  else { // group definition requested, formula entered
    QString* err = kmolcalc->defineGroup(name, def); // try to parse and define formula
    if (err != 0) {
      error (*err);
      return err;
    }
  }
  formula->setFocus();
  return 0;
}

// show error message
void KMolEdit::error(QString mess) {
  QMessageBox::information (0, i18n("Error"), mess, QMessageBox::Ok);
}

 
/** 
 * Save changes to definition file and close editor window.
 */
void KMolEdit::save() {
  if (edit() !=0) return;
  kmolcalc->writeElstable();
  kmolcalc->readElstable();
  done(0);
}

/**
 * Undo changes not yet saved by re-reading definition file.
 */
void KMolEdit::undo() {
  kmolcalc->readElstable();
  done(0);
}

