/* 
 * kmolwidget.cpp 
 *
 * Copyright (C) 1999 Tomislav Gountchev <tomi@socrates.berkeley.edu>
 */

#include <qpushbutton.h>
#include <qlineedit.h>
#include <qmultilinedit.h>
#include <qlabel.h>
#include "kmolwidget.h"



/**
 * This class is the main view for KMol. It handles user input, passes it to KMOLCALC and
 * displays the result.
 *
 * @short Main view
 * @author Tomislav Gountchev <tomi@socrates.berkeley.edu>
 * @version 0.2
 */
KMolWidget::KMolWidget (QWidget *parent, const char* name) 
  : QWidget (parent) {
  
  setFixedSize (220, 300);

  //accepts text input until return is pressed 
  formula = new QLineEdit (this);
  formula->setGeometry (10, 30, 200, 30);
  formula->setFont(kapp->fixedFont);
  formula->setFocus();
  connect(formula, SIGNAL(returnPressed()), this, SLOT(calc()));

  QLabel* formula_l = new QLabel (i18n("Formula"), this, "formula");
  formula_l->setFont(kapp->generalFont);
  formula_l->setGeometry (formula->x(), formula->y()-20, 100, 18);

  //mol weight output field
  result = new QLabel (this);
  result->setGeometry (10, 140, 200, 30);
  result->setFont(kapp->fixedFont);
  result->setFrameStyle (QFrame::Panel | QFrame::Sunken);
  result->setLineWidth (2);
  result->setBackgroundMode(PaletteBase);

  QLabel* result_l = new QLabel (i18n("Mw"), this, "result");
  result_l->setFont(kapp->generalFont);
  result_l->setGeometry (result->x(), result->y()-20, 100, 18);

  //calculate button
  QPushButton* calculate = new QPushButton (i18n("Calculate"), 
					    this, "calculate");
  calculate->setGeometry (10, 80, 95, 30);
  calculate->setFont(kapp->generalFont);
  calculate->setDefault(TRUE);
  connect(calculate, SIGNAL(clicked()), this, SLOT(calc()));
  connect(formula, SIGNAL(returnPressed()), calculate, SLOT(animateClick()));

  //clears FORMULA and RESULT fields
  QPushButton* clear_fields = new QPushButton (i18n("Clear"), this, "clear");
  clear_fields->setGeometry (115, 80, 95, 30);
  clear_fields->setFont(kapp->generalFont);
  connect(clear_fields, SIGNAL(clicked()), this, SLOT(clear()));


  //E.A. display field
  anal_display = new QMultiLineEdit (this, "display");
  anal_display->setGeometry (10, 200, 200, 90);
  anal_display->setFont(kapp->fixedFont);
  anal_display->setReadOnly(1);
  anal_display->setFocusPolicy(QWidget::NoFocus);

  QLabel* anal_display_l = 
    new QLabel (i18n("Elemental Composition (%)"), this, "displaylabel");
  anal_display_l->setFont(kapp->generalFont);
  anal_display_l->setGeometry 
    (anal_display->x(), anal_display->y()-20, 290, 18);

  kmolcalc = new KMolCalc;

}

KMolWidget::~KMolWidget() {
  delete kmolcalc;
}


/**  
 * Make a call to KMOLCALC to parse the string in FORMULA, if any, and get Mw and EA back.
 */
void KMolWidget::calc() {
  QString compound(formula->text());
  if (compound.isEmpty()) {
    clear();
    return;
  }
  QString* errors = kmolcalc->readFormula(compound);
  QString mw, ea;
  double weight = kmolcalc->getWeight();
  if (errors == NULL && weight > 0) {
    mw.setNum(weight);
    ea = kmolcalc->getEmpFormula() + " :\n" + kmolcalc->getEA();
  } else {
    mw = "???";
    ea = i18n("ERROR: \n") + *errors + "\n";
  }
  result->setText(mw);
  anal_display->setText(ea);
}

/**
 * Call the built in editor to edit the definitions file.
 */
void KMolWidget::callEditor() {
  KMolEdit* kmoledit = new KMolEdit(0, "kmoledit", kmolcalc);
  kmoledit->exec();
  delete kmoledit;
}


/** 
 * Clear all text entry / result fields.
 */
void KMolWidget::clear() {
  formula->clear();
  result->clear();
  anal_display->clear();
}


