/*
 * Implements the "Read audio CD" Tab
 *
 * $Id: kraudiocd.cpp,v 1.3 1998/08/05 16:36:03 core Exp $
 *
 */

#include <sys/stat.h>
#include <unistd.h>

#include "kraudiocd.h"
#include "kraudiocd.moc"

#include <qmsgbox.h>
#include <qtimer.h>

#define tmpFilename "cdda.tmp.wav"


KReadAudioCd::KReadAudioCd( QWidget *parent=0, KTrackListBox *aBufListBox=0 )
  : QGroupBox( parent, "ReadAudioCd" )
{
  autostartReading = true;
  bufListBox = aBufListBox;

  // init read queue;
  queueHead = NULL;
  queueTail = NULL;
  queueCount = 0;

  maxQueued = read = 0;
  maxQueuedSize = readSize = 0;
  tmpFileInfo = 0L;

  config = kapp->getConfig();
  config->setGroup("general");
  
  cdrom = new Cdrom( config->readEntry( "ReadingDev", "/dev/cdrom") );

  QSize size;             // we'll need it anyway....
  
  QBoxLayout *layout0 = new QHBoxLayout( this, 8, 3 );  // is TopLevelLayout
  
  /////////////////////////////////////////////////////////////////
  // left side....

  QBoxLayout *layout1 = new QVBoxLayout( this );
  layout0->addLayout( layout1 );
  
  cdListBox = new KTrackListBox( this, "AudiCdListBox", 3);
  cdListBox->allowDrag( TRUE );
  cdListBox->setColumn( 0, klocale->translate("Title"), 120);
  cdListBox->setColumn( 1, klocale->translate("Time"), 50);
  cdListBox->setColumn( 2, klocale->translate("Size"), 35);
  cdListBox->setSeparator('\t');
  cdListBox->setDragSource( 1 );
  layout1->addWidget( cdListBox );
  connect( cdListBox, SIGNAL( dragEnded( int ) ), this, SLOT( dragEnded( int ) ) );
  


  ////////////////////////////////////////////////////////////////////////////
  //  right side.....

  QGroupBox *right = new QGroupBox( this, "right" );
  
  layout1 = new QVBoxLayout( right );
  layout0->addWidget( right );
  
  // add a "control"-panel
  playPanel = new CdPlayPanel( cdrom, right );
  size = playPanel->sizeHint();
  layout1->addSpacing( 5 );
  layout1->addWidget( playPanel );
  connect( playPanel, SIGNAL( skip( int ) ), this, SLOT( trackSkip( int ) ) );
  connect( playPanel, SIGNAL( statusChange( CdPanelStatus ) ), this, SLOT( cdStatusChange( CdPlayPanel::CdPanelStatus ) ) );

  // add bufferOperation buttons
  QPushButton *pbt = new QPushButton( "<-", right );
  pbt->setGeometry( 7, 180, 61, 30 );
  connect( pbt, SIGNAL( clicked() ), this, SLOT( pbtAddTracks() ) );
  pbt = new QPushButton( "<<-", right );
  pbt->setGeometry( 68, 180, 61, 30 );
  connect( pbt, SIGNAL( clicked() ), this, SLOT( pbtAddAllTracks() ) );

  layout1->addSpacing( 47 );
  // the start/stop button
  pbtStartReading = new QPushButton( klocale->translate("Start reading"), right );
  pbtStartReading->setFixedWidth( size.width() );
  layout1->addWidget( pbtStartReading );
  layout1->addSpacing( 10 );
  connect( pbtStartReading, SIGNAL( clicked() ), this, SLOT( pbtStartReadingSlot() ) );

  right->setFixedWidth( size.width()+12 );

  proc = new KShellProcess();
  newCd();
}

KReadAudioCd::~KReadAudioCd()
{
  delete cdrom;
  delete proc;
}

void KReadAudioCd::dragEnded( int index )
{
}

void KReadAudioCd::timerEvent( QTimerEvent *e )
{
  updateStatus();
}

void KReadAudioCd::newCd()
{
  int i, listCount;

  listCount = cdListBox->count();
  for ( i=listCount-1; i>=0; i-- ) {
    delete cdListBox->track( i );
    cdListBox->removeTrack( i );
  };
    
  listCount = cdrom->trackCount();
  cdListBox->setAutoUpdate( false );
  sleep( 1 );
  for ( i = 1; i <=listCount; i++ ) {
    CdTrackInfo *track = new CdTrackInfo();
    cdrom->trackInfo( i, track );
    cdListBox->insertTrack( track );
  };
  cdListBox->setAutoUpdate( true );
  cdListBox->repaint();
}

void KReadAudioCd::trackSkip( int track )
{
  cdListBox->setAutoUpdate( false );
  cdListBox->unmarkAll();
  cdListBox->setCurrentItem( track );
  cdListBox->setAutoUpdate( true );
  cdListBox->repaint();
}

int KReadAudioCd::queueTrack( CdTrackInfo *track )
{
  CdTrackInfo *trackInfo = new CdTrackInfo( track ); // clone track
  
  RQueueItem *item = new RQueueItem;
  item->track = trackInfo;
  item->next = NULL;
  if ( queueTail != NULL ) {
    queueTail->next = item;
  };
  queueTail =  item;
  if ( queueHead == NULL ) {
    queueHead = item;
    maxQueued = 0;
    maxQueuedSize = 0;
    read = 0;
    readSize = 0;
  };
  queueCount++;

  if ( autostartReading && !proc->isRunning() ) startReading( true );

  maxQueued++;
  maxQueuedSize += queueTail->track->trackSize();
  emit queueChanged();
  updateStatus();
  return queueCount;
}

int KReadAudioCd::dequeueTrack()
{
  RQueueItem *item;
  
  if ( queueCount > 0 ) {
    item = queueHead;

    queueHead = queueHead->next;
    if ( queueHead == NULL ) {
      queueTail = NULL;
    };

    delete item->track;
    delete item;
    queueCount--;
  }
  
  return queueCount;
}

void KReadAudioCd::startReading( bool start )
{
  if ( start ) {
    if ( queueCount > 0 ) readNextTrack();
  } else {
    if ( proc->isRunning() ) proc->kill();
  };
}

void KReadAudioCd::readNextTrack()
{
  QString tmp, tmp2, options;;

  if ( queueCount > 0 ) {
    delete proc;                 // there seems to be a bug in KProcess:
    proc = new KShellProcess();  //  cant be used more than 3 times - to build a new one each call. 
    proc->clearArguments();

    switch ( config->readNumEntry("ReadingTool", 0) )
      {
      case 0:                    // use cdparanoia
	execute = config->readEntry( "ReadingToolPath", "cd-paranoia-not-configured" );
	execute = execute + " " + config->readEntry( "ReadingTool0Options", "" );
	tmp.sprintf(" %d ", queueHead->track->track() );
	execute = execute + tmp;
	execute = execute + config->readEntry( "BufferPath", "/tmp/" ) + tmpFilename;
	debug ( execute );
	*proc << execute;
	break;
      }
    connect( proc, SIGNAL(processExited( KProcess * )), this, SLOT( childExited( KProcess * )) );
    proc->start();
    pbtStartReading->setText( klocale->translate( "Stop reading" ) );
    tmpFileInfo = new QFileInfo( config->readEntry( "BufferPath", "/tmp/" ) + tmpFilename );
    tmpFileInfo->setCaching( false );
    updateStatus();
    startTimer( 500 );
    emit setFilenameInBuff( queueHead->track, "-= working =-" );
  } else {
    emit updateStatus( klocale->translate( "reading completed" ), -1 );
  };
}

void KReadAudioCd::childExited( KProcess *aProc )
{
  QString tmp;
  struct stat file_stat;
  int exitStatus;

  killTimers();

  if ( proc->normalExit() ) {
    exitStatus = proc->exitStatus();
    if ( exitStatus == 0 ) {
      QString filename;
      int i, err;
      
      i = 0;
      do {
	i++;
	filename.sprintf( config->readEntry( "BufferPath", "/tmp/" )+"audio%d.wav", i );
	err = stat( filename.data() , &file_stat );
      } while ( (err == 0) );
      delete tmpFileInfo;
      tmpFileInfo = 0L;
      rename( config->readEntry( "BufferPath", "/tmp/" )+tmpFilename, filename.data() );
      
      emit setFilenameInBuff( queueHead->track, filename );
      emit saveBuffer();
      
      read++;
      readSize += queueHead->track->trackSize();

      dequeueTrack();
      if ( queueCount == 0 ) {
	pbtStartReading->setText( klocale->translate( "Start reading" ) );
	emit updateStatus( klocale->translate( "reading completed" ), -1 );
      } else {
	readNextTrack();
      }
    } else {
      pbtStartReading->setText( klocale->translate( "Start reading" ) );
      emit setFilenameInBuff( queueHead->track, " " );
      tmp.sprintf( klocale->translate( "Error:\n\n  \"%s\" exited with error code %d" ), execute.data(), exitStatus );
      QMessageBox::warning( this, "kcdwrite", tmp );
    };
  } else {          // process has been KILLED !  [ somebody hit "Stop" - Button ? ]
    pbtStartReading->setText( klocale->translate( "Start reading" ) ) ;
    emit updateStatus( klocale->translate( "CDDA - reading aborted !" ), -1 );
    emit setFilenameInBuff( queueHead->track, " " );
  }
}

void KReadAudioCd::pbtAddTracks()
{
  int i, listCount;
  bool displayWarning;
  QString warning, tmp;

  warning = klocale->translate( "Please note following errors:\n" );
  displayWarning = false;

  listCount = cdListBox->count();
  for ( i=0; i < listCount; i++ ) {
    if ( cdListBox->isMarked( i ) ) {
      if ( bufListBox->pos( cdListBox->track( i ) ) == -1 )  // check for dublicate entries in buffer
	{
	  emit addTrackToBuff( cdListBox->track( i ), cdListBox->title( i ), " " );
	} else {
	  warning += tmp.sprintf( klocale->translate( "  %s already in buffer - not added\n" ), 
				  cdListBox->title( i ).data() );
	  displayWarning = true;	
	};
    };
  }
  if ( displayWarning ) {
    QMessageBox::warning( this, klocale->translate( "kcdwrite: warning" ),
			  warning, klocale->translate( "&OK" ) );			  
  };
}

void KReadAudioCd::pbtAddAllTracks()
{
  int i, listCount;

  cdListBox->setAutoUpdate( false );
  listCount = cdListBox->count();
  for ( i=0; i < listCount; i++ ) {
    cdListBox->markItem( i );
  }
  pbtAddTracks();
  cdListBox->unmarkAll();
  cdListBox->setAutoUpdate( true );
}

void KReadAudioCd::pbtStartReadingSlot()
{
  if ( proc->isRunning() ) {
    startReading( false ); // stop reading !
  } else {
    startReading( true );
  };
}

void KReadAudioCd::updateStatus()
{
  QString tmp;
  int i;

  
  if ( proc->isRunning() ) {
    tmp.sprintf( klocale->translate( "reading track %d/%d..." ), read+1, maxQueued );
    
    i = -1;
    if ( ( tmpFileInfo != 0L ) && ( maxQueuedSize > 0 ) ) {
      i = ( readSize*120 + tmpFileInfo->size()/10  ) / ( maxQueuedSize * 12 / 10 ) + 1;
    };

    emit updateStatus( tmp, i );
  }
}

bool KReadAudioCd::quitOk()
{
  if ( proc->isRunning() && 
       ( QMessageBox::warning( this, "kcdwrite", klocale->translate( "Currently reading audio tracks - Really quit ?" ),
			       klocale->translate( "&Yes" ), klocale->translate( "&No" ), 0, 1, 1 ) == 1 ) )
    {  
      return false;
    } else {
      return true;
    }
}

void KReadAudioCd::buffDeleteTrack( CdTrackInfo *track )
{
  RQueueItem *lastItem, *item;

  if ( queueCount > 0 ) {
    if ( queueHead->track->trackId() == track->trackId() ) {
      if ( proc->isRunning() ) {
	proc->kill();
      };

      maxQueued--;
      maxQueuedSize -= queueHead->track->trackSize();
      dequeueTrack();
    } else {
      lastItem = queueHead;
      item = queueHead->next;;
      while ( item != NULL ) {
	if ( item->track->trackId() == track->trackId() ) {
	  lastItem->next = item->next;
	  if ( lastItem->next == NULL ) { queueTail = lastItem; };
	  queueCount--;
	  maxQueued--;
	  maxQueuedSize -= item->track->trackSize();
	  delete item->track;
	  delete item;
	  item = lastItem->next;
	} else {
	  lastItem = item;
	  item = item->next;
	};
      }
      updateStatus();
    }
  }
}

void KReadAudioCd::cdStatusChange( CdPlayPanel::CdPanelStatus status )
{
  int i, listCount;

  debug( "cdStatusChange !" );

  switch ( status ) {
  case CdPlayPanel::Ready:
    emit updateStatus( klocale->translate( "Reading source cd directory" ), -1 );
    QTimer::singleShot( 1000, this, SLOT( newCd() ) );
    break;
  case CdPlayPanel::NotReady:
    if ( proc->isRunning() ) {
      KDEBUG( KDEBUG_ERROR, 0, "Huh ! - ReadingTool running while cd not ready !" );
      startReading( false ); // stop NOW !
    };

    emit updateStatus( klocale->translate( "Reading drive not ready: no cd in drive ?" ), -1 );

    cdListBox->setAutoUpdate( false );
    cdListBox->unmarkAll();
    listCount = cdListBox->count();
    for ( i=listCount-1; i>=0; i-- ) {
      delete cdListBox->track( i );
      cdListBox->removeTrack( i );
    }

    cdListBox->setAutoUpdate( true );
    cdListBox->repaint();
    
    i = dequeueTrack();
    while ( i != 0 ) i = dequeueTrack();
  };
}
