	// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#ifndef AUDIOFILE_INCLUDED
#define AUDIOFILE_INCLUDED

#include <qobject.h>


/** 
    @short AudioFile - container class that holds audio file information
    @author Alexander Feigl <Alexander.Feigl@gmx.de>

    The AudioFile class ist a container class which holds all information neccessary to 
    access audio files. To setup a AudioFile you probably want to create a AudioFileOp
    child class which initalizes the AudioFile class with correct information. Other
    AudioFileOp childs are used to modify the AudioFile, to create images or similar 
    operations.
*/

class AudioFile : public QObject
 {
  Q_OBJECT;

  public:

/** 
    <pre>
    Byte endian enum
    Endian_Little : the file is in litte endian format
    Endian_Big : the file is in big endian format
    </pre>
*/
    enum Sample_Endian{Endian_Little,Endian_Big};

/**
    <pre>
    Sample type enum
    Samples_PCM : samples are in PCM format
    </pre>
*/
    enum Sample_Format{Samples_PCM};

/**
    <pre>
    Sample size enum
    Samples_8Bit : sample size is 8 bit
    Samples_16Bit : sample size is 16 bit
    Samples_32Bit : sample size is 32 bit
    </pre>
*/
    enum Sample_Size{Samples_8Bit,Samples_16Bit,Samples_32Bit};

/**
    This method returns the duration (in 1/75 seconds) of the selected part of the audio file

    @return length of selected part in 1/75 seconds
*/
    int getSelectedDuration(void);

/**
    This methods returns the size of the whole audio file (in 1/75 seconds)

    @return length of whole audio file in 1/75 seconds
*/
    int getAbsoluteDuration(void);

/**
    This method returns the balance setting of an audio file

    @return balance (float from -1 to 1)
*/
    float getBalance(void);

/**
    This method changes the balance setting of an audio file

    @param balance floating point number from -1 to 1 
*/
    void setBalance(float balance);

    long int getStartPos(void);
    long int getEndPos(void);
    void setStartPos(long int pos);
    void setEndPos(long int pos);

/**
    This method returns the boost setting of an audio file

    @return boost factor
*/
    float getBoost(void);

/**
    This method changes the boost factor of an audio file

    @param boost floating point boost factor
*/
    void setBoost(float boost);

/**
    This method returns the endianess of the audio file

    @return endianess enumeration 
*/
    enum Sample_Endian   getEndian(void);

/**
    This method returns the size of the samples of the audio file

    @return sample size enumeration
*/
    enum Sample_Size     getSampleSize(void);

/**
    This method returns the number of channels of the audio file

    @return number of channels
*/
    int                  getChannels(void);

/**
    This method returns the sampling rate of the audio file

    @return sampling rate
*/
    long int             getSampleRate(void);

/**
    This method returns the file format of the audio file. This file format is initialized by the 
    AudioFileFormat class at initalization time. The method copies this information to a 
    user-supplied char array

    @param formatstring char array which will be filled with the file format name
*/
    void                 getFormat(char *formatstring);

/**
    This method returns whether this audio file needs to be converted to CD-R before it can be
    written.
   
    @return  the audio file needs a temporary image
*/
    int                  needImage(void);

  protected:

/**
    sample endian
*/
    enum Sample_Endian           SampleEndian;

/**
   sample format (PCM)
*/
    enum Sample_Format           SampleFormat;

/**
    sample channels
*/
    int                          SampleChannels;

/**
    size of samples
*/
    enum Sample_Size             SampleSize;

/**
    sampling rate
*/
    long int                     SampleRate;

/**
    nuber of bytes of one sample (for all channels)
*/
    int                          SampleBytes;

/**
    file offset where the sample data starts
*/ 
    long int                     DataPos;

/**
    number of bytes of the sample date body 
*/
    long int                     DataSize; 

/**
    number of samples of the sample data body
*/
    long int                     DataSamples;

/**
    sample offset where the selection starts
*/
    long int                     SelectedStart;

/**
    sample offset where the selection ends
*/
    long int                     SelectedEnd;

/**
    boost factor
*/
    float                        BoostFactor;

/**
    balance
*/
    float                        Balance;

/**
    cached maximum boost factor - this is initalized by AudioScanImage 
*/
    float                        MaxBoost;

/**
    character string of the file format name
*/
    char                         Format[16];

/**
    file name of the audio file
*/
    char                         FileName[1024];

    friend class AudioFileOp;
    friend class AudioFileConvert;
    friend class AudioScanImage;
    friend class AudioPlayImage;
    friend class AudioBuildImage;
    friend class AudioFileFormat;
};

#endif
