// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "ProcessInterface.h"
#include "ProgressDialog.h"
#include "uidcontrol.h"
#include "Fork.h"

#include <klocale.h>
#include <kprocess.h>

#include <qtimer.h>

#include <stdio.h>
#include <string.h>

extern KLocale *locale;

ProcessInterface::~ProcessInterface(void)
 {
   if (process!=0) closeProcess();
 }

ProcessInterface::ProcessInterface(void)
 {
  forkProcess=false;
  forkAudio=false;
  blockTimer=false;
  process=0;
  progressDialog=0;
  outPending=false;
  errPending=false;
  outTimed=0;
  errTimed=0;
  lineTimeout=5;
 }

ProcessInterface::ProcessInterface(int (*fx)(int argc,char **argv))
 {
  forkProcess=true;
  forkAudio=false;
  blockTimer=false;
  process=0;
  progressDialog=0;
  forkFunc=fx;
  outPending=false;
  errPending=false;
  outTimed=0;
  errTimed=0;
  lineTimeout=5;
 }

ProcessInterface::ProcessInterface(int (*fx)(AudioFileOp *afi,const char *fn),AudioFileOp *afil,const char *fnx)
{
  forkProcess=true;
  forkAudio=true;
  blockTimer=false;
  process=0;
  progressDialog=0;
  forkFuncAudio=fx;
  audioFileOp=afil;
  audioForkArg=fnx;
  outPending=false;
  errPending=false;
  outTimed=0;
  errTimed=0;
  lineTimeout=5;
}

void ProcessInterface::setForkArgument(const char *arg)
 {
   audioForkArg=arg;
 }


void ProcessInterface::prepareProcess(ProgressDialog::ProgressType ptype)
 {
  if (!forkProcess)
   {
    process=new KProcess();
   }
  else
  {
    if (!forkAudio)
     {
      process=new Fork(forkFunc);
     }
     else
     {
       process=new Fork(forkFuncAudio,audioFileOp,audioForkArg);
     }
  }
  stdOutBuffer[0]='\0';
  stdErrBuffer[0]='\0';

  connect(process,SIGNAL(receivedStderr(KProcess *,char *,int)),
          this,SLOT(processStderr(KProcess *,char *,int)));
  connect(process,SIGNAL(receivedStdout(KProcess *,char *,int)),
          this,SLOT(processStdout(KProcess *,char *,int)));
  connect(process,SIGNAL(processExited(KProcess *)),
          this,SLOT(processLeave(KProcess *)));

  progressDialog=new ProgressDialog(0,0,0,ptype);
  connect(progressDialog,SIGNAL(canceled()),this,SLOT(cancelPressed()));

  outputTimer=new QTimer();
  outputTimer->start(100,FALSE);
  connect(outputTimer,SIGNAL(timeout()),this,SLOT(triggerTimer()));

 }

int ProcessInterface::startProcess(bool rootrights,bool progressdialog)
 {
  showProgress=progressdialog;

  if (rootrights)
   {
#ifndef STRICT_PERMISSIONS
    getRootRights();
#endif
   }

 process->start(KProcess::NotifyOnExit,KProcess::All);

 if (rootrights)
  {
#ifndef STRICT_PERMISSIONS
   dropRootRights();
#endif
  }

 if (!showProgress) return(0);
 return(progressDialog->exec());
}

void ProcessInterface::closeProcess(void)
 {
  if (process!=0)
   {
    if (!forkProcess)
     {
      delete (process);
     }
     else
     {
       delete (((Fork *)process));
     }
   }
  if (progressDialog!=0) delete (progressDialog);
  if (outputTimer!=0) delete(outputTimer);
  process=0;
  progressDialog=0;
  outputTimer=0;
 }


void ProcessInterface::cancelPressed(void)
 {
  processCancel();
 }

void ProcessInterface::processLeave(KProcess *)
 {
  progressDialog->abortDialog(processExited());
 }

void ProcessInterface::processStdout(KProcess *,char *buffer,int bufflen)
 {
  int i;
  char *buf,*buf2;

  i=bufflen;
  buf=buffer;
  buf2=stdOutBuffer;
  while (*buf2!=0) ++buf2;
  while (i>0)
   {
    if ( (*buf=='\n') || (*buf=='\r') )
     {
      *buf2=0;
      buf2=stdOutBuffer;
      blockTimer=true;
      if (!processStdoutLine(stdOutBuffer))
       {
        process->kill(SIGINT);
        return;
       }
      stdOutBuffer[0]=0;
      outPending=false;
      outTimed=0;
      blockTimer=false;
     }
     else
     {
      *(buf2++)=*buf;
      *buf2=0;
      outPending=true;
     }
    ++buf;
    --i;
   }
 }

void ProcessInterface::processStderr(KProcess *,char *buffer,int bufflen)
 {
  int i;
  char *buf,*buf2;

  i=bufflen;
  buf=buffer;
  buf2=stdErrBuffer;
  while (*buf2!=0) ++buf2;
  while (i>0)
   {
    if ( (*buf=='\n') || (*buf=='\r') )
     {
      *buf2=0;
      buf2=stdErrBuffer;
      blockTimer=true;
      if (!processStderrLine(stdErrBuffer))
       {
        process->kill(SIGINT);
        return;
       }
      stdErrBuffer[0]=0;
      errPending=false;
      errTimed=0;
      blockTimer=false;
     }
     else
     {
      *(buf2++)=*buf;
      *buf2=0;
      errPending=true;
     }
    ++buf;
    --i;
   }
 }

bool ProcessInterface::processStdoutLine(char *)
 {
  return(true);
 }

bool ProcessInterface::processStderrLine(char *)
 {
  return(true);
 }

int  ProcessInterface::processExited(void)
 {
  return(1);
 }

void ProcessInterface::processCancel(void)
 {
  progressDialog->abortCancel();
 }

ProcessInterface &ProcessInterface::operator<<(const char *arg)
 {
  *process<<arg;
  return(*this);
 }

void ProcessInterface::setProgress(long int val,long int max, bool protect)
 {
  if (!protect)
   {
    progressDialog->setProgress(val,max);
   }
   else
   {
    progressDialog->setProtectedProgress(val,max);
   }
 }

void ProcessInterface::setWorkText(const char *worktext)
 {
  progressDialog->setWorkText(worktext);
 }


void ProcessInterface::writeStdin(char *buffer, int len)
 {
   process->writeStdin(buffer,len);
 }


void ProcessInterface::sendSignal(int signal)
 {
   process->kill(signal);
 }

void ProcessInterface::printStatusLine(QString qs)
 {
   progressDialog->printStatusLine(qs);
 }

void ProcessInterface::addStatusView(void)
 {
   progressDialog->addStatusView();
 }

void ProcessInterface::showDialog(void)
 {
   progressDialog->show();
 }

void ProcessInterface::hideDialog(void)
 {
   progressDialog->hide();
 }

void ProcessInterface::triggerTimer(void)
 {
   if ( (!blockTimer) && (outPending==true) )
    {
     outTimed++; 
     if (outTimed>=lineTimeout)
      {
        blockTimer=true;
        if (!processStdoutLine(stdOutBuffer))
         {
          process->kill(SIGINT);
          return;
         }
        stdOutBuffer[0]=0;
        outPending=false;
        outTimed=0;
        blockTimer=false;
      }
    }

   if ( (!blockTimer) && (errPending==true) )
    {
     errTimed++;
     if (errTimed>=lineTimeout)
      {
        blockTimer=true;
        if (!processStderrLine(stdErrBuffer))
         {
          process->kill(SIGINT);
          return;
         }
        stdErrBuffer[0]=0;
        errPending=false;
        errTimed=0;
        blockTimer=false;
      }
    }
 }

void ProcessInterface::setLineTimeout(int timeout)
 {
   lineTimeout=timeout;
 }
