    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "structbuilder.h"

StructureBuilder::StructureBuilder(Arts::Synthesizer_ptr Synthesizer)
{
	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
	_okay = true;

	haveStructureDesc = false;

	ModuleBroker = Synthesizer->moduleBroker();
	if(!ModuleBroker)
	{
		_error = "No modulebroker provided by Synthesizer";
		_okay = false;
		return;
	}

	StructureDesc = Synthesizer->createStructureDesc();
	if(!StructureDesc)
	{
		_error = "Can't create a new StructureDesc ok Synthesizer";
		_okay = false;
		return;
	}

	haveStructureDesc = true;
	StructureDesc->incRef();
}

StructureBuilder::~StructureBuilder()
{
	while(modules.size())
	{
		delete *modules.begin();
		modules.pop_front();
	}
	if(haveStructureDesc)
	{
		StructureDesc->decRef();
	}
}


// returns true if all operations performed until now were successful
bool StructureBuilder::okay()
{
	return _okay;
}

const char *StructureBuilder::error()
{
	return _error.c_str();
}

void StructureBuilder::createModule(const char *moduletype, const char *name)
{
	if(!_okay) return;

	Arts::ModuleInfo_var minfo = ModuleBroker->lookupModule(moduletype);
	if(!minfo)
	{
		_error = moduletype;
		_error += " not found by ModuleBroker.";
		_okay = false;
		return;
	}

	Arts::ModuleDesc_var md = StructureDesc->createModuleDesc(minfo);
	if(!md)
	{
		_error = "Module ";
		_error += moduletype;
		_error += " could not be created in structure description.";
		_okay = false;
		return;
	}

	modules.push_back(new SBuilderModule(md,name));
}

Arts::PortDesc_ptr StructureBuilder::findPort(const char *portname)
{
	if(!_okay) return 0;

	char *mname = strdup(portname), *pname = mname;

	while(*pname != '.' && *pname != 0) pname++;
	if(*pname != '.') {
		_okay = false;
		_error = "Portname specification ";
		_error += portname;
		_error += " invalid, missing '.' to seperate modulename/portname.";
		return 0;
	}
	*pname = 0;
	pname++;

	list<SBuilderModule *>::iterator i;
	for(i=modules.begin(); i != modules.end(); i++)
	{
		SBuilderModule *m = *i;
		if(strcmp(m->name(),mname) == 0)
		{
			Arts::PortDesc_ptr result = m->findPort(pname);
			if(result == 0)
			{
				_okay = false;
				_error = "No such port ";
				_error += portname;
				_error += " on module.";
				return 0;
			}
			return result;
		}
	}

	_okay = false;
	_error = "Port ";
	_error += portname;
	_error += " couldn't be found, sorry (module missing?).";
	return 0;
}

void StructureBuilder::setStringValue(const char *portname, const char *value)
{
	if(!_okay) return;

	Arts::PortDesc_var port = findPort(portname);
	if(!_okay) return;

	port->StringValue(value);
}

void StructureBuilder::setFloatValue(const char *portname, float value)
{
	if(!_okay) return;

	Arts::PortDesc_var port = findPort(portname);
	if(!_okay) return;

	port->FloatValue(value);
}

void StructureBuilder::connect(const char *srcportname,
											const char *destportname)
{
	if(!_okay) return;

	Arts::PortDesc_var srcport = findPort(srcportname);
	if(!_okay) return;

	Arts::PortDesc_var destport = findPort(destportname);
	if(!_okay) return;

	_okay = destport->connectTo(srcport);
	if(!_okay)
	{
		_error = "Connecting ";
		_error += srcportname;
		_error += " and ";
		_error += destportname;
		_error += "failed.";
	}
}

// creates one instance of the structure
long StructureBuilder::execute(const Arts::ArtsServerSeq& preferredServers)
{
	long result;
	if(!_okay) return 0;

	result = Synthesizer->createStructure(StructureDesc, preferredServers);
	if(!result)
	{
		_okay = false;
		_error = "Creating instance of the structure description failed.";
	}
	return result;
}

Arts::StructureDesc_ptr StructureBuilder::structureDesc()
{
	return Arts::StructureDesc::_duplicate(StructureDesc);
}

//==== SBuilderModule starting here =======================================//

SBuilderModule::SBuilderModule(Arts::ModuleDesc_ptr module, const char *name)
{
	_module = Arts::ModuleDesc::_duplicate(module);
	_name = name;
}

SBuilderModule::~SBuilderModule()
{
	// ?
}

const char *SBuilderModule::name()
{
	return _name.c_str();
}

// returns a duplicated port, so you should put it into a _var
Arts::PortDesc_ptr SBuilderModule::findPort(const char *portname)
{
	Arts::PortDescSeq_var ports = _module->Ports();

	unsigned long i;
	for(i=0;i<ports->length();i++)
	{
		Arts::PortDesc_ptr p = (*ports)[i];
		CORBA::String_var name = p->Name();
		if(strcmp((const char *)name,portname) == 0)
		{
			return Arts::PortDesc::_duplicate(p);
		}
	}
	return 0;
}
