	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "artsbuilder.h"
#include "guiserver_impl.h"
#include "guimodule.h"
#include "sequenceutils.h"

GUIServer_impl::GUIServer_impl(Arts::ModuleBroker *Broker,Arts::Synthesizer *Synthesizer)
{
	describe("GUIServer");

	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
	// FIXME: incRef here? what would be the implications? Could the
	// Synthesizer still terminate...?

	ModuleBroker = Arts::ModuleBroker::_duplicate(Broker);
	ModuleBroker->incRef();

	_ID = ModuleBroker->mkID();
	nextMID = 1;
	nextGlobalID = 1;

	ModuleServer<GuiModule> *MS_GuiModule =
		(ModuleServer<GuiModule> *)GuiModule::get_MS();

	long nr;
	for(nr=0;nr < MS_GuiModule->getModuleCount(); nr++)
	{
		char *name=MS_GuiModule->getModuleName(nr);
		printf("publishing module %s:\n",name);
		if(name)
		{
			GuiModule *m = (GuiModule *)MS_GuiModule->getModule(name);

			printf(" - module created\n");
			if(m)
			{
			    printf(" - not null\n");
				m->publish(this,Broker);
				delete m;
			}
		}
	}	
}

void GUIServer_impl::cleanUp()
{
	ModuleBroker->serverShutdown(this);
	ModuleBroker->decRef();
}

void GUIServer_impl::setGlobalParent(QWidget *globalParent)
{
	this->globalParent = globalParent;
}

void GUIServer_impl::setGlobalLayout(QBoxLayout *globalLayout)
{
	this->globalLayout = globalLayout;
}

/************ GUI Server CORBA functions *************/

CORBA::Long GUIServer_impl::ID()
{
	return _ID;
}

CORBA::Long GUIServer_impl::createModules()
{
	return nextMID++;
}

CORBA::Boolean GUIServer_impl::createModule(CORBA::Long mid,
												Arts::ModuleDesc *desc)
{
	printf("create called\n");
	CORBA::String_var name = desc->Name();
	printf("md->Name() = %s\n",(const char *)name);

	ModuleServer<GuiModule *> *MS_GuiModule =
		(ModuleServer<GuiModule *> *)GuiModule::get_MS();

	GuiModule *m = (GuiModule *)MS_GuiModule->getModule(name);
	assert(m);

	m->initialize(Synthesizer,this);

	/* Well, what are these three IDs?

		- MID: is the ID of the modules that are created in one request by
		  the SynthServer (via CORBA interface).

		  Usually, the same MID is assigned to every module in a (expanded)
		  structure to allow freeing them all at the same time.

		- ID: is the ID of the module as it is described in the StructureDesc.

		  That means, two modules may have the same ID. But the combination
		  of MID and ID is unique.

		- GlobalID is a unique number on this GuiServer that is assigned to
		  each module upon creation. It is used for instance to create visual
		  panels/structures that become visible inside a panel that was created
		  long ago (different MID).
	*/
		  
	m->setMID(mid);
	m->setID(desc->ID());
	m->setGlobalID(nextGlobalID++);
	m->applyParameters(desc);
	modules.push_back(m);

	return true;
}

CORBA::Boolean GUIServer_impl::localConnectModules(CORBA::Long mid)
{
	return true;
}

CORBA::Boolean GUIServer_impl::finalizeModules(CORBA::Long mid)
{
	QWidget *localParent = globalParent;
	list<GuiModule *>::iterator i;
	map<long,GuiModule *> parentmap;

	// search the module which will be parent for the others (usually a panel)
	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;

		if(m->MID() == mid && m->isParent())
		{
			if(!m->parentWidgetID()) localParent = m->isParent();
			printf("added parent widget %ld to map\n",m->ID());
			parentmap[m->ID()] = m;
		}
	}

	// reparent the modules to their parents
	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;

		if(m->MID() == mid)
		{
			list<GuiModule *>::iterator j;
			bool success = false;

			switch(m->parentType())
			{
				case GuiModule::parentGlobal:
						printf("!! searching global parent widget %ld\n",
										m->parentWidgetGlobalID());
						// perhaps solvable more efficient?

						for(j=modules.begin();j != modules.end(); j++)
						{
							GuiModule *p = *j;
							if(p->globalID() == m->parentWidgetGlobalID())
							{
								m->setParent(p->isParent(),p->parentLayout());
								if(p->parentLayout())
									p->parentLayout()->activate();
								success = true;
								printf("!! global parent found.\n");
							}
						}
						if(!success)
							printf("!! global parent *NOT* found.\n");
					break;

				case GuiModule::parentLocal:
						printf("!! module has a parent, using parentmap\n");
						GuiModule *p = parentmap[m->parentWidgetID()];
						assert(p);

						m->setParent(p->isParent(),p->parentLayout());
						if(p->parentLayout())
							p->parentLayout()->activate();
						success = true;
					break;
			}

			/*
			 * catch-all: if no parent setting was achieved until now,
			 * try some guessing
			 */
			if(!success)
			{
				printf("!! module has no explicit parent given\n");
				if(m->isParent())
				{
					printf("!! attaching to globalParent\n");
					m->setParent(globalParent,globalLayout);
				}
				else				// localLayout?
				{
					printf("!! attaching to localParent\n");
					m->setParent(localParent,0);
				}
			}
			m->finalize();
		}

		/*
		if(m->parentWidgetGlobalID())
		{
			printf("!! searching global parent widget %ld\n",
										m->parentWidgetGlobalID());
			// perhaps solvable more efficient?

			list<GuiModule *>::iterator j;
			for(j=modules.begin();j != modules.end(); j++)
			{
				GuiModule *p = *j;
				if(p->globalID() == m->parentWidgetGlobalID())
				{
					m->setParent(p->isParent(),p->parentLayout());
					haveGlobalParent = true;
					printf("!! global parent found.\n");
				}
			}
			if(!haveGlobalParent)
				printf("!! global parent *NOT* found.\n");
		}

		if(m->MID() == mid && !haveGlobalParent)
		{
			if(m->parentWidgetID())
			{
				printf("!! module has a parent, using parentmap\n");
				GuiModule *p = parentmap[m->parentWidgetID()];
				assert(p);

				m->setParent(p->isParent(),p->parentLayout());
			}
			else
			{
				printf("!! module has no explicit parent given\n");
				if(m->isParent())
				{
					printf("!! attaching to globalParent\n");
					m->setParent(globalParent,globalLayout);
				}
				else				// localLayout?
				{
					printf("!! attaching to localParent\n");
					m->setParent(localParent,0);
				}
			}
			m->finalize();
		}
		*/
	}
	return true;
}

void GUIServer_impl::startModules(CORBA::Long mid)
{
	list<GuiModule *>::iterator i;
	printf("start\n");
	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;

		if(m->MID() == mid)
			m->start();
	}
}

void GUIServer_impl::deleteModules(CORBA::Long mid)
{
	list<GuiModule *>::iterator i,next;
	printf("deleteModules(%ld)\n",mid);
	for(i=modules.begin();i != modules.end(); i = next)
	{
		GuiModule *m = *i;
		next = i;
		++next;

		if(m->MID() == mid)
		{
			delete m;
			modules.erase(i);
		}
	}
}

CORBA::Boolean GUIServer_impl::remoteConnectModules(CORBA::Long mid,
	CORBA::Long remotemid, const Arts::ModuleDescSeq& modules,
	Arts::ArtsServer *server)
{
	return true;
}

void GUIServer_impl::requestSignal(CORBA::Long mid, CORBA::Long portid,
	Arts::Receiver *receiver)
{
	list<GuiModule *>::iterator i;
	printf("requestSignal\n");
	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;

		if(m->MID() == mid)
			m->assignReceiver(portid,receiver);
	}
}

void GUIServer_impl::tick()
{
	// TODO: might be optimizable by maintaining a list of modules that
	// actually use the tick
	list<GuiModule *>::iterator i;

	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;
		m->tick();
	}
}

CORBA::Boolean GUIServer_impl::saveSessionModules(CORBA::Long mid,
			Arts::StringSeq*& data, Arts::IDSeq*& structureIDs)
{
	data = new Arts::StringSeq;
	structureIDs = new Arts::IDSeq;

	list<long> ids;

	list<GuiModule *>::iterator i;
	printf("got saveSessionModules\n");
	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;

		if(m->MID() == mid)
		{
			Arts::StringSeq_var moduledata = m->saveSessionParameters(ids);
			if(moduledata->length() > 0)
			{
				sqprintf(data,"module_id=%d",m->ID());
				addSubStringSeq(data,moduledata);
			}
		}
	}

	// create list of structureids that are required

	unsigned long l = 0;

	list<long>::iterator ii;
	structureIDs->length(ids.size());
	for(ii=ids.begin();ii != ids.end(); ii++)
		(*structureIDs)[l++] = (*ii);

	return true;
}

CORBA::Boolean GUIServer_impl::restoreSessionModules(CORBA::Long mid,
			const Arts::StringSeq& data, CORBA::Long restoreID)
{
    unsigned long i;
    char *cmd,*param;

    for(i=0;i<data.length();i++)
    {
        if(parse_line(data[i],cmd,param))   // otherwise: empty or comment
        {
            if(strcmp(cmd,"module_id") == 0) {
                long id = atol(param);
                Arts::StringSeq_var params = getSubStringSeq(&data,i);

				list<GuiModule *>::iterator m;
				for(m=modules.begin();m != modules.end(); m++)
				{
					if((*m)->MID() == mid && (*m)->ID() == id)
					{
						printf("found module to restore parameters\n");
						(*m)->setRestoreID(restoreID);
						(*m)->restoreSessionParameters(*params);
					}
				}
            }
        }
    }                                                                           
	return true;
}

/*
CORBA::Boolean GUIServer_impl::restoreSessionStructures(CORBA::Long mid,
			const Arts::IDSeq& oldIDs, const Arts::IDSeq& newIDs)
{
	list<GuiModule *>::iterator i;

	for(i=modules.begin();i != modules.end(); i++)
	{
		GuiModule *m = *i;

		if(m->MID() == mid)
			m->restoreSessionStructures(oldIDs,newIDs);
	}
	return true;
}
*/
