    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <netdb.h>
#include <netinet/in.h>
#include "arts.h"

// how many _bytes_ we send in each block when playing
#define READ_SIZE	16384

// how many _bytes_ we read in each block when recording
#define RECORD_SIZE	16384

// how many _samples_ should be in the buffer on the server side
#define BUFFER_OK	65536

struct sockaddr_in *parse_tcp_url(const char *url)
{
	static struct sockaddr_in addr;

	char *work = strdup(url);

	char *type = strtok(work,":");
	if(type == 0 || strcmp(type,"tcp") != 0) return 0;

	char *host = strtok(NULL,":");
	if(host == 0) return 0;

	char *port = strtok(NULL,":\n");
	if(port == 0) return 0;

	long portno = atol(port);
	if(portno < 1 || portno > 65535) return 0;

    struct hostent *server = gethostbyname(host);                                        
    memset((void *) &addr, '\0', sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = *(u_long *)server->h_addr;
    addr.sin_port = htons(portno);

	return &addr;
}

int tcp_connect(const char *url)
{
	struct sockaddr_in *remote_addr = parse_tcp_url(url);
	if(remote_addr == 0)
	{
		fprintf(stderr,"couldn't parse url %s\n",url);
		return 0;
	}

	int my_socket = socket(AF_INET,SOCK_STREAM,0);
	if(my_socket < 0)
	{
		fprintf(stderr,"unable to open socket for read");                     
		return 0;
	}

	struct linger lin;
    lin.l_onoff=1;      /* block a closing socket for 1 second */
    lin.l_linger=100;   /* if data is waiting to be sent */
    if ( setsockopt( my_socket, SOL_SOCKET, SO_LINGER,
                     &lin, sizeof(struct linger) ) < 0 )
    {
        fprintf(stderr,"Unable to set socket linger value to %d\n",
                lin.l_linger);
        return 0;
    }

	int rc;
	rc=connect(my_socket,(struct sockaddr *)remote_addr, sizeof(*remote_addr));
	if(rc != 0)
	{
		fprintf(stderr,"can't connect to server");
		return 0;
	}

	return my_socket;
}

char *search_ior(int fd)
{
	static char buffer[1024];
	int i = 0;

	while(read(fd,&buffer[i],1) == 1)
	{
		if(i == 1023) buffer[i] = '\n';		// cut lines that are too long

		if(buffer[i] == '\n')
		{
			buffer[i] = 0;

			int code = atol(buffer);
			if(code == 100)
				return(&buffer[4]);

			if(code != 200 && code != 800)
			{
				fprintf(stderr,"unexpected code %d occured\n",code);
				return 0;
			}

			i = 0;
		}
		else
		{
			i++;
		}
	}
	return 0;
}

void print_buffer_status(long full, long max)
{
	static long lastfilled = 0;
	long filled,i;

	filled = full * 100 / max;
	filled += 5;
	filled -= filled % 10;		// make divisible by 10

	if(filled > 100) filled = 100;
	if(filled < 0) filled = 0;

	if(filled == lastfilled) return;

	lastfilled = filled;
	printf("buffer [");
	for(i=10;i<=100;i+=10)
	{
		if(filled >= i)
			printf("X");
		else
			printf(" ");
	}
	printf("] %3ld%%\r",filled);
	fflush(stdout);
}

#ifdef COMMON_BINARY
int artscat_main(int argc, char **argv)
#else
int main(int argc, char **argv)
#endif
{
	CORBA::ORB_var orb = CORBA::ORB_init( argc, argv, "mico-local-orb" );
	CORBA::BOA_var boa = orb->BOA_init( argc, argv, "mico-local-boa" );

	CORBA::Object_var obj = orb->bind ("IDL:Arts/Synthesizer:1.0",
												"inet:localhost:8888");
	Arts::Synthesizer_var Synthesizer = Arts::Synthesizer::_narrow (obj);

	if(CORBA::is_nil(Synthesizer))
	{
		fprintf(stderr,"%s will only work when the Arts Server is running\n",
					argv[0]);
		exit(1);
	}

	Arts::AudioManager_var AudioManager = Synthesizer->audioManager();
	assert(AudioManager);

	CORBA::String_var url = AudioManager->url();

	int fd = tcp_connect(url);
	if(!fd) return 0;

	char *ior = search_ior(fd);
	if(!ior) return 0;

	char *type = 0, *description = 0;
	Arts::AudioClientDirection direction = Arts::acPlay;
	int optch;

	while((optch = getopt(argc,argv,"t:d:rp")) > 0)
	{
		switch(optch)
		{
			case 't': type = strdup(optarg);
				break;
			case 'd': description = strdup(optarg);
				break;
			case 'r': direction = Arts::acRecord;
				break;
			case 'p': direction = Arts::acPlay;
				break;
			default: 
					fprintf(stderr,"usage: %s [ -t <type> ] [ -d <description> ] [ -r ] [ -p ]\n\n",argv[0]);
					fprintf(stderr,"-p    send audio data from stdin to aRts (default)\n");
					fprintf(stderr,"-r    record audio data from aRts and write to stdout\n");
					exit(1);
				break;
		}
	}

	fprintf(stderr,"got ior='%s'\n",ior);

	Arts::AudioUplink_var uplink = 
		Arts::AudioUplink::_narrow(orb->string_to_object(ior));
	assert(uplink);

	if(type) uplink->type(type);
	if(description) uplink->description(description);
	uplink->direction(direction);

	if(direction == Arts::acPlay)
	{
		bool running = false;
		char buffer[READ_SIZE];
		int n;

		while(!feof(stdin))
		{
			if((n = fread(buffer,1,READ_SIZE,stdin)) > 0)
			{
				if(write(fd,buffer,n) != n)
				{
					fprintf(stderr,"write error while writing to the server");
					close(fd);
					return 1;
				}
			}
			long bufferedSamples = uplink->bufferedSamples();
			print_buffer_status(bufferedSamples,BUFFER_OK);

			while(bufferedSamples > BUFFER_OK)
			{
				if(!running)
				{
					uplink->start();
					running = true;
				}
				print_buffer_status(bufferedSamples,BUFFER_OK);
				usleep(500000);
				bufferedSamples = uplink->bufferedSamples();
			}
		}
		if(!running)
		{
			uplink->start();
			running = true;
		}
	}
	else
	{
		char buffer[RECORD_SIZE];
		bool running = true;

		uplink->start();
		while(running)
		{
			int n = read(fd,buffer,RECORD_SIZE);

			if(n > 0)
			{
				int w = fwrite(buffer,1,n,stdout);
				if(w != n) running = false;
			}
		}
	}
	close(fd);

	// don't access the CORBA interface any more here, because it will
	// terminate itself after receiving the close from the socket.
	return 0;
}
