    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "mbroker_impl.h"
#include "debug.h"
#include <dirent.h>

MB_ServerEntry::MB_ServerEntry(Arts::ArtsServer *server, Arts::ModuleInfo *info)
{
	this->server = server;
	this->info = info;
}

Arts::StringSeq* ModuleBroker_impl::publishedModules()
{
	Arts::StringSeq* seq = new Arts::StringSeq;
	list<MB_ServerEntry *>::iterator i;
	set<string> names;
	set<string>::iterator si;

	for(i=infolist.begin(); i != infolist.end(); i++)
	{
		Arts::ModuleInfo& mi = *((*i)->info);

		names.insert((const char *)mi.name);
	}

	unsigned long n = 0;
	seq->length(names.size());
	for(si=names.begin();si != names.end();si++)
		(*seq)[n++] = CORBA::string_dup((*si).c_str());

	return seq; 
}


void ModuleBroker_impl::publishModule( Arts::ArtsServer *server,
                                       const Arts::ModuleInfo& info )
{
	Arts::ModuleInfo *newinfo = new Arts::ModuleInfo(info);

	artsdebug("Broker::publishModule called:\n");
	artsdebug("  Server ID=%ld does registering: %s\n",
							server->ID(),(const char *)info.name);


	// FIXME: do name collision checking here!
	// two modules may have the same name, if they have exactly the same
	// interface... otherwise: complain

	_updateCount++;
	infolist.push_back(
		new MB_ServerEntry(Arts::ArtsServer::_duplicate(server),newinfo));
}


Arts::ModuleInfo* ModuleBroker_impl::lookupModule( const char* name )
{
	list<MB_ServerEntry *>::iterator i;

	for(int tries = 0; tries < 2; tries++)
	{
		// first try (0): just look what is published
		// second try (1): if there is still nothing found, try to load
		//     structure from "paths"
		if(tries == 1) loadStructure(name);

		for(i=infolist.begin(); i != infolist.end(); i++)
		{
			Arts::ModuleInfo& mi = *((*i)->info);
			if(strcmp(mi.name,name) == 0)
				return(new Arts::ModuleInfo(mi));
		}
	}

	return 0; 	// not found
}

void
ModuleBroker_impl::removeModule( Arts::ArtsServer *server, const char* name )
{
	list<MB_ServerEntry *>::iterator i;
	long sid = server->ID();

	i = infolist.begin();
	while(i != infolist.end())
	{
		Arts::ModuleInfo& mi = *((*i)->info);

		if(strcmp((const char *)mi.name,name) == 0 && (*i)->server->ID() == sid)
		{
			delete (*i);
			infolist.erase(i);
			i = infolist.begin();
		}
		else i++;
	}
}

void
ModuleBroker_impl::serverShutdown( Arts::ArtsServer *server )
{
	list<MB_ServerEntry *>::iterator i;
	long sid = server->ID();

	for(i=infolist.begin(); i != infolist.end();)
	{
		if((*i)->server->ID() == sid)
		{
			delete (*i);
			infolist.erase(i);
			i = infolist.begin();
		}
		else i++;
	}
	artsdebug("Broker serverShutdown(ID %d) processed, %ld modules remaining\n",
                sid,infolist.size());
}

Arts::ArtsServerSeq* ModuleBroker_impl::whichServerProvides( char const *name )
{
	Arts::ArtsServerSeq *result = new Arts::ArtsServerSeq;

	list<MB_ServerEntry *>::iterator i;

	for(i=infolist.begin(); i != infolist.end(); i++)
	{
		Arts::ModuleInfo& mi = *((*i)->info);
		if(strcmp(mi.name,name) == 0)
		{
			unsigned long l = result->length();
			result->length(l+1);
			(*result)[l] = Arts::ArtsServer::_duplicate((*i)->server);
		}
	}
	return(result);
}

void ModuleBroker_impl::addPublishingPath(const char *path)
{
	char *pathname = strdup(path);

	while(strlen(pathname) && pathname[strlen(pathname)-1] == '/')
		pathname[strlen(pathname)-1]=0;

	list<string>::iterator i;
	for(i=paths.begin();i!=paths.end();i++)
	{
		if(strcmp((*i).c_str(),pathname) == 0)
		{
			free(pathname);
			return;
		}
	}
	paths.push_back(pathname);

	DIR *dir = opendir(pathname);
	struct dirent *de;
	while((de = readdir(dir)) != 0)
	{
		char buffer[1024];
		strcpy(buffer,de->d_name);
		// cut .arts-extension and look if that module is present (that
		// will trigger loading if it isn't)
		if(strlen(buffer) > 5 &&
			strncmp(&buffer[strlen(buffer)-5],".arts",5) == 0)
		{
			buffer[strlen(buffer)-5] = 0;
			artsdebug("ModuleBroker: trying to autoload module %s\n",buffer);
			Arts::ModuleInfo_var mi = lookupModule(buffer);
		}
	}
	closedir(dir);
}

CORBA::Long ModuleBroker_impl::mkID()
{
	return(nextID++);
}

CORBA::Long ModuleBroker_impl::updateCount()
{
	return(_updateCount);
}

void ModuleBroker_impl::loadStructure(const char *name)
{
	list<string>::iterator i;

	for(i=paths.begin();i!=paths.end();i++)
	{
		string filename = *i+"/"+string(name)+".arts";
		artsdebug("ModuleBroker: trying to load %s\n",filename.c_str());

		FILE *infile = fopen(filename.c_str(),"r");
		if(infile)
		{
			Arts::StructureDesc_var sd = Synthesizer->createStructureDesc();

			sd->incRef();
			artsdebug("ModuleBroker: loading structure %s\n",filename.c_str());
			Arts::StringSeq_var strseq = new Arts::StringSeq;
			printf("loading structure %s ... ",filename.c_str());
			fflush(stdout);

			char line[1024];
			unsigned long i = 0;

			while(fgets(line,1024,infile))
			{
				// cut eventual CR and/or LFs at the end of the line
				while(strlen(line) && line[strlen(line)-1] < 14)
					line[strlen(line)-1] = 0;

				strseq->length(i+1);
				(*strseq)[i++] = CORBA::string_dup(line);
			}
			fclose(infile);

			sd->loadFromList(strseq);

			CORBA::String_var structurename = sd->Name();
			artsdebug("ModuleBroker: loaded structure is called %s\n",
					(const char *)structurename);

			if(strcmp(structurename,name) == 0 && sd->valid())
			{
				Synthesizer->publishStructureDesc(sd);
				sd->decRef();
				printf("loaded.\n");
				return;
			}
			printf("failed (is called %s).\n",(const char *)structurename);
			if(!sd->valid())
			{
				printf(
"!! The saved structure couldn't be loaded correctly from the description.\n"
"!! Perhaps some required modules are not supported by your version of aRts?\n"
				);
			}
			sd->decRef();
		}
	}
}

ModuleBroker_impl::ModuleBroker_impl(Arts::Synthesizer_ptr Synthesizer)
{
	describe("ModuleBroker");

	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
	_updateCount = 0;	
	nextID = 1;
}
