	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include "resources.h"
#include "debug.h"

long SynthConnection::instances = 0;
ResourceMonitor RM_SynthConnection_instances(&SynthConnection::instances,"SynthConnection object instances");

long SynthBuffer::instances = 0;
static ResourceMonitor RM_SynthBuffer_instances(&SynthBuffer::instances,"SynthBuffer object instances");

long SynthBuffer::allbytes = 0;
static ResourceMonitor RM_SynthBuffer_allbytes(&SynthBuffer::allbytes,"Total bytes in all SynthBuffers");

string SynthModule::getArgument(int nr, ArgType direction)
{
	string result, arglist = getParams();
	int wanted = nr, have = 0;
	unsigned long i;

	if(direction == arg_out)
		wanted += 512;	// second string of args behind ';'
						// more than 512 arguments currently unsupported ;)

	for (i=0;i<arglist.length();i++)
	{
		switch(arglist[i])
		{
			case ',': have++;
				break;
			case ';': have += 512 - (have & 511);
				break;
			default:  if(have == wanted) result += arglist[i];
				break;
		}
	}
	return(result);
};

void SynthModule::ProcessData(string data)
{
	fprintf(stderr,"data could not be processed by %s\n",getName().c_str());
}

void SynthModule::FirstInitialize()
{
}

void SynthModule::DeInitialize()
{
	// do nothing
}

void SynthModule::getfds(int &infd, int &outfd)
{
	// if we are an io performing module (midi event processing, sound output,
    // ..., we are likely to open out files here and return them to the
    // synthesizer, which will perform scheduling if io is to do

	// setting fds to -1 means, we don't have them

    infd = -1;   // no input performed by this module
	outfd = -1;  // no output performed by this module
}

ArtsNotifyIO *SynthModule::ioHandler()
{
	// no custom io handler defined
	return 0;
}

void *SynthModule::get_MS()
{
	static ModuleServer<SynthModule> *MS = 0;
	if(!MS)
	{
		MS = new ModuleServer<SynthModule>;
#ifndef MODULESERVER_DEBUG
	}
#else
		artsdebug("Creating a ModuleServer for SynthModules\n");
	}
	else
	{
		artsdebug("Obtaining a ModuleServer reference for SynthModules\n");
	}
	artsdebug("ModuleServer is at %lx\n",MS);
#endif

	return(MS);
}

SynthModule::SynthModule()
{
	inConn = 0;
	outConn = 0;

	in = out = 0;		// for fast access from Calculate(Block) to ringbuffer data

	inConnCount = 0;
	outConnCount = 0;

	clients = 0;

	samplingRate = SYNTHMODULE_DEFAULT_SAMPLINGRATE;

	// to speed up modules that need the samplingrate in integer like
	// calculations
	samplingRateULong = (unsigned long)SYNTHMODULE_DEFAULT_SAMPLINGRATE;

	enabled = false;

	haveCalculateBlock = false;	// default is currently still single sample
}

SynthModule::~SynthModule()
{
	unsigned long i;
	artsdebug("inConnCount = %d, outConnCount = %d\n",inConnCount,outConnCount);
	for(i = 0;i<inConnCount;i++) delete inConn[i];
	for(i = 0;i<outConnCount;i++) delete outConn[i];
	if(inConnCount)
	{
		free(inConn);
		free(in);
	}
	if(outConnCount)
	{
		free(outConn);
		free(out);
	}
}

void SynthModule::assignInConn(SynthConnection *newInConn)
{
	inConn = (SynthConnection **)realloc(inConn,
				sizeof(SynthConnection *) * (inConnCount+1));
	inConn[inConnCount] = newInConn;

	inConnCount++;

	in = (float **)realloc(in, sizeof(float *) * inConnCount);
}

void SynthModule::assignOutConn(SynthConnection *newOutConn)
{
	outConn = (SynthConnection **)realloc(outConn,
					sizeof(SynthConnection *) * (outConnCount+1));
	outConn[outConnCount] = newOutConn;

	outConnCount++;

	out = (float **)realloc(out, sizeof(float *) * outConnCount);
}

void SynthModule::prepareExecution()
{
	unsigned long i;

	// reset scheduling information
    NeedCycles = 0;
    CanPerform = 0;
    Busy = 0;
    BusyHit = 0;
 
	clients = 0;

	for(i=0;i<outConnCount;i++) clients += outConn[i]->destcount;

	unsigned long preprod = 1024;
	for(i=0;i<outConnCount;i++)
	{
		unsigned long room = outConn[i]->outRoom();
		if(preprod > room) preprod = room;
		assert(outConn[i]->buffer->position == 0);
		out[i] = outConn[i]->buffer->data;
	}

	preprod = PreProduce(preprod);
	if(preprod)
	{
		for(i=0;i<outConnCount;i++)
		{
			outConn[i]->buffer->position += preprod;
			outConn[i]->buffer->needread += preprod * outConn[i]->destcount;
		}
	}
}

/* request a module to calculate a number of turns
   will return -1 if busy, count that has been done otherwise */

long SynthModule::request(long amount)
{
	unsigned long in;
	int have_in,need_more,have_done,have_done_total = 0;

	if(Busy) { BusyHit++; return(-1); }

	Busy = 1;
	if(NeedCycles < amount)
	{
		NeedCycles = amount;
	}

	//artsdebug("DSP %s: request of %d cycles\n",getClassName(),amount);
	do
	{
		/* assume we can satisfy the request */
		CanPerform = NeedCycles;

		/* then, check wether the input channels supply enough data to do so. */

		for(in=0;in<inConnCount;in++)
		{
			have_in = inConn[in]->haveIn();

			if(have_in < NeedCycles)
			{
				//artsdebug("connection %d indicates to have %d, "
                //       "thats not enough\n", in,have_in);
				/* if we can't calculate enough stuff due to a certain
				   ingoing connection, go to the associated module and
				   tell it that we need more data
				*/
				need_more = NeedCycles - have_in;
				//artsdebug("requesting %d\n", need_more);
				/* when there is no source (constant input value), then
					we don't need to request something, it's just that
					it can't supply more because the buffer isn't big enough
				*/
				if(inConn[in]->sourcemodule)
					inConn[in]->sourcemodule->request(need_more);

				have_in = inConn[in]->haveIn();
	
				//artsdebug("now connection %d indicates to have %d\n",in,have_in);
				if(CanPerform > have_in) CanPerform = have_in;
			}
		}
		have_done = calc(CanPerform);
		have_done_total += have_done;

		/*
		if(dsp->m_BusyHit != 0) artsdebug("got busyhit: %s\n",dsp->m_Name);
		*/

	} while(BusyHit && NeedCycles != CanPerform && have_done);
	/* makes only sense to repeat if
		 - there was a busyhit which indicates we are in a feedback loop
		 - we should have done more than we have done
         - actually something could be calculated
     */
	Busy = 0;
	return(have_done);
}

/* This routine would now actually let the plugin calculate some data
   that means generate sinus waves, mix audio signals etc.

   The number of cycles is guaranteed to work without input underrun
   by the flow system. But the routine still needs to check output
   stall situations (ring buffer full).
*/

unsigned long SynthModule::calc(unsigned long cycles)
{
	unsigned long i,room;

	/* output sanity check:
	   when there is not enough room in one of the buffers, we
	   can't calculate that much cycles
	*/
	for(i=0;i<outConnCount;i++)
	{
		room = outConn[i]->outRoom();
		if(room < cycles)
		{
			cycles = room;
			/*
			artsdebug("- reduced calculation to %d due to lack of space\n",cycles);
			*/
		}
	}

	if(cycles == 0) return(0);

	//artsdebug("DSP %s: calculation of %d cycles\n",getClassName(),cycles);
	/* input sanity check:
		it's guaranteed that we have enough input, but do the check
		anyway... - you never know
	*/
	for(i=0;i<inConnCount;i++)
	{
		/* otherwise input is "overconsumed" */
		assert(inConn[i]->haveIn() >= cycles);

		/* check sanity of the needread setting:
			either data comes from fixed value (sourcemodule not assigned)
			then we can't expect needread to contain sensible setting,
			but otherwise: needread should have expected that we'll want
			to read the data and thus be more than the cycles! */

		assert((!inConn[i]->sourcemodule)
				|| (inConn[i]->buffer->needread >= cycles));
	}

	if(haveCalculateBlock)
	{
		unsigned long j, donecycles = 0, cando = 0;
	
		while(donecycles != cycles)
		{	
			cando = cycles-donecycles;

			for(j=0;j<inConnCount;j++)
			{
				unsigned long rbsize = inConn[j]->buffer->size;
				unsigned long rbmask = rbsize - 1;
				unsigned long rbpos =
					(inConn[j]->position+donecycles) & rbmask;

				in[j] = &inConn[j]->buffer->data[rbpos];
				if(cando > (rbsize-rbpos)) cando = rbsize - rbpos;
			}

			for(j=0;j<outConnCount;j++)
			{
				unsigned long rbsize = outConn[j]->buffer->size;
				unsigned long rbmask = rbsize - 1;
				unsigned long rbpos =
					(outConn[j]->buffer->position+donecycles) & rbmask;

				out[j] = &outConn[j]->buffer->data[rbpos];
				if(cando > (rbsize-rbpos)) cando = rbsize - rbpos;
			}
	
			CalculateBlock(cando);
			donecycles += cando;
		}
		assert(donecycles == cycles);
	}
	else
	{
		for(i=0;i<cycles;i++)
		{
			unsigned long j;

			for(j=0;j<inConnCount;j++)
			{
				int rbm = inConn[j]->buffer->size-1;	// assume size is 2^n
				in[j] = &inConn[j]->buffer->data[(inConn[j]->position+i) & rbm];
			}

			for(j=0;j<outConnCount;j++)
			{
				int rbm = outConn[j]->buffer->size-1;	// assume size is 2^n
				out[j] =
			&outConn[j]->buffer->data[(outConn[j]->buffer->position+i) & rbm];
			}

			Calculate();
		}
	}

	// actually update buffer by subtracting consumed input
	for(i=0;i<inConnCount;i++)
	{
		inConn[i]->position += cycles;				/* consumes input */
		inConn[i]->buffer->needread -= cycles;		/* consumes input */
	}

	// and adding fresh output
	for(i=0;i<outConnCount;i++)
	{
		outConn[i]->buffer->position += cycles;
		outConn[i]->buffer->needread += cycles * outConn[i]->destcount;
	}

	NeedCycles -= cycles;
	CanPerform -= cycles;
	return(cycles);
}

void SynthModule::CalculateBlock(unsigned long size)
{
	assert(false);	// never do this ;)
}

void SynthModule::CallBack()
{
	// nobody should send me a callback when I didn't request it
	assert(false);
}

unsigned long SynthModule::PreProduce(unsigned long maxsamples)
{
	// default: no preproduction possible
	return(0);
}

const char *SynthModule::getStringProperty(unsigned long property)
{
	SynthProperty *p = inPorts[property]->property();
	assert(p);
	return p->stringValue();
}

void SynthModule::setStringProperty(unsigned long index, const char *newValue)
{
	SynthProperty *p = outPorts[index]->property();
	assert(p);
	p->setStringValue(newValue);
	outPorts[index]->notifyChanges();
}

void SynthModule::propertyChanged(unsigned long index)
{
	// reimplement that if you care about property changes ;)
}

Arts::StringSeq *SynthModule::saveSessionParameters(list<long>& IDs)
{
	// reimplement for session management
	return new Arts::StringSeq;
}

void SynthModule::restoreSessionParameters(const Arts::StringSeq& params)
{
	// reimplement for session management
}

void SynthModule::setRestoreID(long restoreID)
{
	this->restoreID = restoreID;
}

// SynthBuffer...:

SynthBuffer::SynthBuffer(float initialvalue, unsigned long size)
{
	this->size = size;
	data = (float *)malloc(sizeof(float) * size);

	setValue(initialvalue);

	position = 0;
	needread = 0;
	instances++;
	allbytes += size;
}

void SynthBuffer::setValue(float value)
{
	unsigned long i;
	for(i=0;i<size;i++) data[i] = value;
}

SynthBuffer::~SynthBuffer()
{
	instances--;
	allbytes -= size;
	free(data);
}

// SynthConnection...:

SynthConnection::SynthConnection(SynthModule *module, long ID)
{
	this->ID = ID;
	this->module = module;
	position = 0;
	destcount = 0;
	destmodule = 0;
	sourcemodule = 0;
	buffer = 0;
	sourceconn = 0;
	isDynamic = false;
	needConnect = false;
	freeBufferOnExit = false;
	instances++;
}

SynthConnection::~SynthConnection()
{
	if(destcount) free(destmodule);
	if(freeBufferOnExit)
	{
		assert(buffer);
		delete buffer;
	}
	instances--;
}

void SynthConnection::connectToSource(SynthConnection *source)
{
	sourceconn = source;
    buffer = source->buffer;
    sourcemodule = source->module;
 
    source->destmodule = (SynthModule **)realloc(source->destmodule,
								sizeof(SynthModule *)*(source->destcount+1));
    source->destmodule[source->destcount] = module;
	source->destcount++;
	artsdebug("ConnectToSource: %s->destcount set to %d\n",source->module->getClassName(),source->destcount);

	position = buffer->position;
}

void SynthConnection::disconnectFromSource(SynthConnection *source)
{
	SynthModule **ndestmodule = 0;
	unsigned long i,ndestcount = 0;

	artsdebug("DISCONNECT -> recalculating %s destcount: before %ld\n",source->module->getClassName(),source->destcount);
	for(i=0;i<source->destcount;i++)
	{
		if(source->destmodule[i] != module)
		{
    		ndestmodule = (SynthModule **)realloc(ndestmodule,
								sizeof(SynthModule *)*(ndestcount+1));
    		ndestmodule[ndestcount++] = source->destmodule[i];
		}
	}
	if(source->destcount) free(source->destmodule);
	source->destmodule = ndestmodule;
	source->destcount = ndestcount;
	artsdebug("DISCONNECT -> recalculating %s destcount: now %ld\n",source->module->getClassName(),source->destcount);
}

void SynthConnection::allocBuffer(float initialvalue, unsigned long size)
{
	assert(!buffer);
	assert(!freeBufferOnExit);
	buffer = new SynthBuffer(initialvalue,size);
	freeBufferOnExit = true;
}

SynthProperty::SynthProperty(const char *value)
{
	setStringValue(value);
}

const char *SynthProperty::stringValue()
{
	return _stringValue.c_str();
}

void SynthProperty::setStringValue(const char *newValue)
{
	_stringValue = newValue;
}

SynthPort::SynthPort(SynthModule *parent, unsigned long index, long ID)
{
	_module = parent;
	_property = 0;
	_connection = 0;
	_remotePort = 0;
	_needConnect = false;
	_wantedID = 0;
	_index = index;
	_ID = ID;
}

SynthModule *SynthPort::module()
{
	return _module;
}

bool SynthPort::needConnect()
{
	return _needConnect;
}

void SynthPort::setNeedConnect(bool newNeedConnect, long wantedID = 0)
{
	_wantedID = wantedID;
	_needConnect = newNeedConnect;
}

long SynthPort::wantedID()
{
	return _wantedID;
}

long SynthPort::ID()
{
	return _ID;
}

// this also will transparently return the "remote" property if the port
// is connected
SynthProperty *SynthPort::property()
{
	if(_remotePort)
		return _remotePort->property();

	return _property;
}

void SynthPort::setProperty(SynthProperty *property)
{
	_property = property;
}

void SynthPort::propertyChanged()
{
	_module->propertyChanged(_index);
}

void SynthPort::notifyChanges()
{
	list<SynthPort *>::iterator i;

	for(i=clients.begin();i != clients.end();i++)
	{
		SynthPort *p = *i;
		p->propertyChanged();
	}
}

// conventional audio connections

SynthConnection *SynthPort::connection()
{
	return _connection;
}

void SynthPort::setConnection(SynthConnection *property)
{
	_connection = property;
}

void SynthPort::connectToPort(SynthPort *port)
{
	assert(!_remotePort);	// only connect to one port at a time currently
	_remotePort = port;
	_remotePort->addClient(this);

	// assume that in the current "one input port" = "one connection"
	// implementation, we can drop needConnect as soon as a connection is
	// made
	_needConnect = false;
}

void SynthPort::disconnectFromPort(SynthPort *port)
{
	assert(_remotePort == port);
	assert(_remotePort);

	_remotePort->removeClient(this);
	_remotePort = 0;
}

void SynthPort::addClient(SynthPort *port)
{
	clients.push_back(port);
}

void SynthPort::removeClient(SynthPort *port)
{
	list<SynthPort *>::iterator i;
	for(i=clients.begin(); i != clients.end(); i++)
	{
		if(*i == port)
		{
			clients.erase(i);
			return;
		}
	}
	
	assert(false);
}

SynthPort::~SynthPort()
{
	if(_remotePort)
		disconnectFromPort(_remotePort);
}
