/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "fadeDevice.h"
#include "kjukebox.h"
#include "../config.h"

FadeDevice::FadeDevice() : 
  QObject(0,"fadeDevice"){
  config = (KApplication::getKApplication())->getConfig();  
  config->setGroup(KCONFIG_PLAYER);

  streamInfoDevice = NULL;
  volumeDevice = NULL;
  playMode = _STATUS_STOPPED;
  fadeMode = _FADEMODE_FADEIN;  
  fadeDuration = config->readNumEntry(KCONFIG_PLAYER_FADEINDURATION,3000);
  startVolume = 0;
  timerId   = 0; 
  counter   = 0.0; 
}


FadeDevice::~FadeDevice(){ 
  if (timerId)  
    killTimer(timerId);
  if (streamInfoDevice != 0)
    disconnect(streamInfoDevice->getEventQueue(),NULL, this, NULL);
}


void FadeDevice::attachTo(AudioStreamInfoDevice* streamDevice){
  if (streamInfoDevice != NULL) {
    debugOutput( cout << "error:  connecting to already connected FadeDevice!"<<endl );
    exit(1);
  }
  streamInfoDevice=streamDevice;
  connect(streamInfoDevice->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(processEvent(char)));
}


void FadeDevice::attachTo(VolumeDevice  *volDevice){
  int vol;
  volumeDevice = volDevice;
  config->setGroup(KCONFIG_PLAYER);
  if (config->readBoolEntry(KCONFIG_PLAYER_FADEIN,true))
    vol = 0;
  else
    vol = 100;  

  volumeDevice->setVolume(vol); 
  emit volumeChanged(vol);
}  


void FadeDevice::detach() {
  disconnect(streamInfoDevice->getEventQueue(),NULL,this,NULL);
  streamInfoDevice = NULL;
  volumeDevice     = NULL;
}


void FadeDevice::timerEvent( QTimerEvent * ){
  float  faktor;
  int value;
  int vol;
  faktor =  1 + cos( 3.141592654 * counter / _FADE_STEPS );
  value  = (int) ((float) volumeIntervall * faktor / 2.0);

  if (faktor <= 0.001){
    counter = 0.0;
    stopFading();
    return;
  }    

  if (fadeMode == _FADEMODE_FADEOUT)
    vol = _FADE_MIN_VOLUME + value;
  else 
    vol = _FADE_MAX_VOLUME - value;

  volumeDevice->setVolume(vol); 
  emit volumeChanged(vol);

  counter += 1.0;;
}

/**
   HardFade indicates, if the fade device should fade every time or only if the 
   user has set it in the settings.
*/
void FadeDevice::startFadeTimer(bool hardFade=true){
  int maxMiliSecondsToFade;
  if (timerId) return;
  startVolume = volumeDevice->getVolume();
  config->setGroup(KCONFIG_PLAYER);
  if (fadeMode == _FADEMODE_FADEOUT){
    if (!hardFade && !(config->readBoolEntry(KCONFIG_PLAYER_FADEOUT,true))) return;
    volumeIntervall = abs(startVolume - _FADE_MIN_VOLUME);
    fadeDuration = config->readNumEntry(KCONFIG_PLAYER_FADEOUTDURATION,3000);
    maxMiliSecondsToFade = 1000 *(streamInfoDevice->getMusicInfo()->getLen() 
				  - streamInfoDevice->getTimeInfo()->getTotalSec()
				  - 1 );
    emit fadeOutStarted();
  }
  else {
    if (!hardFade && !(config->readBoolEntry(KCONFIG_PLAYER_FADEIN,true))) return;
    volumeIntervall = abs(_FADE_MAX_VOLUME - startVolume);
    fadeDuration = config->readNumEntry(KCONFIG_PLAYER_FADEINDURATION,3000);
    maxMiliSecondsToFade = 45000;
    emit fadeInStarted();
  }

  if (fadeDuration > maxMiliSecondsToFade)
   fadeDuration = maxMiliSecondsToFade;

  if (fadeDuration > 500 )
    timerId = startTimer((float ) fadeDuration / _FADE_STEPS);
}


void FadeDevice::setFadeDuration(int microSeconds){ 
  fadeDuration = microSeconds; 
}


void FadeDevice::setFadeMode(int mode){  
  fadeMode = mode; 
}


int  FadeDevice::getFadeMode(){  
  return fadeMode; 
}


bool FadeDevice::isFading(){
  if (timerId != 0) return true;
  return false;
}


void FadeDevice::processEvent(char eventId) {
  if (eventId == _AUDIOSTREAMINFO_STATUS_CHANGE) {
    switch(streamInfoDevice->getStatusInfo()->getStatus()){
    case _STATUS_STOPPED: {
      playMode = _STATUS_STOPPED;
      songStopped();  break;
    }
    case _STATUS_PLAYING: {
      if (playMode == _STATUS_STOPPED){
	songStarted();  
      }
      playMode = _STATUS_PLAYING;
      break;
    }
    case _STATUS_PAUSED: {
      if (playMode == _STATUS_PLAYING){
	playMode = _STATUS_PAUSED;
      }
      break;
    }
    }
  }
}

void FadeDevice::songStarted(){
  int vol;
  config->setGroup(KCONFIG_PLAYER);
  if (config->readBoolEntry(KCONFIG_PLAYER_FADEIN,true)){
    vol = 0;
    setFadeMode(_FADEMODE_FADEIN);
    startFadeTimer(false);
  }
  else{
    vol = 100;
  }
  volumeDevice->setVolume(vol); 
  emit volumeChanged(vol);
}

void FadeDevice::songStopped(){
  int vol;
  stopFading();
  config->setGroup(KCONFIG_PLAYER);
  if (config->readBoolEntry(KCONFIG_PLAYER_FADEIN,true))
    vol =   0;
  else
    vol = 100;
  volumeDevice->setVolume(vol); 
  emit volumeChanged(vol);
}



/* 
 * Public Slots
 */
void FadeDevice::setVolume(int value){  
  if(value > 100) value = 100;
  if(value <   0) value = 0;
  if(isFading()) stopFading();
  volumeDevice->setVolume(value); 
}

void FadeDevice::fadeIn(){
  if(isFading()) 
    stopFading();
  setFadeMode(_FADEMODE_FADEIN);
  startFadeTimer(true);
}


void FadeDevice::fadeOut(){
  if(isFading()) 
    stopFading();
  setFadeMode(_FADEMODE_FADEOUT);
  startFadeTimer(true);
}

void FadeDevice::fadeOutSoftly(){
  if(isFading()) 
    stopFading();
  setFadeMode(_FADEMODE_FADEOUT);
  startFadeTimer(false);
}

void FadeDevice::stopFading(){
  if (timerId)  {
    killTimer(timerId);
    counter = 0.0;
    timerId = 0;
    if(fadeMode == _FADEMODE_FADEIN)
      emit fadeInReady();
    else
      emit fadeOutReady();
  }
}





