/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "songInfo.h"
#include "mp3.h"
#include <kurl.h>
#include "iostream.h"

SongInfo::SongInfo(QWidget *parent, const char *name)
    : QDialog (parent,name)
{

  config = (KApplication::getKApplication())->getConfig();

  setMinimumSize(320,490);

  boxTag       = new QGroupBox("MP3 Tags" , this, "boxtag" );  
  catTitle     = new QLabel(i18n("Title:"),boxTag,"titel");
  valueTitle   = new QLineEdit(boxTag,"valtitel");
  valueTitle->setMaxLength (30);
  catArtist    = new QLabel(i18n("Artist:"),boxTag,"artist");
  valueArtist  = new QLineEdit(boxTag,"valartist");
  valueArtist->setMaxLength (30);
  catAlbum     = new QLabel(i18n("Album:"),boxTag,"album");
  valueAlbum   = new QLineEdit(boxTag,"valalbum");
  valueAlbum->setMaxLength (30);
  catYear      = new QLabel(i18n("Year:"),boxTag,"Year");
  valueYear    = new QLineEdit(boxTag,"valyear");
  valueYear->setMaxLength (4);
  catComment   = new QLabel(i18n("Comment:"),boxTag,"Comment");
  valueComment = new QLineEdit(boxTag,"valcomment");
  valueComment->setMaxLength (30);
  catGenre     = new QLabel(i18n("Genre:"),boxTag,"genre");
  valueGenre   = new KCombo(false,boxTag,"valgenre");
  setGenreBox();
  
  boxLayer     = new QGroupBox(i18n("MP3 Layer"), this, "boxlayer" );  
  catTime      = new QLabel(i18n("Time:"),boxLayer,"dauer");
  valueTime    = new QLabel(boxLayer,"time");
  catSize      = new QLabel(i18n("Size:"),boxLayer,"Size");
  valueSize    = new QLabel(boxLayer,"valsize");
  catPath      = new QLabel(i18n("Path:"),boxLayer,"path");
  valuePath    = new QLabel(boxLayer,"valPath");
  catFilename  = new QLabel(i18n("Filename:"),boxLayer,"filename");
  valueFilename   = new QLabel(boxLayer,"valFilename");
  catSamplerate   = new QLabel(i18n("Samplerate:"),boxLayer,"samplerate");
  valueSamplerate = new QLabel(boxLayer,"valSamplerate");
  catBitrate   = new QLabel(i18n("Bitrate:"),boxLayer,"bitrate");
  valueBitrate = new QLabel(boxLayer,"valBitrate");
  catStereo    = new QLabel(i18n("Stereo:"),boxLayer,"Stereo");
  valueStereo  = new QLabel(boxLayer,"valStereo");
  catLayer     = new QLabel(i18n("Layer:"),boxLayer,"Layer");
  valueLayer   = new QLabel(boxLayer,"valLayer");

  boxPrivate = new QGroupBox(i18n("Private")  , this, "boxprivate" );  
  catPlayed    = new QLabel(i18n("Played:"),boxPrivate,"Played");
  valuePlayed  = new QLabel(boxPrivate,"valplayed");
  catLastPlayed   = new QLabel(i18n("Last Played:"),boxPrivate,"lastplayed");
  valueLastPlayed = new QLabel(boxPrivate,"valLastPlayed");
  catFirstSeconds   = new QLabel(i18n("First Seconds"),boxPrivate,"firstsec");
  valueFirstSeconds = new QLineEdit(boxPrivate,"vfs");
  valueFirstSeconds->setMaxLength (4);
  catLastSeconds    = new QLabel(i18n("Last Seconds"), boxPrivate,"firstsec");
  valueLastSeconds  = new QLineEdit(boxPrivate,"vls");
  valueLastSeconds->setMaxLength (4);
  
  acceptButton = new QPushButton(i18n("OK"), this, "accept");
  connect(acceptButton, SIGNAL(clicked()), this, SLOT(accept()));
  applyButton = new QPushButton(i18n("Accept"), this, "apply");
  connect(applyButton, SIGNAL(clicked()), this, SLOT(apply()));
  cancelButton = new QPushButton(i18n("Cancel"), this, "cancel");
  connect(cancelButton, SIGNAL(clicked()), this, SLOT(closeWindow()));
  setGeometry(100,100,320,470);
}

void SongInfo::setGenreBox(){
  Tag tag;
  for(int i=0; i<=tag.getMaxGenres(); i++){
    valueGenre->insertItem(tag.getSortedGenreAsString(i),i);
  }
}

void SongInfo::showSong(){
  QString tmpS;
  Tag tag;

  tmpS  = songCopy.getArtist().data();
  tmpS += "(";
  tmpS += songCopy.getTitle().data();
  tmpS += ")";
  setCaption(tmpS.data());
  setMinimumSize(220,360);
  
  acceptButton->show();
  applyButton->show();
  
  valueTitle      ->setText(songCopy.getTitle().data());
  valueArtist     ->setText(songCopy.getArtist().data());
  valueAlbum      ->setText(songCopy.getAlbum().data());
  tmpS.sprintf("%d", songCopy.getYear());
  valueYear       ->setText(tmpS.data());
  valueComment    ->setText(songCopy.getComment().data());
  config->setGroup(KCONFIG_FILE);
  if( config->readBoolEntry(KCONFIG_FILE_MP3COMMENT, false) )
    valueComment->setEnabled(false);
  else
    valueComment->setEnabled(true);

  valueGenre->setCurrentItem(tag.getSortedGenreAsNumber(songCopy.getGenre().data()));
  tmpS.sprintf("%d:%02d", (songCopy.getSeconds()/60),(songCopy.getSeconds()%60));
  valueTime       ->setText(tmpS.data());
  tmpS.sprintf("%d", songCopy.getSize());
  valueSize       ->setText(tmpS.data());
  KURL file(songCopy.getFilename().data());
  valuePath       ->setText(songCopy.getFilename().data());  
  valueFilename   ->setText(file.filename());
  tmpS.sprintf("%d", songCopy.getSamplerate());
  valueSamplerate ->setText(tmpS.data());

  if(songCopy.getBitrate() == 0)
    tmpS = "variable (vbr)";
  else
    tmpS.sprintf("%d", songCopy.getBitrate());
  valueBitrate    ->setText(tmpS.data());
  valueStereo     ->setText(songCopy.getStereomode().data());
  valueLayer      ->setText(songCopy.getLayer().data());
  if (songCopy.getPlayed())    
    valuePlayed  ->setText("True");
  else     
    valuePlayed   ->setText("False");
  tmpS.sprintf("%d",songCopy.getLastPlayed());
  valueLastPlayed ->setText(tmpS.data());

  tmpS.sprintf("%d",songCopy.getFirstSeconds());
  valueFirstSeconds ->setText(tmpS.data());
  tmpS.sprintf("%d",songCopy.getLastSeconds());
  valueLastSeconds  ->setText(tmpS.data());
  if(!isVisible()) show();
}

void SongInfo::showMP3Song(Song *song){
  if(song == NULL) return;
  if(song->isNull()) return;
  songCopy = *song;
  songReference = song;
  showSong();
}

void SongInfo::referenceChanged(Song *oldRef, Song *newRef){
  if (songReference == oldRef){
    songReference = newRef;
    songCopy = *newRef;
    showSong();
  }
}

void SongInfo::referenceRemoved(Song *song){
  if (songReference == song){
    songReference = NULL;
    hide();
  }
}

void SongInfo::songRemoved(){
  songReference = NULL;
  hide();
}

void SongInfo::closeWindow(){
  hide();
}

void SongInfo::accept(){
  apply();
  closeWindow();
}

void SongInfo::apply(){
  Tag tag;
  bool changed;
  changed = false;

  songReference->setTitle(valueTitle->text(), false);
  songReference->setArtist(valueArtist->text(), false);
  songReference->setAlbum(valueAlbum->text(), false);
  songReference->setComment(valueComment->text(), false);
  songReference->setYear(atoi(valueYear->text()), false);
  songReference->setGenre(tag.getSortedGenreAsString(valueGenre->currentItem()), false);
  songReference->setFirstSeconds(atoi(valueFirstSeconds->text()), false);
  songReference->setLastSeconds (atoi(valueLastSeconds ->text()));

  config->setGroup(KCONFIG_FILE);
  strncpy(tag.title,  songReference->getTitle(),  30);
  strncpy(tag.artist, songReference->getArtist(), 30);
  strncpy(tag.album,  songReference->getAlbum(),  30);
  strncpy(tag.comment,songReference->getComment(),30);
  strncpy(tag.genre,  songReference->getGenre(),  30);
  tag.gennum = tag.getGenreAsNumber(songReference->getGenre());
  sprintf(tag.year,"%d",songReference->getYear());
  
  mp3.saveTag(songReference->getFilename(),tag);
  if( config->readBoolEntry(KCONFIG_FILE_MP3COMMENT, false) ){
    saveComment();
  }

  if (!(songCopy == *songReference)) 
    emit songChanged(songReference);
}

bool SongInfo::saveComment(){
  QString comment;
  comment.sprintf(MP3_COMMENT_FIELD,
		  songReference->getFirstSeconds(),
		  songReference->getLastSeconds());
  songReference->setComment(comment);
  return mp3.saveCommentTag(songReference->getFilename(),comment);
}



void SongInfo::resizeEvent( QResizeEvent * ){
  int fb = 100;
  int fs = 110;
  boxTag->setGeometry(5,5,width()-10,140);
  catTitle   ->setGeometry( 10, 15, fb, 20);
  catArtist  ->setGeometry( 10, 35, fb, 20);
  catAlbum   ->setGeometry( 10, 55, fb, 20);
  catYear    ->setGeometry( 10, 75, fb, 20);
  catComment ->setGeometry( 10, 95, fb, 20);
  catGenre   ->setGeometry( 10,115, fb, 20);
  valueTitle   ->setGeometry(fs, 15,boxTag->width()-10-fs, 20);
  valueArtist  ->setGeometry(fs, 35,boxTag->width()-10-fs, 20);
  valueAlbum   ->setGeometry(fs, 55,boxTag->width()-10-fs, 20);
  valueYear    ->setGeometry(fs, 75,boxTag->width()-10-fs, 20);
  valueComment ->setGeometry(fs, 95,boxTag->width()-10-fs, 20);
  valueGenre   ->setGeometry(fs,115,boxTag->width()-10-fs, 20);

  boxLayer->setGeometry( 5,150,width()-10,185);
  catTime      ->setGeometry( 10, 15, fb, 20);
  catSize      ->setGeometry( 10, 35, fb, 20);
  catPath      ->setGeometry( 10, 55, fb, 20);
  catFilename  ->setGeometry( 10, 75, fb, 20);
  catSamplerate->setGeometry( 10, 95, fb, 20);
  catBitrate   ->setGeometry( 10,115, fb, 20);
  catStereo    ->setGeometry( 10,135, fb, 20);
  catLayer     ->setGeometry( 10,155, fb, 20);
  valueTime      ->setGeometry(fs, 15,boxLayer->width()-10-fs, 20);
  valueSize      ->setGeometry(fs, 35,boxLayer->width()-10-fs, 20);
  valuePath      ->setGeometry(fs, 55,boxLayer->width()-10-fs, 20);
  valueFilename  ->setGeometry(fs, 75,boxLayer->width()-10-fs, 20);
  valueSamplerate->setGeometry(fs, 95,boxLayer->width()-10-fs, 20);
  valueBitrate   ->setGeometry(fs,115,boxLayer->width()-10-fs, 20);
  valueStereo    ->setGeometry(fs,135,boxLayer->width()-10-fs, 20);
  valueLayer     ->setGeometry(fs,155,boxLayer->width()-10-fs, 20);

  boxPrivate->setGeometry( 5,340,width()-10,105);
  catPlayed    ->setGeometry( 10, 15, fb, 20);
  catLastPlayed->setGeometry( 10, 35, fb, 20);
  catFirstSeconds ->setGeometry( 10, 55, fb, 20);
  catLastSeconds  ->setGeometry( 10, 75, fb, 20);
  valuePlayed    ->setGeometry(fs, 15,boxLayer->width()-10-fs, 20);
  valueLastPlayed->setGeometry(fs, 35,boxLayer->width()-10-fs, 20);
  valueFirstSeconds ->setGeometry( fs, 55, 40, 20);
  valueLastSeconds  ->setGeometry( fs, 75, 40, 20);

  acceptButton ->setGeometry(10,height()-35,80,30);
  applyButton  ->setGeometry(((width()/2)-40),height()-35,80,30);
  cancelButton ->setGeometry(width()-90,height()-35,80,30);
}






