/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include <qfileinfo.h> 
#include "songLogger.h"
#include "stdlib.h"

SongLogger::SongLogger(QWidget *parent, const char *name )
  :QObject(parent, name){
  int width, height;

  config = (KApplication::getKApplication())->getConfig();  
  config->setGroup(KCONFIG_SONGLOGGER);
  enable = config->readBoolEntry(KCONFIG_SONGLOGGER_ENABLE,"true");
  width  = config->readNumEntry(KCONFIG_SONGLOGGER_WIDTH,  600);
  height = config->readNumEntry(KCONFIG_SONGLOGGER_HEIGHT, 200);

  

  table = new KTabListBox(0, KCONFIG_SONGLOGGER, 6);
  table->setColumn(0, i18n("Date"), 20, KTabListBox::TextColumn, KTabListBox::SimpleOrder, KTabListBox::Descending, false, compareDateTime );
  table->setColumn(1, i18n("Artist"), 20, KTabListBox::TextColumn, KTabListBox::SimpleOrder, KTabListBox::Descending, false, compareString);
  table->setColumn(2, i18n("Album"), 20, KTabListBox::TextColumn, KTabListBox::SimpleOrder, KTabListBox::Descending, false, compareString);
  table->setColumn(3, i18n("Song"), 20, KTabListBox::TextColumn, KTabListBox::SimpleOrder, KTabListBox::Descending, false, compareString);
  table->setColumn(4, i18n("Time"), 20, KTabListBox::TextColumn, KTabListBox::SimpleOrder, KTabListBox::Descending, false, compareTime);
  table->setColumn(5, i18n("Filename"), 20, KTabListBox::TextColumn, KTabListBox::SimpleOrder, KTabListBox::Descending, false, compareString);

  table->setSeparator('|');
  table->setTableFlags(Tbl_vScrollBar|Tbl_hScrollBar);
  QString caption;
  caption.sprintf("%s: %s",
		  i18n("View of the song logs"),
		  config->readEntry(KCONFIG_SONGLOGGER_LOGFILE,"kjukebox.log").data());
  table->setCaption(caption.data());
  table->resize(width,height);
  table->readConfig();
  table->setAutoUpdate(true);  

  openLogFile();
  readLogFile();    
}

SongLogger::~SongLogger(){
  config->setGroup(KCONFIG_SONGLOGGER);
  QString width;

  table->writeConfig();
  config->writeEntry(KCONFIG_SONGLOGGER_WIDTH, table->width());  
  config->writeEntry(KCONFIG_SONGLOGGER_HEIGHT, table->height());  
  config->sync();
  logFile.close();
}


void SongLogger::show(){  if(!(table->isVisible())) table->show();}
void SongLogger::hide(){  if(  table->isVisible() ) table->hide();}
void SongLogger::toggleVisible(){
  if(table->isVisible())
    table->hide();
  else 
    table->show();
}

void SongLogger::reinitialize(){
  config->setGroup(KCONFIG_SONGLOGGER);
  enable = config->readBoolEntry(KCONFIG_SONGLOGGER_ENABLE,"true");
  openLogFile();
  readLogFile();
}

void SongLogger::openLogFile(){
  logFile.close();
  
  if(!enable) return ;
  QString caption;
  caption.sprintf("%s: %s",
		  i18n("View of the song logs"),
		  config->readEntry(KCONFIG_SONGLOGGER_LOGFILE,"kjukebox.log").data());
  table->setCaption(caption.data());

  config->setGroup(KCONFIG_SONGLOGGER);
  logFile.open(config->readEntry(KCONFIG_SONGLOGGER_LOGFILE,"kjukebox.log"), ios::out|ios::app);
  if(!logFile) 
    perror("open logfile: ");
}

void SongLogger::readLogFile(){
  if(!enable) return;

  QString tmp;
  char *entry = new char[4000];

  config->setGroup(KCONFIG_SONGLOGGER);
  ifstream inLogFile;
  inLogFile.open(config->readEntry(KCONFIG_SONGLOGGER_LOGFILE,"kjukebox.log"), ios::in);
  
  table->setAutoUpdate(false);
  table->clear();
  while(inLogFile){
    inLogFile.getline(entry,4000);
    tmp = entry;
    tmp.stripWhiteSpace();
    if(!(tmp.isEmpty()))
      table->appendItem(tmp);
  }
  table->setAutoUpdate(true);
  table->repaint();
  inLogFile.close();
}

void SongLogger::addLogEntry(const char *entry){
  debugOutput(cout << entry << endl);
  if(enable) 
    logFile << entry << endl;
}

void SongLogger::appendSong(Song *song){
  if(!enable) return;
  if(song==0) return;
  QDateTime time = QDateTime::currentDateTime();
  QString dateTimeDescr;
  QString timeDescr;
  QString description;
  QFileInfo file;
  

  timeDescr.sprintf("%d:%02d", song->getSeconds()/60, song->getSeconds()%60);
  dateTimeDescr.sprintf("%d:%02d:%02d %d.%d.%d", 
			time.time().hour(),
			time.time().minute(),
			time.time().second(),
			time.date().day(),
			time.date().month(),
			time.date().year());

  file.setFile(song->getFilename().data());

  description.sprintf("%s|%s|%s|%s|%s|%s", dateTimeDescr.data(), song->getArtist().data(), song->getAlbum().data(), song->getTitle().data(), timeDescr.data(), file.fileName().data());

  table->appendItem(description);
  addLogEntry(description.data());
}


/* 
   Some static compare function for sorting the log view 
*/
int SongLogger::compareString(const char *a, const char *b){
  return stricmp(a,b);
}

int SongLogger::compareTime(const char *a, const char *b){
  int aSec, bSec;
  int aMin, bMin;

  sscanf(a,"%d:%d", &aMin, &aSec);
  sscanf(b,"%d:%d", &bMin, &bSec);
  
  if(aMin < bMin) return -1;
  if(aMin > bMin) return  1;
  if(aSec < bSec) return -1;
  if(aSec > bSec) return  1;
  return 0;
}

int SongLogger::compareDateTime(const char *a, const char *b){
  int aSec, bSec;
  int aMin, bMin;
  int aH, bH;
  int aDay, bDay;
  int aMon, bMon;
  int aYear, bYear;

  sscanf(a,"%d:%d:%d %d.%d.%d", &aH, &aMin, &aSec, &aDay, &aMon, &aYear);
  sscanf(b,"%d:%d:%d %d.%d.%d", &bH, &bMin, &bSec, &bDay, &bMon, &bYear);

  if(aYear < bYear) return -1;
  if(aYear > bYear) return  1;

  if(aMon < bMon) return -1;
  if(aMon > bMon) return  1;

  if(aDay < bDay) return -1;
  if(aDay > bDay) return  1;

  if(aH < bH) return -1;
  if(aH > bH) return  1;

  if(aMin < bMin) return -1;
  if(aMin > bMin) return  1;

  if(aSec < bSec) return -1;
  if(aSec > bSec) return  1;

  return 0;


}
