/*
 *	      Main module for kmikmod
 *	(C)1998, 1999 Roope Anttinen - roope.anttinen@ntc.nokia.com
 *	
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#include "config.h"
#include "Main.h"
#include <qtooltip.h>
#include <kmsgbox.h>
#include <qstring.h>

KApplication *app = NULL;
int main (int argc, char **argv)
{
	app = new KApplication (argc, argv, "kmikmod");
	Main *ikkuna = new Main (argc, argv);
	ikkuna->setCaption ("KMikMod");
	app->setMainWidget (ikkuna);
	ikkuna->show();
	return app->exec();
}

void Main::zeroLCD ()
{
	LCD_play_time->display("00:00");
	LCD_pat_position->display (0);
	LCD_pat_lenght->display (0);
	LCD_pattern->display (0);
	LCD_tot_pats->display (0);
	LCD_bpm->display (0);
}

Main::Main (int argc, char **argv)
{
	QString tmp1, tmp2, tmp3;
	int R, G, B;
	configuration conf;

	playlist = new PlayList();
	conf_dlg = new ConfigDLG();

	feed_current = false;
	
	saveDir =  KApplication::localkdedir().data();
	saveDir += "/share/apps/kmikmod";

	KConfig *config = KApplication::getKApplication()->getConfig();

	config->setGroup ("Geometry");
	tmp1 = config->readEntry ("X pos", "100");
	tmp2 = config->readEntry ("Y pos", "200");
	Geometry = QRect(tmp1.toInt(), tmp2.toInt(),
			 MainW_H_Size, MainW_W_Size);
	setGeometry (Geometry);

	config->setGroup ("MikMod");
	tmp1 = config->readEntry ("mixfreq", "44100");
	conf.md_mixfreq = tmp1.toInt();
	
	conf.md_mode = DMODE_SOFT_MUSIC;
	
	tmp1 = config->readEntry ("stereo", "yes");
	if (tmp1 == "yes") conf.md_mode |= DMODE_STEREO;
	else conf.md_mode = 0;
	
	tmp1 = config->readEntry ("16 bit", "yes");
	if (tmp1 == "yes") conf.md_mode |= DMODE_16BITS;
	
	tmp1 = config->readEntry ("surround", "no");
	if (tmp1 == "yes") conf.md_mode |= DMODE_SURROUND;
	
	tmp1 = config->readEntry ("interpolate", "no");
	if (tmp1 == "yes") conf.md_mode |= DMODE_INTERP;
	
	tmp1 = config->readEntry ("hqmixer", "no");
	if (tmp1 == "yes") conf.md_mode |= DMODE_HQMIXER;
	
	tmp1 = config->readEntry ("driver", "0");
	conf.md_device = tmp1.toInt();
	
	tmp1 = config->readEntry ("volume", "96");
	conf.volume = tmp1.toInt();
	
	tmp1 = config->readEntry ("sleep time", "1100");
	conf.sleep_time = tmp1.toLong();

	tmp1 = config->readEntry ("fadeout", "no");
	if (tmp1 == "yes") conf.fadeout = true;
	else conf.fadeout = false;

#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	tmp1 = config->readEntry ("realtime scheduler", "no");
	if (tmp1 == "yes") conf.use_rt = true;
	else conf.use_rt = false;

	if (geteuid()!=0) {
		conf.use_rt = false;
		conf_dlg->disableSuid();
	}
#else
	conf_dlg->disableSuid();
#endif // _POSIX_THREAD_PRIORITY_SCHEDULING
	setConfigInfo(conf);
	
	if (argc>1) for (int i=1; i<argc; i++) playlist->addToList(argv[i]);
	itemsChanged();

	config->setGroup ("Colour");
	tmp1 = config->readEntry ("baseBg R", "0");
	tmp2 = config->readEntry ("baseBg G", "0");
	tmp3 = config->readEntry ("baseBg B", "0");
	R = tmp1.toInt();
	G = tmp2.toInt();
	B = tmp3.toInt();
	t_LCD_bg = QColor(R, G, B);
	
	tmp1 = config->readEntry ("baseFg R", "127");
	tmp2 = config->readEntry ("baseFg G", "255");
	tmp3 = config->readEntry ("baseFg B", "212");
	R = tmp1.toInt();
	G = tmp2.toInt();
	B = tmp3.toInt();
	t_LCD_fg = QColor(R, G, B);
	
	tmp1 = config->readEntry ("modinfoBg R", "0");
	tmp2 = config->readEntry ("modinfoBg G", "0");
	tmp3 = config->readEntry ("modinfoBg B", "0");
	R = tmp1.toInt();
	G = tmp2.toInt();
	B = tmp3.toInt();
	LCD_bg = QColor(R, G, B);
	
	tmp1 = config->readEntry ("modinfoFg R", "255");
	tmp2 = config->readEntry ("modinfoFg G", "230");
	tmp3 = config->readEntry ("modinfoFg B", "2");
	R = tmp1.toInt();
	G = tmp2.toInt();
	B = tmp3.toInt();
	LCD_fg = QColor(R, G, B);

	setColors();
	conf_dlg->setColors(t_LCD_fg, LCD_fg);

	player = new ModPlayer (playlist, conf);
	conf_dlg->setDriverInfo(player->getDrivers());

	VolumeSlider->setValue(conf.volume);

	QToolTip::add (LCD_pat_position, i18n("Position in pattern"));
	QToolTip::add (LCD_pat_lenght, i18n("Current pattern lenght"));
	QToolTip::add (LCD_pattern, i18n("Current pattern"));
	QToolTip::add (LCD_tot_pats, i18n("Total patterns (song lenght)"));
	QToolTip::add (LCD_bpm, i18n("bpm (song speed)"));
	QToolTip::add (PB_Previous, i18n("Previous song"));
	QToolTip::add (PB_Next, i18n("Next song"));
	QToolTip::add (PB_Stop, i18n("Stop player"));
	QToolTip::add (PB_PlayPause, i18n("Pause/Play"));
	QToolTip::add (PB_Quit, i18n("Quit KMikMod"));
	QToolTip::add (VolumeSlider, i18n("Set volume"));
	QToolTip::add (PB_FR, i18n("Step backward"));
	QToolTip::add (PB_FF, i18n("Step forward"));
	QToolTip::add (PB_Eject, i18n("Playlist editor"));
	QToolTip::add (PB_Configure, i18n("Configure KMikMod"));

	tmp1 = KApplication::kde_datadir().data();
	tmp2 = tmp1 + "/kmikmod/pics/km_logo.xpm";
	PB_About->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_configure.xpm";
	PB_Configure->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_eject.xpm";
	PB_Eject->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_fb.xpm";
	PB_FR->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_ff.xpm";
	PB_FF->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_next.xpm";
	PB_Next->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_playpause.xpm";
	PB_PlayPause->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_prev.xpm";
	PB_Previous->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_quit.xpm";
	PB_Quit->setPixmap(QPixmap(tmp2));
	tmp2 = tmp1 + "/kmikmod/pics/km_stop.xpm";
	PB_Stop->setPixmap(QPixmap(tmp2));

// Disable unfunctional pushbuttons
//	PB_About->setEnabled(false);

	timer = new QTimer ( this );
	
	KDNDDropZone *dnd = new KDNDDropZone(FR_Main, DndURL);
	
	connect (timer, SIGNAL(timeout()), SLOT (timeoutCB()));
	connect (app,SIGNAL(appearanceChanged()),this,SLOT(setColors()));
	connect (app, SIGNAL(lastWindowClosed()),this,SLOT(aboutToExit()));
	connect (PB_Eject, SIGNAL(clicked()), SLOT(ejectCB()));
	connect (PB_Configure, SIGNAL(clicked()), SLOT(confCB()));
	connect (playlist, SIGNAL(applyClicked()), SLOT(playCB()));
	connect (conf_dlg, SIGNAL(applyButtonPressed()), SLOT(confChangedCB()));
	connect (dnd, SIGNAL(dropAction(KDNDDropZone *)),
		SLOT(Dropped(KDNDDropZone *)));
	connect (playlist, SIGNAL(itemsChanged()), SLOT(itemsChanged()));
	timer->start(250);
}

Main::~Main ()
{
	KConfig *config = KApplication::getKApplication()->getConfig();

	config->setGroup ("Geometry");
	config->writeEntry ("X pos", x());
	config->writeEntry ("Y pos", y());

	configuration conf = player->getConfig();
	config->setGroup ("MikMod");
 	config->writeEntry ("mixfreq", conf.md_mixfreq);
	config->writeEntry ("stereo", conf.md_mode&DMODE_STEREO ? "yes" : "no");
	config->writeEntry ("16 bit", conf.md_mode&DMODE_16BITS ? "yes" : "no");
	config->writeEntry ("interpolate", conf.md_mode&DMODE_INTERP ?
			    "yes" : "no");
	config->writeEntry ("hqmixer", conf.md_mode&DMODE_HQMIXER ?
			    "yes" : "no");
	config->writeEntry ("surround", conf.md_mode&DMODE_SURROUND ?
			    "yes" : "no");
	config->writeEntry ("driver", conf.md_device);
	config->writeEntry ("volume", conf.volume);
	config->writeEntry ("sleep time", conf.sleep_time);
	config->writeEntry ("fadeout", conf.fadeout ? "yes" : "no");

#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	config->writeEntry ("realtime scheduler", conf.use_rt ? "yes": "no");
#endif	
	config->setGroup ("Colour");
	config->writeEntry ("baseBg R", t_LCD_bg.red());
	config->writeEntry ("baseBg G", t_LCD_bg.green());
	config->writeEntry ("baseBg B", t_LCD_bg.blue());
	
	config->writeEntry ("baseFg R", t_LCD_fg.red());
	config->writeEntry ("baseFg G", t_LCD_fg.green());
	config->writeEntry ("baseFg B", t_LCD_fg.blue());
	
	config->writeEntry ("modinfoBg R", LCD_bg.red());
	config->writeEntry ("modinfoBg G", LCD_bg.green());
	config->writeEntry ("modinfoBg B", LCD_bg.blue());
	
	config->writeEntry ("modinfoFg R", LCD_fg.red());
	config->writeEntry ("modinfoFg G", LCD_fg.green());
	config->writeEntry ("modinfoFg B", LCD_fg.blue());
 	config->sync();
	
	delete playlist;
	delete conf_dlg;
	delete player;
}

void Main::playPauseCB()
{
	if (player->playerActive()) player->togglePause();
	else playCB();
}

void Main::playCB()
{
	player->play();
	feed_current = false;
}

void Main::quitCB()
{
	playlist->close();
	conf_dlg->close();
	close();
}

void Main::volumeCB(int p)
{
	char tmp [5];
	sprintf (tmp,"%3i%%", p*100/128);
	LBL_Volume->setText(tmp);
	player->setVolume (p);
}

void Main::nextCB()
{
	if(playlist->next()) playCB();
}

void Main::prevCB()
{
	if(playlist->previous()) playCB();
}

void Main::stopCB()
{
	zeroLCD();
	player->stop();
}

void Main::frCB()
{
	player->step (-2);
}

void Main::ffCB()
{
	player->step (+2);
}

void Main::timeoutCB()
{
	static bool first_time = true;
	int playtime = player->playtime();
	int minutes = playtime/60;
	int seconds = playtime-((playtime/60)*60);
	char tmp [5];

	sprintf (tmp, "%02i:%02i", minutes, seconds);
	LCD_play_time->display(tmp);

	playdata Data = player->getData();
	LCD_pat_position->display ((int)Data.PatPosition);
	LCD_pat_lenght->display ((int)Data.PatLenght);
	LCD_pattern->display ((int)Data.PatNumber);
	LCD_tot_pats->display ((int)Data.TotPatterns);
	LCD_bpm->display ((int)Data.Tempo);
	
	if (first_time) {
		playCB();
		first_time = false;
	}
	
	if (player->infoChanged()) {
		playinfo Info = player->getInfo();
		QString stat ("KMikMod - ");
		stat += Info.Status;
		this->setCaption (stat);
	}

	if (player->feedPlayer()){
		if (feed_current) playCB();
		else if (playlist->next()) playCB();
		else player->stopFeeding();
	}
}

void Main::setColors()
{
	QFont font( "Helvetica", 10, 75, 0 );
	font.setStyleHint( (QFont::StyleHint)0 );
	font.setCharSet( (QFont::CharSet)0 );

	LCD_off = LCD_fg.dark(300);
	t_LCD_off = t_LCD_fg.dark(300);
	i_LCD_fg = t_LCD_fg.dark(150);

	QColorGroup normal(t_LCD_fg, t_LCD_bg,
			   t_LCD_bg, t_LCD_fg, t_LCD_off,
			   i_LCD_fg, t_LCD_bg);
			    
	QColorGroup disabled(t_LCD_bg, t_LCD_off,
			     t_LCD_bg, t_LCD_fg, t_LCD_off,
			     t_LCD_off, t_LCD_bg);
	palette = QPalette ( normal, disabled, normal );

	FR_Main->setPalette( palette );
	LCD_play_time->setColors ( t_LCD_fg, t_LCD_bg, t_LCD_off );
	LCD_pat_position->setColors( LCD_fg, LCD_bg, LCD_off );
	LCD_pat_lenght->setColors( LCD_fg, LCD_bg, LCD_off );
	LCD_pattern->setColors( LCD_fg, LCD_bg, LCD_off );
	LCD_tot_pats->setColors( LCD_fg, LCD_bg, LCD_off );
	LCD_bpm->setColors( LCD_fg, LCD_bg, LCD_off );
	LBL_Info->setPalette (palette);
	LBL_Info->setFont (font);
	LBL_Volume->setPalette (palette);
	LBL_Volume->setFont (font);
	LBL_PBInfo->setPalette (palette);
	LBL_PBInfo->setFont (font);
}

void Main::ejectCB()
{
	if (playlist->isVisible()) playlist->hide();
	else playlist->show();
}

void Main::confCB()
{
	if (conf_dlg->isVisible()) conf_dlg->hide();
	else {
		conf_dlg->setConfig(player->getConfig());
		conf_dlg->setColors(t_LCD_fg, LCD_fg);
		conf_dlg->show();
	}
}

void Main::confChangedCB()
{
	if (conf_dlg->colorsChanged()) {
		LCD_fg = conf_dlg->infoColor();
		t_LCD_fg = conf_dlg->timeColor();
		setColors();
	}
	configuration tmp=conf_dlg->getConfig(), tmp2=player->getConfig();
	if (tmp != tmp2) {
		feed_current = true;
		player->setConfig(tmp);
		setConfigInfo(tmp);
	}
}

void Main::setConfigInfo (const configuration &conf)
{
	QString s_tmp;
	char tmp[8];

	if (conf.md_mode&DMODE_16BITS) s_tmp = "16b  ";
	else s_tmp = "8b    ";
	
	sprintf (tmp, "%iHz", conf.md_mixfreq);
	s_tmp += tmp;

	if (conf.md_mode&DMODE_STEREO) s_tmp += "  Stereo ";
	else s_tmp += "  Mono ";
	
	if (conf.md_mode&DMODE_INTERP) s_tmp += " I";
	else s_tmp += "  ";
	
	if (conf.md_mode&DMODE_HQMIXER) s_tmp += " H ";
	else s_tmp += "   ";
	
	if (conf.md_mode&DMODE_SURROUND) s_tmp += "S ";
	else s_tmp += "  ";
	
	if (conf.fadeout) s_tmp += "F";
	
	LBL_Info->setText(s_tmp);
}

void Main::itemsChanged ()
{
	QString tmp;
	tmp.sprintf("%i/%i",playlist->current(),playlist->items());
	LBL_PBInfo->setText(tmp);
}

void Main::aboutToExit ()
{
	cerr << "kmikmod: exit" << endl;
	delete this;
}

void Main::Dropped(KDNDDropZone *p)
{
	pthread_mutex_lock (&playlist->PL);
	playlist->emptyList();
	playlist->Dropped(p);
	pthread_mutex_unlock (&playlist->PL);
	player->play();
}
