/*
  encapsulates the syncing methods, to use it in other classes
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "avSyncer.h"



AVSyncer::AVSyncer(int bufferSize) {
  oneFrameTime=0;
  onePicFrameInAudioBytes=0;
  this->bufferSize=bufferSize;

  pthread_mutex_init(&writeInMut,NULL);
  pthread_mutex_init(&changeMut,NULL);
  pthread_cond_init(&changeCond,NULL);
  
  audioDataInsert=new AudioData();
  audioDataArray=new AudioDataArray(400);
  audioDataCurrent=audioDataArray->readAudioData();
  
  startAudio=new TimeStamp();
  endAudio=new TimeStamp();
  audioTime=new AudioTime();
  lAudioRunning=false;
}


AVSyncer::~AVSyncer() {
  delete audioDataArray;
  delete audioDataInsert;
  delete audioTime;
  delete startAudio;
  delete endAudio;
}



int AVSyncer::audioSetup(int frequency,int stereo,int sign,
			 int big,int sixteen) {
  audioTime->setFormat(stereo,sixteen,frequency);
  return true;
}


int AVSyncer::audioPlay(TimeStamp* startStamp,
			TimeStamp* endStamp,char *buffer, int size) {



  audioDataInsert->setStart(startStamp);
  audioDataInsert->setEnd(endStamp);
  audioDataInsert->setAudioTime(audioTime);
  setAudioSync(audioDataInsert);
  return size;
}



void AVSyncer::audioClose(void) {
  lockSyncData();
  setAudioRunning(false);
  audioDataArray->clear();
  unlockSyncData();
}


void AVSyncer::audioFlush() {
  audioClose();
}



int AVSyncer::getPreferredDeliverSize() {
  return onePicFrameInAudioBytes;
}



int AVSyncer::getFrameusec() {
  lockSyncData();
  int back=oneFrameTime;
  unlockSyncData();
  return back; 
}



void AVSyncer::config(char* key, char* value) {
  if (strcmp(key,"-b")==0) {
    bufferSize=strtol(value,(char **)NULL,10);
    cout << "simulated audio buffersize:"<<bufferSize<<" bytes"<<endl;
  }
}



void AVSyncer::setAudioSync(AudioData* audioData) {


  if (onePicFrameInAudioBytes <= 0) {
    // no
    return;
  }
  int entries=bufferSize/onePicFrameInAudioBytes;

  lockSyncData();

  audioDataArray->insertAudioData(audioData);
  if (audioDataArray->getFillgrade() >= entries) {
    audioDataCurrent=audioDataArray->readAudioData();
    setAudioRunning(true);
    audioDataArray->forward();
  }
  unlockSyncData();    
  
}


int AVSyncer::avSync(TimeStamp* startVideo,
		     TimeStamp* endVideo,
		     TimeStamp* waitTime,
		     Picture* pic) {

  long picTime=1000000;
  float picPerSec=pic->getPicturePerSecond();

  int videoStartAudioPacketNr=startVideo->getAudioPacketNr();
  int videoStartVideoPacketNr=startVideo->getVideoPacketNr();
  
  int videoEndAudioPacketNr=endVideo->getAudioPacketNr();
  int videoEndVideoPacketNr=endVideo->getVideoPacketNr();

  int avRatioVideoStart=startVideo->getAVRatioGlobal();
  int avRatioVideoEnd=endVideo->getAVRatioGlobal();
  
  int videoStartAudioRatio=startVideo->getAudioRatioLocal();
  int videoStartAudioPacketNrMinor=startVideo->getAudioPacketNrMinor();

  double videoStartFloatTime=startVideo->getFloatTimeStamp();
  double videoStartPTSTime=startVideo->getPTSTimeStamp();
  double videoStartDTSTime=startVideo->getDTSTimeStamp();
  double videoStartEndPTSTime=startVideo->getEndPTSTimeStamp();
  double videoEndPTSTime=endVideo->getPTSTimeStamp();
 
  int oneFrameTime_90KHz;
  lockSyncData();
  
  if (picPerSec > 0) {
    oneFrameTime=(long)1000000.0/picPerSec;
    oneFrameTime_90KHz=90000/picPerSec;

    onePicFrameInAudioBytes=(float)audioTime->calculateBytes(1.0/picPerSec);

  }
  if (lAudioRunning==false) {
    unlockSyncData();
    return true;
  }
   
  TimeStamp* startAudio=audioDataCurrent->getStart();
  TimeStamp* endAudio=audioDataCurrent->getEnd();

  /*
  startAudio->print("audio");
  startVideo->print("video");
  */

  double audioStartFloatTime=startAudio->getFloatTimeStamp();
  double audioStartPTSTime=startAudio->getPTSTimeStamp();
  double audioStartEndPTSTime=startAudio->getEndPTSTimeStamp();
  double audioStartDTSTime=startAudio->getDTSTimeStamp();

  
  int audioStartAudioPacketNrMinor=startAudio->getAudioPacketNrMinor();
 
  int audioStartAudioPacketNr=startAudio->getAudioPacketNr();
  int audioStartVideoPacketNr=startAudio->getVideoPacketNr();
  int audioStartAudioRatio=startAudio->getAudioRatioLocal();

  int avRatioAudioStart=startAudio->getAVRatioGlobal();
  int avRatioAudioEnd=endAudio->getAVRatioGlobal();

  int audioEndAudioPacketNr=endAudio->getAudioPacketNr();
  int audioEndVideoPacketNr=endAudio->getVideoPacketNr();

 
  float foneFrameTime=(float)oneFrameTime/1000000.0;

  /*
  cout << "audioStartAudioPacketNrMinor:"<<audioStartAudioPacketNrMinor<<endl;
  cout << "audioStartPTSTime:"<<audioStartPTSTime<<endl;
  cout << "audioStartEndPTSTime:"<<audioStartEndPTSTime<<endl;
  cout << "videoStartPTSTime:"<<videoStartPTSTime<<endl;
  */



  if (audioStartPTSTime > videoStartPTSTime) {

    if (audioStartAudioPacketNrMinor == 1) {
      unlockSyncData();
      return true;
    } 
    unlockSyncData();
    return false;
  }

  if (audioStartAudioPacketNrMinor > 1) {
    unlockSyncData();
    return true;
  } 


  float videoDiff=videoStartPTSTime-audioStartPTSTime;

  videoDiff=videoDiff*1000000;


  if (videoDiff > oneFrameTime) {
    int add=oneFrameTime;
    if (videoDiff > 5*oneFrameTime) {
      add=5*oneFrameTime;
    }
    if (videoDiff > 10*oneFrameTime) {
      add=10*oneFrameTime;
    }
    if (videoDiff > 20*oneFrameTime) {
      add=20*oneFrameTime;
    }
    cout << "add diff is"<<add<<endl;
    waitTime->set(0,add);
  }


  unlockSyncData();
  return true;
}
  
int AVSyncer::getAudioRunning() {
  return lAudioRunning;
}


void AVSyncer::setAudioRunning(int lAudioRunning) {
  this->lAudioRunning=lAudioRunning;
}
 

void AVSyncer::lockSyncData() {
  pthread_mutex_lock(&changeMut);
  pthread_mutex_lock(&writeInMut);
  pthread_mutex_unlock(&changeMut);
}

void AVSyncer::unlockSyncData() {
  pthread_mutex_unlock(&writeInMut); 
}
