/*
  device to stream to esd daemon
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <devices/esdDevice.h>


#ifndef HAVE_ESD_H

// define empty prototypes:
ESDDevice::ESDDevice():OutputDevice("ESDDevice"){}
ESDDevice::~ESDDevice(){}

int ESDDevice::getEnabled(){return false;}
void ESDDevice::setEnabled(int){};
int ESDDevice::open(){return false;}
int ESDDevice::close(){return true;}
int ESDDevice::isOpen(){return false;}
int ESDDevice::init(int ,int ,int ){return false;}
int ESDDevice::getAutoInit(){return false;}
void ESDDevice::setAutoInit(int){}
StatusInfo* ESDDevice::getStatusInfo(){return NULL;}
char* ESDDevice::getNodeName(){return "ESDDevice";}
void ESDDevice::writeIn(NodeDevice* ,DeviceConfig* ){printf("esd not compiled\n");}
int ESDDevice::rawWrite(char* ,int len){return len;}
int ESDDevice::isCompiledIn() {return false;}

#else

#include <esd.h>
 

ESDDevice::ESDDevice():OutputDevice("ESDDevice") {
  statusInfo=new StatusInfo();
  lneedInit=true;
  lAutoInit=true;
  lEnabled=true;

  rate=0;
  sock=-1;
  bits=ESD_BITS16;
  channels=ESD_STEREO;
  mode=ESD_STREAM;
  func=ESD_PLAY;
  host="";
  name="";

  nativeMixerDevice= new NativeMixerDevice("");
  nativeMixerDevice->open();
}


ESDDevice::~ESDDevice() {
  close();
  delete statusInfo;
  delete nativeMixerDevice;
}


int ESDDevice::isCompiledIn() {
  return true;
}


char* ESDDevice::getNodeName() {
  return "ESDDevice";
}

int ESDDevice::open() {
  int format=(bits | channels | mode | func);
  sock = esd_play_stream_fallback(format,rate,NULL,name );
  cout << "********* esd sock:"<<sock<<endl;
  cout << "********* esd rate:"<<rate<<endl;
  printf("format:%8x\n",format);
  // Ok here something important if your programm forks:
  if (sock > 0) {
    if (fcntl(sock,F_SETFD,true) < 0) {
      perror("fcntl sock");exit(1);
    }
  }

  return (sock > 0);
}


int ESDDevice::isOpen() {
  return (sock > 0);
}


int ESDDevice::close() {
  if (sock > 0) {
    if (::close(sock) < 0) {
      perror("error close ESDDevice:");
    }
    sock=-1;
  }
  return true;
}


int ESDDevice::init(int sampleSize,int speed,int stereo) {
  int newRate=speed;
  int newChannel=channels;
  int newBits=bits;

  if (stereo == true) {
    newChannel=ESD_STEREO;
  } else {
    newChannel=ESD_MONO;
  }

  if (sampleSize == 16) {
    newBits=ESD_BITS16;
  }
  if (sampleSize == 8) {
    newBits=ESD_BITS8;
  }

  if ( (newRate != rate) || 
       (newChannel != channels) ||
       (newBits != bits)) {
    bits=newBits;
    rate=newRate;
    channels=newChannel;

    return init(true);
  }

  return init(false);
}


int ESDDevice::init(int lForceSet) {

  if (isOpen() == false) {
    return false;
  } 
  if (lForceSet) {
    close();
    return open();
  }
  if (lAutoInit) {
    close();
    return open();
  } 
  return false;
}


int ESDDevice::getAutoInit() {
  return lAutoInit;
}

void ESDDevice::setAutoInit(int lAutoInit) {
  this->lAutoInit=lAutoInit;
  if (lAutoInit == true) {
    lneedInit=true;
  }
}


StatusInfo* ESDDevice::getStatusInfo() {
  return statusInfo;
}


void ESDDevice::writeIn(NodeDevice* source,DeviceConfig* config) {
  if (lEnabled == false) {
    return;
  }
  AudioStream* audioStream=config->getAudioStream();
  AudioBuffer* buffer=audioStream->getAudioBuffer();
  MemChunk* memChunk=buffer->getMemChunk();
  StatusInfo* statusInfoStream=audioStream->getStatusInfo();
  AudioInfo* audioInfoStream=audioStream->getAudioInfo();

  // We ignore statusChange messages, because otherwise
  // it is possible that we initialize /dev/dsp with "strange"
  // values. (Linux ignores this FreeBSD hangs)
  if (statusInfoStream->getChange()) {
    statusInfoStream->copyTo(statusInfo);
    return;
  }
  int newRate=audioInfoStream->getSpeed();
  int newChannel=audioInfoStream->getStereo()+1;
  int newBits=audioInfoStream->getSampleSize();

  // mapping from normal values to esd bitmask:

  if (newChannel == 2) {
    newChannel=ESD_STEREO;
  } else {
    newChannel=ESD_MONO;
  }

  if (newBits == 16) {
    newBits=ESD_BITS16;
  } else {
    if (newBits == 8) {
      newBits=ESD_BITS8;
    }
  }
  
  if ((newRate != rate) || 
      (newChannel != channels) ||
      (newBits != bits)
      || lneedInit) {
    bits=newBits;
    rate=newRate;
    channels=newChannel;
    init(false);
  }

  nativeMixerDevice->writeInLock();
  nativeMixerDevice->writeIn(source,config);
  nativeMixerDevice->writeInUnlock();


  lneedInit=false;
  if (isOpen()) {
    if (memChunk != NULL) {
      if (memChunk->getLen() > 0) {
	int didWrite=rawWrite(memChunk->getPtr(),memChunk->getLen());
	if (didWrite == -1) {
	  perror("********* write in ESDDevice");
	}
	if (didWrite != memChunk->getLen()){
	  cout << "didWrite != memChunk->getLen()"<<endl;
	  lneedInit=true;
	}
      } else {
	cout << "memChunk len is 0"<<endl;
      }
    } else {
      cout << "memChunk is NULL!"<<endl;
    }
  } else {
    getEventQueue()->sendEvent(_AUDIODEVICE_CANNOT_OPEN);
    NodeDevice::doSleep(500);
    lneedInit=true;
  }
}
  



int ESDDevice::rawWrite(char* buf,int len) {
  return ::write(sock,buf,len);
}






int ESDDevice::getEnabled() {
  return lEnabled;
}


void ESDDevice::setEnabled(int lEnable) {
  // block the device for a short time
  writeInLock();
  this->lEnabled=lEnable;
  if ((isOpen() == true) && (lEnabled==false)) {
    close();
  }
  writeInUnlock();
}


#endif
